/* LICENSE */

package net.smartlab.config;

import java.util.HashMap;
import java.util.Iterator;

import org.xml.sax.Attributes;

/**
 * This class provides a skeletal implementation of a configuration tree to
 * provide a set of common operations with a generic implementation.
 *
 * @author rlogiacco@smartlab.net
 */
abstract public class Configuration extends Node {

	/* # net.smartlab.config.ConfigurationException ce */
	private boolean resolve;

	/**
	 * Creates an instance of the class as an <code>Element</code> without a
	 * parent node. This constructor is providen for subclasses convenience.
	 */
	protected Configuration() {
		super(null, null, null);
	}

	/**
	 * Initializes this configuration with a name and a set of attributes.
	 *
	 * @param name the name of the configuration tree.
	 * @param attributes a set of attributes relative to the global
	 *            configuration tree.
	 */
	protected void init(String name, Attributes attributes) {
		this.name = name;
		if (attributes != null) {
			this.attributes = new HashMap(attributes.getLength());
			for (int i = 0; i < attributes.getLength(); i++) {
				this.attributes.put(attributes.getQName(i), attributes.getValue(i));
			}
		}
		this.parent = this;
	}

	/**
	 * @see net.smartlab.config.Node#resolve(java.lang.String, java.lang.String)
	 */
	protected Node resolve(String name, String id) throws ConfigurationException {
		Iterator children = this.children.iterator();
		while (children.hasNext()) {
			Element child = (Element)children.next();
			if (child instanceof Node && name.equals(child.name) && id.equals(child.getId())) {
				return (Node)child;
			}
		}
		throw new ConfigurationException("Cannot resolve element " + name + " with id " + id);
	}

	/**
	 * This method checks if the source from the configuration structure was
	 * modified since it was last read.
	 *
	 * @return <code>true</code> if the underlying representation of the
	 *         structure was modified since last read.
	 * @exception ConfigurationException if something wrong occurred while
	 *                checking the underlaying structure like the file is no
	 *                more readable or the stream no more available.
	 */
	public abstract boolean isChanged() throws ConfigurationException;

	/**
	 * Ensures the in memory representation of the configuration matches with
	 * the underlying representation.
	 *
	 * @exception ConfigurationException if the underlying representation of the
	 *                structure is no more accessible or the decoding process
	 *                fails due to an unappropriately initialized
	 *                <code>Cipher</code>.
	 */
	public abstract void update() throws ConfigurationException;

	/**
	 * Sets the resolution method to adopt while querying this configuration
	 * tree. When set to <code>true</code> all internal references are
	 * automatically resolved to the destination node if present in the
	 * configuration tree.
	 *
	 * @param resolve the resolution method to adopt.
	 */
	public void setResolve(boolean resolve) {
		this.resolve = resolve;
	}

	/**
	 * Returns the resolution method adopted by this configuration tree.
	 *
	 * @return <code>true</code> if internal references should be
	 *         automatically resolved.
	 */
	public boolean getResolve() {
		return resolve;
	}
}