/*
 * Backpack for Android - Skyscanner's Design System
 *
 * Copyright 2018 Skyscanner Ltd
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.skyscanner.backpack.compose.calendar.internal

import androidx.compose.foundation.layout.Arrangement
import androidx.compose.foundation.layout.Row
import androidx.compose.foundation.layout.padding
import androidx.compose.runtime.Composable
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.semantics.heading
import androidx.compose.ui.semantics.semantics
import net.skyscanner.backpack.calendar2.CalendarParams
import net.skyscanner.backpack.calendar2.data.CalendarCell
import net.skyscanner.backpack.compose.button.BpkButton
import net.skyscanner.backpack.compose.button.BpkButtonType
import net.skyscanner.backpack.compose.text.BpkText
import net.skyscanner.backpack.compose.theme.BpkTheme
import net.skyscanner.backpack.compose.tokens.BpkSpacing

@Composable
internal fun BpkCalendarHeaderCell(
    model: CalendarCell.Header,
    modifier: Modifier = Modifier,
    onSelectWholeMonth: (CalendarCell.Header) -> Unit,
) {
    Row(
        modifier = modifier.padding(horizontal = BpkSpacing.Base),
        verticalAlignment = Alignment.CenterVertically,
        horizontalArrangement = Arrangement.spacedBy(BpkSpacing.Base),
    ) {

        BpkText(
            text = model.title,
            maxLines = 1,
            color = BpkTheme.colors.textPrimary,
            style = BpkTheme.typography.heading4,
            modifier = Modifier
                .weight(1f)
                .semantics { heading() }
                .padding(vertical = BpkSpacing.Lg),
        )

        val monthSelectionMode = model.monthSelectionMode
        if (monthSelectionMode is CalendarParams.MonthSelectionMode.SelectWholeMonth) {
            BpkButton(
                text = monthSelectionMode.label,
                enabled = model.calendarSelectionMode !is CalendarParams.SelectionMode.Disabled,
                type = BpkButtonType.Link,
                onClick = { onSelectWholeMonth(model) },
            )
        }
    }
}
