/**
 * Backpack for Android - Skyscanner's Design System
 *
 * Copyright 2018-2021 Skyscanner Ltd
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package net.skyscanner.backpack.calendar.model

import android.content.Context
import android.content.res.Configuration
import androidx.annotation.ColorInt
import androidx.core.graphics.ColorUtils
import net.skyscanner.backpack.R
import net.skyscanner.backpack.calendar.model.CalendarCellStyle.TextStyle.Dark
import net.skyscanner.backpack.calendar.model.CalendarCellStyle.TextStyle.Light
import net.skyscanner.backpack.calendar.presenter.HighlightedDaysAdapter
import org.threeten.bp.LocalDate

data class CalendarColoring(
  val coloredBuckets: Set<ColoredBucket>,
)

data class ColoredBucket(
  val calendarCellStyle: CalendarCellStyle,
  val days: Set<LocalDate>,
)

sealed class CalendarCellStyle {

  /**
   * Text style of the calendar cell. Where [Light] and [Dark] refer to the background colour.
   * I.e.
   *  [Light] should be used to indicate the background is light and black text should be used.
   *  [Dark] should be used to indicate the background is dark and white text should be used.
   */
  enum class TextStyle {
    Light,
    Dark
  }

  /**
   * A positive cell style which is suitable to indicate for example
   * a date which has a comparatively low price among the dates in
   * the calendar.
   */
  object Positive : CalendarCellStyle() {
    override fun color(context: Context) =
      context.getColor(R.color.__calendarCellStylePositiveColor)
  }

  /**
   * A neutral cell style which is suitable to indicate for example
   * a date which has a comparatively average price among the dates in
   * the calendar.
   */
  object Neutral : CalendarCellStyle() {
    override fun color(context: Context) =
      context.getColor(R.color.__calendarCellStyleNeutralColor)
  }

  /**
   * A negative cell style which is suitable to indicate for example
   * a date which has a comparatively high price among the dates in
   * the calendar.
   */
  object Negative : CalendarCellStyle() {
    override fun color(context: Context) =
      context.getColor(R.color.__calendarCellStyleNegativeColor)

    override fun textStyle(context: Context): TextStyle {
      val nightModeFlags: Int = context.resources.configuration.uiMode and Configuration.UI_MODE_NIGHT_MASK

      return if (nightModeFlags == Configuration.UI_MODE_NIGHT_YES) {
        Light
      } else {
        Dark
      }
    }
  }

  /**
   * A cell style which is suitable to indicate a holiday.
   * Use this in conjunction with [HighlightedDaysAdapter] to
   * show a footer with the list of holidays for the month.
   *
   * @see HighlightedDaysAdapter
   */
  object Hightlight : CalendarCellStyle() {
    override fun color(context: Context) =
      context.getColor(R.color.__calendarHighlightedDayDot)
  }

  /**
   * A custom cell style.
   */
  data class Custom(
    @ColorInt private val color: Int,
    private val textStyle: TextStyle? = null,
  ) : CalendarCellStyle() {
    override fun color(context: Context) = color
    override fun textStyle(context: Context): TextStyle {
      return textStyle ?: super.textStyle(context)
    }
  }

  @ColorInt
  abstract fun color(context: Context): Int

  open fun textStyle(context: Context): TextStyle =
    if (ColorUtils.calculateLuminance(color(context)) < 0.5f)
      TextStyle.Dark
    else
      TextStyle.Light
}
