package net.sinyax.sofa.web;

import net.sinyax.sofa.ConflictingVersionException;
import net.sinyax.sofa.doc.Document;
import net.sinyax.sofa.doc.ImmutableDocumentImpl;
import net.sinyax.sofa.storage.DocumentNotFoundException;
import net.sinyax.sofa.storage.InMemoryStorage;
import net.sinyax.sofa.web.dto.BulkUpdate;
import net.sinyax.sofa.web.dto.RevDiffResponse;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.*;

import java.util.*;
import java.util.stream.Collectors;

import net.sinyax.sofa.Database;

@RestController
@RequestMapping("/{dbName:[a-z][a-zA-Z0-9_.-]*}")
public class DbController {
  Logger log = LoggerFactory.getLogger(DbController.class);
  private final Map<String, Database> openDbs = new HashMap<>();
  private Database getDb(String dbName) {
    var db = openDbs.get(dbName);
    if (db != null) return db;

    var newDb = new Database(new InMemoryStorage());
    openDbs.put(dbName, newDb);
    return newDb;
  }

  @GetMapping("")
  Map<String, Object>
  getDbInfo(@PathVariable String dbName) {
    return Map.of(
      "db_name", dbName,
      "purge_seq", "0",
      "update_seq", "0",
      "doc_del_count", 0,
      "doc_count", 0,
      "compact_running", false,
      "instance_start_time", "0"
    );
  }

  @PostMapping("_ensure_full_commit")
  @ResponseStatus(HttpStatus.CREATED)
  Map<String, Object>
  ensureFullCommit(@PathVariable String dbName) {
    var db = getDb(dbName);
    db.ensureFullCommit();
    return Map.of(
      "ok", true,
      "instance_start_time", "0"
    );
  }

  @GetMapping("_all_docs")
  List<Document> allDocs(
    @PathVariable String dbName
  ) {
    var db = getDb(dbName);
    return db.getAllDocuments().collect(Collectors.toList());
  }

  @PostMapping("_bulk_docs")
  @ResponseStatus(HttpStatus.CREATED)
  List<Map<String, Object>> bulkDocs(
    @PathVariable String dbName,
    @RequestBody BulkUpdate bulkUpdate
  ) {
    var db = getDb(dbName);
    var result = new ArrayList<Map<String, Object>>();
    boolean newEdits = bulkUpdate.getNewEdits();
    for (Document doc : bulkUpdate.getDocs()) {
      try {
        var saved = db.save(doc, newEdits);
        if (newEdits) {
          result.add(Map.of(
            "ok", true,
            "id", saved.getId(),
            "rev", saved.getRevision()
          ));
        }
      } catch (ConflictingVersionException e) {
        if (newEdits) {
          result.add(Map.of(
            "error", "conflict",
            "reason", e.getMessage()
          ));
        }
      } catch (RuntimeException e) {
        if (newEdits) {
          result.add(Map.of(
            "error", "unknown",
            "reason", e.getMessage()
          ));
        }
      }
    }
    return result;
  }

  @PostMapping("")
  Map<String, Object> createNewDocument(
    @PathVariable String dbName,
    @RequestBody Map<String, Object> newDoc
  ) {
    var db = getDb(dbName);
    var saved = db.save(new ImmutableDocumentImpl(
      (String)newDoc.get("_id"),
      (String)newDoc.get("_rev"),
      newDoc.entrySet()
        .stream()
        .filter(
          (Map.Entry<String, Object> e) -> !e.getKey().equals("_id") && !e.getKey().equals("_rev")
        )
        .collect(Collectors.toMap(
          (Map.Entry<String, Object> e) -> e.getKey(),
          (Map.Entry<String, Object> e) -> e.getValue()
        ))
    ));
    return Map.of(
      "ok", true,
      "id", saved.getId(),
      "rev", saved.getRevision()
    );
  }

  @PostMapping("/_revs_diff")
  Map<String, RevDiffResponse> revsDiff(
    @PathVariable String dbName,
    @RequestBody Map<String, List<String>> req
  ) {
    log.info("Revs DIFF: {} : {}", dbName, req);
    return req.entrySet().stream()
      .map(e -> {
          // TODO: actually check which revs are missing
          return new AbstractMap.SimpleEntry<>(e.getKey(), new RevDiffResponse(e.getValue(), List.of()));
        }
      )
      .collect(Collectors.toMap(Map.Entry::getKey, Map.Entry::getValue))
      ;
  }

  @GetMapping("/{docId:[^_].*}")
  Document getDocumentById(
    @PathVariable String dbName,
    @PathVariable String docId
  ) {
    log.info("DB GET: {} / {}", dbName, docId);
    var db = getDb(dbName);
    var doc = db.getDocument(docId);
    return doc;
  }

  @PutMapping("/{docId:[^_].*}")
  @ResponseStatus(HttpStatus.CREATED)
  Map<String, Object> putDocumentById(
    @PathVariable String dbName,
    @PathVariable String docId,
    @RequestParam(name="rev", required=false) String rev,
    @RequestParam(name="new_edits", defaultValue = "true", required=false) boolean newEdits,
    @RequestHeader(value = "if-match", required = false) String ifMatch,
    @RequestBody Document doc
  ) {
    var db = getDb(dbName);
    final String docRev;
    if (doc.getRevision() != null) {
      docRev = doc.getRevision();
    } else if (rev != null) {
      docRev = rev;
    } else {
      docRev = ifMatch;
    }
    var docToSave = new ImmutableDocumentImpl(docId, docRev, doc.isDeleted(), doc.copyBody(), null);
    var savedDoc = db.save(docToSave, newEdits);
    return Map.of(
      "id", savedDoc.getId(),
      "rev", savedDoc.getRevision(),
      "ok", true
    );
  }

  @GetMapping("/_local/{docId}")
  Document getLocalDocument(
    @PathVariable String dbName,
    @PathVariable String docId
  ) {
    var db = getDb(dbName);
    return db.getLocalDocument(docId);
  }

  @PutMapping("/_local/{docId}")
  @ResponseStatus(HttpStatus.CREATED)
  Map<String, Object> putLocalDocument(
    @PathVariable String dbName,
    @PathVariable String docId,
    @RequestParam(name="rev", required=false) String rev,
    @RequestHeader(value = "if-match", required = false) String ifMatch,
    @RequestBody Document doc
    ) {
    var db = getDb(dbName);
    final String docRev;
    if (doc.getRevision() != null) {
      docRev = doc.getRevision();
    } else if (rev != null) {
      docRev = rev;
    } else {
      docRev = ifMatch;
    }
    var docToSave = new ImmutableDocumentImpl(docId, docRev, doc.isDeleted(), doc.copyBody(), null);
    var savedDoc = db.saveLocal(docToSave);
    return Map.of(
      "id", savedDoc.getId(),
      "rev", savedDoc.getRevision(),
      "ok", true
    );
  }

  @ResponseStatus(HttpStatus.NOT_FOUND)
  public static class HttpDocumentNotFoundException extends DocumentNotFoundException {
    public HttpDocumentNotFoundException(String docId) {
      super(docId);
    }
  }
}
