package net.sinyax.sofa;

import net.sinyax.sofa.doc.Document;
import net.sinyax.sofa.doc.ImmutableDocumentImpl;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.*;
import java.util.concurrent.ExecutionException;

public class Replicator {
  private static final Logger LOGGER = LoggerFactory.getLogger(Replicator.class);
  private static final String VERSION = "1";
  private final ReplicatorDbAdapter source;
  private final ReplicatorDbAdapter target;
  private final int batchSize;
  private final String replicationDocId;
  private int replicatedDocumentCount;
  private String startSeq;
  private String syncSeq;
  private Optional<Document> replDocSource;
  private Optional<Document> replDocTarget;

  public Replicator(ReplicatorDbAdapter source, ReplicatorDbAdapter target) {
    this.source = source;
    this.target = target;

    this.startSeq = "0";
    this.syncSeq = "0";
    this.batchSize = 100;

    this.replicationDocId = "_local/sofa.replication_status:" + replicationId();
    this.replicatedDocumentCount = 0;
  }

  private String replicationId() {
    var sourceId = source.identifier();
    var targetId = target.identifier();
    var full = sourceId.replace(">", ">>") + "->" + targetId;
    MessageDigest md = null;
    try {
      md = MessageDigest.getInstance("SHA-1");
    } catch (NoSuchAlgorithmException e) {
      throw new RuntimeException(e);
    }
    var digest = md.digest(full.getBytes(StandardCharsets.UTF_8));
    return CommonEncoding.hexEncode(digest);
  }

  public void preflight() throws ExecutionException, InterruptedException {
    replDocSource = source.getLocalDocument(replicationDocId).get();
    replDocTarget = target.getLocalDocument(replicationDocId).get();
    LOGGER.debug("begin repl preflight from {} to {}", source.identifier(), target.identifier());

    if (replDocSource.isPresent() && replDocTarget.isPresent()) {
      var versionFromSource = replDocSource.map(d -> d.getString("sofa_replication_id_version")).orElse("-");
      var versionFromTarget = replDocTarget.map(d -> d.getString("sofa_replication_id_version")).orElse("-");
      var sourceLastSeqFromSource = replDocSource.map(d -> d.getString("source_last_seq")).orElse(null);
      var sourceLastSeqFromTarget = replDocTarget.map(d -> d.getString("source_last_seq")).orElse(null);
      if (sourceLastSeqFromSource != null && Objects.equals(sourceLastSeqFromSource, sourceLastSeqFromTarget)
        && VERSION.equals(versionFromSource) && VERSION.equals(versionFromTarget)) {
        startSeq = sourceLastSeqFromSource;
        LOGGER.debug("start seq for replication set to: {}", startSeq);
      }
    }

    syncSeq = startSeq;
  }

  public boolean step() throws ExecutionException, InterruptedException {
    var changes = source.getChanges(syncSeq, batchSize).get();
    var idsToReplicate = new HashMap<String, HashSet<String>>();
    for (var change : changes) {
      var revSet = idsToReplicate.get(change.id);
      var currentRevs = change.leaves;
      if (revSet == null) {
        revSet = new HashSet<>(currentRevs);
        idsToReplicate.put(change.id, revSet);
      } else {
        revSet.addAll(currentRevs);
      }
    }
    if (changes.isEmpty()) {
      return false;
    }
    var lastSeqInBatch = changes.get(changes.size() - 1).seq;

    HashMap<String, List<String>> revisionList = new HashMap<>(idsToReplicate.size());
    for (Map.Entry<String, HashSet<String>> entry : idsToReplicate.entrySet()) {
      revisionList.put(entry.getKey(), List.copyOf(entry.getValue()));
    }
    var diffIds = target.getRevisionDiff(revisionList).get();

    List<Document> toUpload = new ArrayList<>(idsToReplicate.size());
    for (var taskEntry : diffIds.entrySet()) {
      var missing = taskEntry.getValue().get("missing");
      if (missing == null || missing.isEmpty()) {
        continue;
      }
      var docId = taskEntry.getKey();
      source.getDocument(docId, true, missing).get()
        .stream()
        .filter(drr -> drr.doc != null)
        .map(drr -> drr.doc)
        .forEach(toUpload::add);
    }

    target.bulkSaveDocuments(toUpload, false).get();
    replicatedDocumentCount += toUpload.size();

    // if everything went smoothly until here, we can store the last seq
    // of the processed batch as starting point for the next batch
    syncSeq = lastSeqInBatch;

    return true;
  }

  public void storeCheckpoint() {
    if (syncSeq.equals(startSeq)) {
      // nothing was done, no need to update checkpoint
      return;
    }

    Document newReplDocSource = new ImmutableDocumentImpl(
      replicationDocId,
      replDocSource.map(Document::getRevision).orElse(null),
      Map.of(
        "sofa_replication_id_version", VERSION,
        "source_last_seq", syncSeq
      ));
    Document newReplDocTarget = new ImmutableDocumentImpl(
      replicationDocId,
      replDocTarget.map(Document::getRevision).orElse(null),
      newReplDocSource.copyBody());
    source.saveLocalDocument(newReplDocSource);
    target.saveLocalDocument(newReplDocTarget);

    // checkpoint can be saved between steps, so set start to current seq
    startSeq = syncSeq;
  }

  public int getReplicatedDocumentCount() {
    return replicatedDocumentCount;
  }
}
