package net.sinyax.sofa;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.core.type.TypeReference;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.ObjectMapper;
import net.sinyax.sofa.doc.ChangeRecord;
import net.sinyax.sofa.doc.Document;
import net.sinyax.sofa.doc.ImmutableDocumentImpl;
import net.sinyax.sofa.dto.*;
import net.sinyax.sofa.json.JacksonModule;
import okhttp3.*;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;

import java.io.IOException;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

public class RemoteHttpDatabase implements ReplicatorDbAdapter {

  private final String baseUrl;
  private final OkHttpClient http;
  private final ObjectMapper om;

  @Nullable
  private final String authentication;

  private static final MediaType APPLICATION_JSON = MediaType.parse("application/json");
  private final String userAgent;

  /**
   * Create a replication endpoint for a remote (HTTP or HTTPS) with a CouchDB-compatible REST interface
   * @param baseUrl URL of the DB info endpoint (https://docs.couchdb.org/en/stable/api/database/common.html#get--db)
   * @param authentication If not null, this String is passed literally as "Authorization" header for all requests
   * @param userAgent If not null, this String is set as User Agent for all HTTP requests
   */
  public RemoteHttpDatabase(String baseUrl, @Nullable String authentication, @Nullable String userAgent) {
    if (baseUrl.endsWith("/")) {
      this.baseUrl = baseUrl;
    } else {
      this.baseUrl = baseUrl + "/";
    }
    this.http = new OkHttpClient.Builder()
      .build();
    this.om = new ObjectMapper();
    this.om.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
    this.om.registerModule(new JacksonModule());
    this.authentication = authentication;
    this.userAgent = Objects.requireNonNullElse(userAgent, "sofa-replication");
  }

  public RemoteHttpDatabase(String baseUrl, @Nullable String authentication) {
    this(baseUrl, authentication, null);
  }

  private CompletableFuture<Response> asyncCall(Request request) {
    var cf = new CompletableFuture<Response>();
    http.newCall(request).enqueue(
      new Callback() {
        @Override
        public void onFailure(@NotNull Call call, @NotNull IOException e) {
          cf.completeExceptionally(e);
        }

        @Override
        public void onResponse(@NotNull Call call, @NotNull Response response) throws IOException {
          if (response.code() >= 200 && response.code() < 400) {
            cf.complete(response);
          } else {
            cf.completeExceptionally(new HttpResponseStatusException(response.code(), response.message()));
          }
        }
      }
    );
    return cf;
  }

  @Override
  public String identifier() {
    return baseUrl;
  }

  @Override
  public CompletableFuture<DatabaseInfo> info() {
    Request req = request("", null)
      .get()
      .build();
    return asyncCall(req).thenApply((res) -> {
      try {
        return om.readValue(res.body().byteStream(), DatabaseInfo.class);
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
    });
  }

  @Override
  public CompletableFuture<List<ChangeRecord>> getChanges(String since, int limit) {
    Request req = request("_changes", Map.of(
      "since", since,
      "style", "all_docs",
      "limit", Integer.toString(limit)
      ))
      .get()
      .build();
    return asyncCall(req).thenApply((res) -> {
      try {
        return om.readValue(res.body().byteStream(), ChangesResponse.class)
          .results.stream()
          .map(cr -> new ChangeRecord(cr.id, cr.seq, cr.changes.stream().map(chRev -> chRev.rev).collect(Collectors.toList()), cr.deleted))
          .collect(Collectors.toList());
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
    });
  }

  private Request.Builder request(String path, @Nullable Map<String,String> params) {
    var urlBuilder = HttpUrl.parse(baseUrl + path)
      .newBuilder();
    if (params != null) {
      for (var entry : params.entrySet()) {
        urlBuilder = urlBuilder.addQueryParameter(entry.getKey(), entry.getValue());
      }
    }
    var url = urlBuilder.build();
    var builder = new Request.Builder()
      .header("User-Agent", this.userAgent)
      .url(url);
    if (authentication != null) {
      builder = builder.addHeader("authorization", authentication);
    }
    return builder;
  }

  @Override
  public CompletableFuture<Document> getDocument(String docId, boolean revs) {
    var req = request(docId, Map.of("revs", Boolean.toString(revs)))
      .get().build();
    return asyncCall(req).thenApply((res) -> {
      try {
        return om.readValue(res.body().byteStream(), Document.class);
      } catch (IOException e) {
        throw new RuntimeException(e);
      }
    });
  }

  @Override
  public CompletableFuture<List<DocRevResponse>> getDocument(String docId, boolean revs, List<String> revisions) {
    try {
      Request req = request(docId, Map.of(
        "revs", Boolean.toString(revs),
        "open_revs", om.writeValueAsString(revisions)
        ))
        .addHeader("accept", "application/json")
        .get().build();
      return asyncCall(req).thenApply((res) -> {
        try {
          return om.readValue(res.body().byteStream(), new TypeReference<List<DocRevResponse>>() { });
        } catch (IOException e) {
          throw new RuntimeException(e);
        }
      });
    } catch (JsonProcessingException e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public CompletableFuture<List<DocSaveResponse>> bulkSaveDocuments(List<Document> docs, boolean newEdits) {
    try {
      BulkUpdate body = new BulkUpdate();
      body.setDocs(docs);
      body.setNewEdits(newEdits);
      Request req = request("_bulk_docs", Map.of(
      ))
        .post(RequestBody.create(om.writeValueAsBytes(body), APPLICATION_JSON))
        .build();
      return asyncCall(req).thenApply((res) -> {
        try {
          return om.readValue(res.body().byteStream(), new TypeReference<List<DocSaveResponse>>() { });
        } catch (IOException e) {
          throw new RuntimeException(e);
        }
      });
    } catch (JsonProcessingException e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public CompletableFuture<Optional<Document>> getLocalDocument(String docId) {
    assert docId.startsWith("_local/");

    var req = request(docId, Map.of()).get().build();
    var cf = new CompletableFuture<Optional<Document>>();
    http.newCall(req).enqueue(
      new Callback() {
        @Override
        public void onFailure(@NotNull Call call, @NotNull IOException e1) {
          cf.completeExceptionally(e1);
        }

        @Override
        public void onResponse(@NotNull Call call, @NotNull Response response) throws IOException {
          if (response.code() >= 200 && response.code() < 400) {
            Document doc = om.readValue(response.body().byteStream(), Document.class);
            cf.complete(Optional.of(doc));
          } else if (response.code() == 404) {
            cf.complete(Optional.empty());
          } else {
            cf.completeExceptionally(new HttpResponseStatusException(response.code(), response.message()));
          }
        }
      }
    );
    return cf;
  }

  @Override
  public CompletableFuture<Document> saveLocalDocument(Document doc) {
    var docId = doc.getId();
    assert docId.startsWith("_local/");

    var body = doc.copyBody();
    try {
      Request req = request(docId, Map.of())
        .put(RequestBody.create(om.writeValueAsBytes(body), APPLICATION_JSON))
        .build();
      return asyncCall(req).thenApply((res) -> {
        try {
          var response = om.readValue(res.body().byteStream(), DocSaveResponse.class);
          if (!response.ok) {
            throw new RuntimeException("doc saving failed: " + docId);
          }
          return new ImmutableDocumentImpl(response.id, response.rev, doc.isDeleted(), body, null);
        } catch (IOException e) {
          throw new RuntimeException(e);
        }
      });
    } catch (JsonProcessingException e) {
      throw new RuntimeException(e);
    }
  }

  @Override
  public CompletableFuture<Map<String, Map<String, List<String>>>> getRevisionDiff(Map<String, List<String>> revs) {
    try {
      Request req = request("_revs_diff", Map.of())
        .post(RequestBody.create(om.writeValueAsBytes(revs), APPLICATION_JSON))
        .build();
      return asyncCall(req).thenApply((res) -> {
        try {
          return om.readValue(res.body().byteStream(), new TypeReference<>() {});
        } catch (IOException e) {
          throw new RuntimeException(e);
        }
      });
    } catch (JsonProcessingException e) {
      throw new RuntimeException(e);
    }
  }
}
