package net.sinyax.sofa;

import net.sinyax.sofa.doc.Document;
import net.sinyax.sofa.doc.ChangeRecord;
import net.sinyax.sofa.dto.DatabaseInfo;
import net.sinyax.sofa.dto.DocRevResponse;
import net.sinyax.sofa.dto.DocSaveResponse;
import net.sinyax.sofa.storage.DocumentNotFoundException;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

public class LocalDatabase implements ReplicatorDbAdapter {
  private final Database db;

  public LocalDatabase(Database db) {
    this.db = db;
  }

  private static <T> CompletableFuture<T> wrap(T value) {
    CompletableFuture<T> cf = new CompletableFuture<>();
    cf.complete(value);
    return cf;
  }

  @Override
  public String identifier() {
    return db.getDatabaseIdentifier();
  }

  @Override
  public CompletableFuture<DatabaseInfo> info() {
    var dbInfo = new DatabaseInfo();
    return wrap(dbInfo);
  }

  @Override
  public CompletableFuture<List<ChangeRecord>> getChanges(String since, int limit) {
    return wrap(
      db.getChanges(since)
        .limit(limit)
        .collect(Collectors.toList()));
  }

  @Override
  public CompletableFuture<Document> getDocument(String docId, boolean revs) {
    if (revs) {
      throw new RuntimeException("not implemented: fetch revs=true");
    }
    return wrap(
     db.getDocument(docId)
    );
  }

  @Override
  public CompletableFuture<List<DocRevResponse>> getDocument(String docId, boolean revs, List<String> revisions) {
    var docs = db.getDocument(docId, revs, revisions);
    List<DocRevResponse> result = new ArrayList<>(docs.size());
    for (int i = 0; i < docs.size(); ++i) {
      var r = new DocRevResponse();
      var doc = docs.get(i);
      if (doc.isPresent()) {
        r.doc = doc.orElseThrow();
      } else {
        r.missingRevId = revisions.get(i);
      }
      result.add(r);
    }
    return wrap(result);
  }

  @Override
  public CompletableFuture<List<DocSaveResponse>> bulkSaveDocuments(List<Document> docs, boolean newEdits) {
    var response = new ArrayList<DocSaveResponse>(docs.size());
    for (var doc : docs) {
      var saved = db.save(doc, newEdits);
      if (newEdits) {
        response.add(new DocSaveResponse(saved.getId(), saved.getRevision(), true));
      }
    }
    return wrap(response);
  }

  @Override
  public CompletableFuture<Optional<Document>> getLocalDocument(String docId) {
    try {
      return wrap(Optional.of(db.getLocalDocument(docId)));
    } catch (DocumentNotFoundException e) {
      return wrap(Optional.empty());
    }
  }

  @Override
  public CompletableFuture<Document> saveLocalDocument(Document doc) {
    Document saved = db.saveLocal(doc);
    return wrap(saved);
  }

  @Override
  public CompletableFuture<Map<String, Map<String, List<String>>>> getRevisionDiff(Map<String, List<String>> revs) {
    return wrap(db.revsDiff(revs));
  }
}
