package net.sinodawn.data.mybatis;

import com.zaxxer.hikari.HikariDataSource;
import net.sinodawn.common.env.SinoPropertyResolver;
import net.sinodawn.common.exception.SinoGenericException;
import net.sinodawn.data.DbType;
import org.apache.ibatis.logging.stdout.StdOutImpl;
import org.apache.ibatis.session.Configuration;
import org.apache.ibatis.session.SqlSessionFactory;
import org.mybatis.spring.SqlSessionFactoryBean;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;

import java.util.Optional;

/**
 * MyBatisDataSource单例
 *
 * @author liuxl
 */
public enum Datasource {
    // MyBatis DataSource单例
    INSTANCE;

    private final HikariDataSource dataSource;
    private final SqlSessionFactory sqlSessionFactory;

    Datasource() {
        HikariDataSource var = new HikariDataSource();
        SinoPropertyResolver resolver = SinoPropertyResolver.getInstance();
        var.setJdbcUrl(
            Optional.ofNullable(resolver.getProperty("spring.datasource.url"))
                .orElseThrow(() -> new SinoGenericException("Property spring.datasource.url can not be null"))
        );
        var.setUsername(resolver.getProperty("spring.datasource.username"));
        var.setPassword(resolver.getProperty("spring.datasource.password"));
        dataSource = var;
        sqlSessionFactory = initSqlSessionFactory();
    }

    public HikariDataSource getDataSource() {
        return dataSource;
    }

    public SqlSessionFactory getSqlSessionFactory() {
        return sqlSessionFactory;
    }

    private SqlSessionFactory initSqlSessionFactory() {
        DbType.DB db = DbType.getDb();
        SqlSessionFactoryBean sqlSessionFactoryBean = new SqlSessionFactoryBean();
        sqlSessionFactoryBean.setDataSource(dataSource);
        Configuration configuration = new Configuration();
        configuration.setLogImpl(StdOutImpl.class);
        sqlSessionFactoryBean.setConfiguration(configuration);

        PathMatchingResourcePatternResolver resolver = new PathMatchingResourcePatternResolver();
        try {
            sqlSessionFactoryBean.setMapperLocations(
                resolver.getResources(db.getMapperLocation())
            );
            return sqlSessionFactoryBean.getObject();
        } catch (Exception e) {
            throw new SinoGenericException(e.getMessage());
        }
    }

}
