/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.module.sys.role.service.impl;

import net.sinodawn.framework.audit.aunnotation.Audit;
import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.data.page.Page;
import net.sinodawn.framework.i18n.I18nHelper;
import net.sinodawn.framework.mybatis.mapper.MapperParameter;
import net.sinodawn.framework.mybatis.page.PageRowBounds;
import net.sinodawn.framework.restful.data.RestJsonWrapperBean;
import net.sinodawn.framework.restful.data.RestValidationResultBean;
import net.sinodawn.framework.support.table.bean.TableHeadBean;
import net.sinodawn.module.sys.role.bean.CoreRoleBean;
import net.sinodawn.module.sys.role.bean.CoreRolePermissionBean;
import net.sinodawn.module.sys.role.bean.CoreRoleUserBean;
import net.sinodawn.module.sys.role.dao.CoreRoleDao;
import net.sinodawn.module.sys.role.service.CoreRolePermissionService;
import net.sinodawn.module.sys.role.service.CoreRoleService;
import net.sinodawn.module.sys.role.service.CoreRoleUserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Repository
public class CoreRoleServiceImpl implements CoreRoleService {
   @Autowired
   private CoreRoleDao roleDao;
   @Autowired
   private CoreRoleUserService roleUserService;
   @Autowired
   private CoreRolePermissionService permissionService;

   public CoreRoleDao getDao() {
      return this.roleDao;
   }

   @Transactional
   public Long insert(RestJsonWrapperBean wrapper) {
      CoreRoleBean role = (CoreRoleBean)wrapper.parseUnique(CoreRoleBean.class);
      role.setId(ApplicationContextHelper.getNextIdentity());
      this.getDao().insert(role);
      return role.getId();
   }

   public Page<CoreRoleUserBean> selectRoleUserPaginationByRole(Long roleId, RestJsonWrapperBean wrapper) {
      return this.roleUserService.selectPaginationByRole(roleId, wrapper);
   }

   @Transactional
   public void insertRoleUser(Long id, RestJsonWrapperBean wrapper) {
      List<CoreRoleUserBean> roleUserList = wrapper.parse(CoreRoleUserBean.class);
      roleUserList.forEach((u) -> {
         this.roleUserService.insertByUserId(u.getUserId(), Arrays.asList(id));
      });
   }

   @Transactional
   @Audit("AUDIT.DELETE")
   public void deleteRoleUser(RestJsonWrapperBean wrapper) {
      List<Long> roleUserIdList = wrapper.parseId(Long.class);
      List<CoreRoleUserBean> roleUserList = this.roleUserService.selectListByIds(roleUserIdList);
      Map<String, List<Long>> mapList = (Map)roleUserList.stream().collect(Collectors.groupingBy(CoreRoleUserBean::getUserId, Collectors.mapping(CoreRoleUserBean::getRoleId, Collectors.toList())));
      mapList.forEach((k, v) -> {
         this.roleUserService.deleteByUserId(k, v);
      });
   }

   @Transactional
   public void insertRolePermission(Long id, String permissionTypeId, RestJsonWrapperBean wrapper) {
      List<CoreRolePermissionBean> permissionList = wrapper.parse(CoreRolePermissionBean.class);
      this.permissionService.insert(id, permissionTypeId, (List)permissionList.stream().map(CoreRolePermissionBean::getTargetId).collect(Collectors.toList()));
   }

   @Transactional
   public void deleteRolePermission(Long id, String permissionTypeId, RestJsonWrapperBean wrapper) {
      List<CoreRolePermissionBean> permissionList = wrapper.parse(CoreRolePermissionBean.class);
      this.permissionService.delete(id, permissionTypeId, (List)permissionList.stream().map(CoreRolePermissionBean::getTargetId).collect(Collectors.toList()));
   }

   @Transactional
   public void delete(RestJsonWrapperBean wrapper) {
      List<CoreRoleBean> roleList = wrapper.parse(CoreRoleBean.class);
      if (!roleList.isEmpty()) {
         roleList.forEach((r) -> {
            this.roleUserService.deleteByRoleId(r.getId());
            this.permissionService.deleteByRoleId(r.getId());
         });
         CoreRoleService.super.delete(wrapper);
      }

   }

   public void doSuspendAction(RestJsonWrapperBean wrapper, boolean suspend) {
      List<CoreRoleBean> roleList = wrapper.parse(CoreRoleBean.class);
      if (!roleList.isEmpty()) {
//         CoreRoleService.super.doSuspendAction(wrapper, suspend);
      }

   }

   public Page<CoreRoleBean> selectChoosablePagination(RestJsonWrapperBean wrapper) {
      MapperParameter parameter = wrapper.extractMapFilter();
      parameter.setSuspendedFlag("0");
      parameter.setOrgAuthority();
      PageRowBounds rowBounds = wrapper.extractPageRowBounds();
      return this.selectPagination(parameter, rowBounds);
   }

   public boolean isPermissionTypeAdminRole(Long roleId, String permissionTypeId) {
      List<CoreRolePermissionBean> permissionList = this.permissionService.selectListByRole(roleId);
      return permissionList.stream().anyMatch((p) -> {
         return permissionTypeId.equals(p.getPermissionTypeId()) && "0".equals(p.getTargetId());
      });
   }

   public List<TableHeadBean> selectPermissionMatrixHeadList() {
      List<CoreRoleBean> roleList = this.getDao().selectAll();
      return (List)roleList.stream().map((e) -> {
         TableHeadBean head = new TableHeadBean();
         head.setTitle(e.getRoleName());
         head.setField(e.getId().toString());
         return head;
      }).collect(Collectors.toList());
   }

   public RestValidationResultBean deleteValidate(RestJsonWrapperBean wrapper) {
      List<CoreRoleBean> roleList = wrapper.parse(CoreRoleBean.class);
      List<Long> roleIdList = (List)roleList.stream().map(CoreRoleBean::getId).collect(Collectors.toList());
      List<CoreRoleUserBean> roleUserList = this.roleUserService.getDao().selectListByOneColumnValues(roleIdList, "ROLEID");
      List<CoreRolePermissionBean> rolePermissionList = this.permissionService.getDao().selectListByOneColumnValues(roleIdList, "ROLEID");
      boolean validateFlag = true;
      StringBuilder messageBuilder = new StringBuilder();
      Iterator var8 = roleList.iterator();

      while(true) {
         CoreRoleBean role;
         List subRoleUserList;
         List subRolePermissionList;
         do {
            if (!var8.hasNext()) {
               return new RestValidationResultBean(validateFlag, messageBuilder.toString());
            }

            role = (CoreRoleBean)var8.next();
            CoreRoleBean finalRole = role;
            subRoleUserList = (List)roleUserList.stream().filter((f) -> {
               return f.getRoleId().equals(finalRole.getId());
            }).collect(Collectors.toList());
            CoreRoleBean finalRole1 = role;
            subRolePermissionList = (List)rolePermissionList.stream().filter((f) -> {
               return f.getRoleId().equals(finalRole1.getId());
            }).collect(Collectors.toList());
         } while(subRoleUserList.isEmpty() && subRolePermissionList.isEmpty());

         validateFlag = false;
         messageBuilder.append(role.getRoleName() + ":");
         if (!subRoleUserList.isEmpty() && subRolePermissionList.isEmpty()) {
            messageBuilder.append(I18nHelper.getMessage("MODULE.SYS.ROLES.TIP.EXISTS_ROLE_USERS"));
         } else if (subRoleUserList.isEmpty() && !subRolePermissionList.isEmpty()) {
            messageBuilder.append(I18nHelper.getMessage("MODULE.SYS.ROLES.TIP.EXISTS_ROLE_PERMISSIONS"));
         } else {
            messageBuilder.append(I18nHelper.getMessage("MODULE.SYS.ROLES.TIP.EXISTS_ROLE_USERS_AND_PERMISSIONS"));
         }

         messageBuilder.append("<br />");
      }
   }
}
