/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.module.sys.password.service.impl;

import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.context.LocalContextHelper;
import net.sinodawn.framework.database.sql.Order;
import net.sinodawn.framework.exception.PasswordException;
import net.sinodawn.framework.i18n.I18nHelper;
import net.sinodawn.framework.restful.data.RestJsonWrapperBean;
import net.sinodawn.framework.utils.EncryptUtils;
import net.sinodawn.framework.utils.NumberUtils;
import net.sinodawn.framework.utils.StringUtils;
import net.sinodawn.module.mdm.user.bean.CoreUserBean;
import net.sinodawn.module.sys.password.bean.CorePasswordHisBean;
import net.sinodawn.module.sys.password.bean.CorePasswordPolicyBean;
import net.sinodawn.module.sys.password.dao.CorePasswordPolicyDao;
import net.sinodawn.module.sys.password.service.CorePasswordHisService;
import net.sinodawn.module.sys.password.service.CorePasswordPolicyService;
import net.sinodawn.module.sys.role.bean.CoreRoleBean;
import net.sinodawn.module.sys.role.bean.CoreRoleUserBean;
import net.sinodawn.module.sys.role.service.CoreRolePermissionService;
import net.sinodawn.module.sys.role.service.CoreRoleService;
import net.sinodawn.module.sys.role.service.CoreRoleUserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.time.Duration;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

@Repository
public class CorePasswordPolicyServiceImpl implements CorePasswordPolicyService {
   @Autowired
   private CorePasswordPolicyDao policyDao;
   @Autowired
   @Lazy
   private CoreRoleUserService roleUserService;
   @Autowired
   @Lazy
   private CoreRolePermissionService rolePermissionService;
   @Autowired
   @Lazy
   private CoreRoleService roleService;
   @Autowired
   private CorePasswordHisService hisService;

   public CorePasswordPolicyDao getDao() {
      return this.policyDao;
   }

   @Transactional
   public Long insert(RestJsonWrapperBean jsonWrapper) {
      CorePasswordPolicyBean policy = (CorePasswordPolicyBean)jsonWrapper.parseUnique(CorePasswordPolicyBean.class);
      policy.setId(ApplicationContextHelper.getNextIdentity());
      policy.setType("ROLE");
      this.getDao().insert(policy);
      return policy.getId();
   }

   public List<CorePasswordPolicyBean> selectEffectedList(String userId) {
      List<CorePasswordPolicyBean> effectedList = new ArrayList();
      List<CoreRoleUserBean> roleUserList = this.roleUserService.selectByUserId(userId);
      if (!roleUserList.isEmpty()) {
         List<CoreRoleBean> roleList = (List)roleUserList.stream().map((u) -> {
            return (CoreRoleBean)this.roleService.selectById(u.getRoleId());
         }).collect(Collectors.toList());
         List<Long> idList = (List)roleList.stream().filter((r) -> {
            return "0".equals(r.getLastSuspendedFlag()) && r.getPasswordPolicyId() != null;
         }).map((r) -> {
            return r.getPasswordPolicyId();
         }).collect(Collectors.toList());
         if (!idList.isEmpty()) {
            idList.forEach((i) -> {
               CorePasswordPolicyBean policy = (CorePasswordPolicyBean)this.getDao().selectByIdIfPresent(i);
               if (policy != null) {
                  effectedList.add(policy);
               }

            });
         }
      }

      if (effectedList.isEmpty()) {
         effectedList.add(this.getDao().selectById(1L));
      }

      return effectedList;
   }

   public void checkPassword(String userId, String rawPassword) {
      List<CorePasswordPolicyBean> policyList = this.selectEffectedList(userId);
      if (!policyList.isEmpty()) {
         CorePasswordHisBean hisFilter = new CorePasswordHisBean();
         hisFilter.setUserId(userId);
         List<CorePasswordHisBean> hisList = this.hisService.selectList(hisFilter, new Order[]{Order.desc("ID")});
         LocalDateTime now = LocalDateTime.now();
         Iterator var7 = policyList.iterator();

         CorePasswordPolicyBean policy;
         int count;
         do {
            do {
               do {
                  if (!var7.hasNext()) {
                     return;
                  }

                  policy = (CorePasswordPolicyBean)var7.next();
                  StringBuilder sb = new StringBuilder();
                  if (policy.getMinNumber() != null && !rawPassword.matches(".*(\\d{1}\\D*){" + policy.getMinNumber() + ",}.*")) {
                     sb.append(I18nHelper.getMessage("SINO.USER.PASSWORD.MIN_NUMBER_REQUIRED")).append(":").append(policy.getMinNumber());
                  }

                  if (policy.getMinLowercase() != null && !rawPassword.matches(".*([a-z]{1}[^a-z]*){" + policy.getMinLowercase() + ",}.*")) {
                     if (sb.length() > 0) {
                        sb.append(",");
                     }

                     sb.append(I18nHelper.getMessage("SINO.USER.PASSWORD.MIN_LOWERCASE_REQUIRED")).append(":").append(policy.getMinLowercase());
                  }

                  if (policy.getMinUppercase() != null && !rawPassword.matches(".*([A-Z]{1}[^A-Z]*){" + policy.getMinUppercase() + ",}.*")) {
                     if (sb.length() > 0) {
                        sb.append(",");
                     }

                     sb.append(I18nHelper.getMessage("SINO.USER.PASSWORD.MIN_UPPERCASE_REQUIRED")).append(":").append(policy.getMinUppercase());
                  }

                  if (policy.getMinSpecialCharacters() != null && !rawPassword.matches(".*([^A-Za-z0-9]{1}[A-Za-z0-9]*){" + policy.getMinSpecialCharacters() + ",}.*")) {
                     if (sb.length() > 0) {
                        sb.append(",");
                     }

                     sb.append(I18nHelper.getMessage("SINO.USER.PASSWORD.MIN_SPECIALCHARACTERS_REQUIRED")).append(":").append(policy.getMinSpecialCharacters());
                  }

                  if (policy.getMinLength() != null && (long)rawPassword.length() < policy.getMinLength()) {
                     if (sb.length() > 0) {
                        sb.append(",");
                     }

                     sb.append(I18nHelper.getMessage("SINO.USER.PASSWORD.MIN_LENGTH")).append(":").append(policy.getMinLength());
                  }

                  if (policy.getMaxLength() != null && (long)rawPassword.length() > policy.getMaxLength()) {
                     if (sb.length() > 0) {
                        sb.append(",");
                     }

                     sb.append(I18nHelper.getMessage("SINO.USER.PASSWORD.MAX_LENGTH")).append(":").append(policy.getMaxLength());
                  }

                  if (policy.getComposition() != null && policy.getComposition() > 0L) {
                     count = 0;
                     if (StringUtils.containsNumber(rawPassword)) {
                        ++count;
                     }

                     if (StringUtils.containsLowercaseLetter(rawPassword)) {
                        ++count;
                     }

                     if (StringUtils.containsCapitalLetter(rawPassword)) {
                        ++count;
                     }

                     if (StringUtils.containsSpecialCharacter(rawPassword)) {
                        ++count;
                     }

                     if (policy.getComposition() > (long)count) {
                        if (sb.length() > 0) {
                           sb.append(",");
                        }

                        sb.append(I18nHelper.getMessage("SINO.USER.PASSWORD.COMPOSITION")).append(":").append(policy.getComposition());
                     }
                  }

                  if (sb.length() > 0) {
                     throw new PasswordException(sb.toString());
                  }
               } while(hisList.isEmpty());

               if (policy.getReuseInterval() != null) {
                  CorePasswordHisBean passwordHis = (CorePasswordHisBean)hisList.stream().filter((h) -> {
                     return h.getPassword().equals(EncryptUtils.MD5Encrypt(rawPassword));
                  }).findFirst().orElse(null);
                  if (passwordHis != null && passwordHis.getCreatedTime().plusDays(policy.getReuseInterval()).isAfter(now)) {
                     throw new PasswordException(I18nHelper.getMessage("SINO.USER.PASSWORD.REUSE_PASSWORD_INTERVAL") + ":" + policy.getReuseInterval());
                  }
               }
            } while(policy.getHistoryLength() == null);

            count = 0;

            for(Iterator var11 = hisList.iterator(); var11.hasNext(); ++count) {
               CorePasswordHisBean passwordHis = (CorePasswordHisBean)var11.next();
               if (passwordHis.getPassword().equals(EncryptUtils.MD5Encrypt(rawPassword))) {
                  break;
               }
            }
         } while((long)count >= policy.getHistoryLength());

         throw new PasswordException(I18nHelper.getMessage("SINO.USER.PASSWORD.REUSE_PASSWORD_HISTORY_LENGTH") + ":" + policy.getHistoryLength());
      }
   }

   public CorePasswordPolicyBean selectMatchingPasswordPolicy() {
      String userId = LocalContextHelper.getLoginUserId();
      return this.selectMatchingPasswordPolicy(userId);
   }

   public CorePasswordPolicyBean selectMatchingPasswordPolicy(String userId) {
      List<CorePasswordPolicyBean> passwordPolicyList = this.selectEffectedList(userId);
      if (passwordPolicyList.isEmpty()) {
         return null;
      } else {
         CorePasswordPolicyBean matchingPasswordPolicy = new CorePasswordPolicyBean();
         int i = 0;

         for(int j = passwordPolicyList.size(); i < j; ++i) {
            CorePasswordPolicyBean passwordPolicy = (CorePasswordPolicyBean)passwordPolicyList.get(i);
            matchingPasswordPolicy.setMinNumber((Long)NumberUtils.max(matchingPasswordPolicy.getMinNumber(), passwordPolicy.getMinNumber()));
            matchingPasswordPolicy.setMinLowercase((Long)NumberUtils.max(matchingPasswordPolicy.getMinLowercase(), passwordPolicy.getMinLowercase()));
            matchingPasswordPolicy.setMinUppercase((Long)NumberUtils.max(matchingPasswordPolicy.getMinUppercase(), passwordPolicy.getMinUppercase()));
            matchingPasswordPolicy.setMinSpecialCharacters((Long)NumberUtils.max(matchingPasswordPolicy.getMinSpecialCharacters(), passwordPolicy.getMinSpecialCharacters()));
            matchingPasswordPolicy.setMinLength((Long)NumberUtils.max(matchingPasswordPolicy.getMinLength(), passwordPolicy.getMinLength()));
            matchingPasswordPolicy.setMaxLength((Long)NumberUtils.min(matchingPasswordPolicy.getMaxLength(), passwordPolicy.getMaxLength()));
            matchingPasswordPolicy.setComposition((Long)NumberUtils.max(matchingPasswordPolicy.getComposition(), passwordPolicy.getComposition()));
            matchingPasswordPolicy.setMaxFailedAttempts((Long)NumberUtils.min(matchingPasswordPolicy.getMaxFailedAttempts(), passwordPolicy.getMaxFailedAttempts()));
            matchingPasswordPolicy.setExpiryInterval((Long)NumberUtils.min(matchingPasswordPolicy.getExpiryInterval(), passwordPolicy.getExpiryInterval()));
            matchingPasswordPolicy.setExpiryWarningInterval((Long)NumberUtils.max(matchingPasswordPolicy.getExpiryWarningInterval(), passwordPolicy.getExpiryWarningInterval()));
            matchingPasswordPolicy.setExpiredMaxUse((Long)NumberUtils.min(matchingPasswordPolicy.getExpiredMaxUse(), passwordPolicy.getExpiredMaxUse()));
            matchingPasswordPolicy.setReuseInterval((Long)NumberUtils.min(matchingPasswordPolicy.getReuseInterval(), passwordPolicy.getReuseInterval()));
            matchingPasswordPolicy.setHistoryLength((Long)NumberUtils.max(matchingPasswordPolicy.getHistoryLength(), passwordPolicy.getHistoryLength()));
         }

         if (matchingPasswordPolicy.getMinNumber() != null && matchingPasswordPolicy.getMinNumber() == 0L) {
            matchingPasswordPolicy.setMinNumber((Long)null);
         }

         if (matchingPasswordPolicy.getMinLowercase() != null && matchingPasswordPolicy.getMinLowercase() == 0L) {
            matchingPasswordPolicy.setMinLowercase((Long)null);
         }

         if (matchingPasswordPolicy.getMinUppercase() != null && matchingPasswordPolicy.getMinUppercase() == 0L) {
            matchingPasswordPolicy.setMinUppercase((Long)null);
         }

         if (matchingPasswordPolicy.getMinSpecialCharacters() != null && matchingPasswordPolicy.getMinSpecialCharacters() == 0L) {
            matchingPasswordPolicy.setMinSpecialCharacters((Long)null);
         }

         if (matchingPasswordPolicy.getMinLength() != null && matchingPasswordPolicy.getMinLength() == 0L) {
            matchingPasswordPolicy.setMinLength((Long)null);
         }

         if (matchingPasswordPolicy.getMaxLength() != null && matchingPasswordPolicy.getMaxLength() == 0L) {
            matchingPasswordPolicy.setMaxLength((Long)null);
         }

         if (matchingPasswordPolicy.getComposition() != null && matchingPasswordPolicy.getComposition() == 0L) {
            matchingPasswordPolicy.setComposition((Long)null);
         }

         if (matchingPasswordPolicy.getMaxFailedAttempts() != null && matchingPasswordPolicy.getMaxFailedAttempts() == 0L) {
            matchingPasswordPolicy.setMaxFailedAttempts((Long)null);
         }

         if (matchingPasswordPolicy.getExpiryInterval() != null && matchingPasswordPolicy.getExpiryInterval() == 0L) {
            matchingPasswordPolicy.setExpiryInterval((Long)null);
         }

         if (matchingPasswordPolicy.getExpiryWarningInterval() != null && matchingPasswordPolicy.getExpiryWarningInterval() == 0L) {
            matchingPasswordPolicy.setExpiryWarningInterval((Long)null);
         }

         if (matchingPasswordPolicy.getExpiredMaxUse() != null && matchingPasswordPolicy.getExpiredMaxUse() == 0L) {
            matchingPasswordPolicy.setExpiredMaxUse((Long)null);
         }

         if (matchingPasswordPolicy.getReuseInterval() != null && matchingPasswordPolicy.getReuseInterval() == 0L) {
            matchingPasswordPolicy.setReuseInterval((Long)null);
         }

         if (matchingPasswordPolicy.getHistoryLength() != null && matchingPasswordPolicy.getHistoryLength() == 0L) {
            matchingPasswordPolicy.setHistoryLength((Long)null);
         }

         return matchingPasswordPolicy;
      }
   }

   public Long selectPasswordExpireRemainingDays() {
      CoreUserBean loginUser = LocalContextHelper.getLoginUser();
      List<CorePasswordPolicyBean> passwordPolicyList = this.selectEffectedList(loginUser.getId());
      return this.selectPasswordExpireRemainingDays(passwordPolicyList);
   }

   public Long selectPasswordExpireRemainingDays(List<CorePasswordPolicyBean> passwordPolicyList) {
      CoreUserBean loginUser = LocalContextHelper.getLoginUser();
      List<CorePasswordPolicyBean> expiryWarningIntervalPasswordPolicyList = (List)passwordPolicyList.stream().filter((p) -> {
         return p.getExpiryInterval() != null && p.getExpiryInterval() > 0L && p.getExpiryWarningInterval() != null && p.getExpiryWarningInterval() > 0L;
      }).collect(Collectors.toList());
      if (!expiryWarningIntervalPasswordPolicyList.isEmpty()) {
         LocalDateTime now = LocalDateTime.now();
         List<Long> passwordExpireRemainingDaysList = new ArrayList();
         Iterator var6 = expiryWarningIntervalPasswordPolicyList.iterator();

         while(var6.hasNext()) {
            CorePasswordPolicyBean passwordPolicy = (CorePasswordPolicyBean)var6.next();
            if (loginUser.getPasswordUpdatedTime().plusDays(passwordPolicy.getExpiryInterval() - passwordPolicy.getExpiryWarningInterval()).isBefore(now)) {
               long passwordExpireRemainingDays = Duration.between(now, loginUser.getPasswordUpdatedTime().plusDays(passwordPolicy.getExpiryInterval())).toDays();
               passwordExpireRemainingDaysList.add(passwordExpireRemainingDays);
            }
         }

         if (passwordExpireRemainingDaysList.isEmpty()) {
            return null;
         } else {
            return passwordExpireRemainingDaysList.stream().mapToLong((e) -> {
               return e;
            }).min().getAsLong();
         }
      } else {
         return null;
      }
   }

   public Long selectPasswordExpiredMaxUse(List<CorePasswordPolicyBean> passwordPolicyList) {
      List<CorePasswordPolicyBean> expiredMaxUsePasswordPolicyList = (List)passwordPolicyList.stream().filter((p) -> {
         return p.getExpiredMaxUse() != null && p.getExpiredMaxUse() > 0L;
      }).collect(Collectors.toList());
      return !expiredMaxUsePasswordPolicyList.isEmpty() ? expiredMaxUsePasswordPolicyList.stream().mapToLong((e) -> {
         return e.getExpiredMaxUse();
      }).min().getAsLong() : null;
   }
}
