/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.module.sys.metadata.service.impl;

import net.sinodawn.framework.exception.UnsupportedException;
import net.sinodawn.framework.support.PersistableHelper;
import net.sinodawn.framework.support.PersistableMetadataHelper;
import net.sinodawn.framework.support.domain.Persistable;
import net.sinodawn.framework.utils.CollectionUtils;
import net.sinodawn.framework.utils.ConvertUtils;
import net.sinodawn.framework.utils.StringUtils;
import net.sinodawn.module.sys.metadata.bean.CoreValidatorBean;
import net.sinodawn.module.sys.metadata.dao.CoreValidatorDao;
import net.sinodawn.module.sys.metadata.service.CoreValidatorService;
import net.sinodawn.module.sys.metadata.support.ValidatorType;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

@Repository

public class CoreValidatorServiceImpl implements CoreValidatorService {
   @Autowired
   private CoreValidatorDao validatorDao;

//   @SinoCacheable({"core-validator"})
   public List<CoreValidatorBean> selectEnabledValidatorByTable(String table) {
      return this.validatorDao.selectEnabledValidatorByTable(table);
   }

   public <T extends Persistable<?>> List<String> validate(T item) {
      if (item == null) {
         return CollectionUtils.emptyList();
      } else {
         String table = PersistableMetadataHelper.getTableName(item.getClass());
         List<CoreValidatorBean> validatorList = this.selectEnabledValidatorByTable(table);
         return validatorList.isEmpty() ? CollectionUtils.emptyList() : (List)validatorList.stream().map((v) -> {
            return this.validate(v, item);
         }).filter((s) -> {
            return !StringUtils.isEmpty(s);
         }).collect(Collectors.toList());
      }
   }

   private <T extends Persistable<?>> String validate(CoreValidatorBean validator, T item) {
      if (validator.getTypeName().equalsIgnoreCase(ValidatorType.STRING.name())) {
         String value = (String)ConvertUtils.convert(PersistableHelper.getPropertyValue(item, validator.getColumnName()), String.class);
         if (value == null && (!Integer.valueOf(1).equals(validator.getNullable()) || validator.getMinSize() > 0)) {
            return validator.getMessage();
         }

         if (value != null && (value.length() < validator.getMinSize() || value.length() > validator.getMaxSize())) {
            return validator.getMessage();
         }
      } else if (validator.getTypeName().equalsIgnoreCase(ValidatorType.NUMBER.name())) {
         Double value = (Double)ConvertUtils.convert(PersistableHelper.getPropertyValue(item, validator.getColumnName()), Double.class);
         if (value == null && !Integer.valueOf(1).equals(validator.getNullable())) {
            return validator.getMessage();
         }

         if (value != null && (value < (double)validator.getMin() || value > (double)validator.getMax())) {
            return validator.getMessage();
         }
      } else {
         if (!validator.getTypeName().equalsIgnoreCase(ValidatorType.DATE.name())) {
            throw new UnsupportedException("Type name [" + validator.getTypeName() + "] is not supported!");
         }

         LocalDateTime value = (LocalDateTime)ConvertUtils.convert(PersistableHelper.getPropertyValue(item, validator.getColumnName()), LocalDateTime.class);
         if (value == null && !"1".equals(validator.getNullable())) {
            return validator.getMessage();
         }

         if (value != null) {
            String future = validator.getFuture();
            if (!StringUtils.isEmpty(future)) {
               if ("now".equals(future) && value.isAfter(LocalDateTime.now())) {
                  return validator.getMessage();
               }

               if (!"now".equals(future)) {
                  LocalDateTime futureValue = (LocalDateTime)ConvertUtils.convert(PersistableHelper.getPropertyValue(item, future), LocalDateTime.class);
                  if (futureValue != null && value.isAfter(futureValue)) {
                     return validator.getMessage();
                  }
               }
            }

            String past = validator.getPast();
            if (!StringUtils.isEmpty(past)) {
               if ("now".equals(past) && value.isBefore(LocalDateTime.now())) {
                  return validator.getMessage();
               }

               if (!"now".equals(past)) {
                  LocalDateTime pastValue = (LocalDateTime)ConvertUtils.convert(PersistableHelper.getPropertyValue(item, past), LocalDateTime.class);
                  if (pastValue != null && value.isBefore(pastValue)) {
                     return validator.getMessage();
                  }
               }
            }
         }
      }

      return null;
   }
}
