/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.module.sys.log.service.impl;

import com.blueconic.browscap.Capabilities;
import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.context.LocalContextHelper;
import net.sinodawn.framework.data.page.Page;
import net.sinodawn.framework.database.sql.Order;
import net.sinodawn.framework.mybatis.mapper.MatchPattern;
import net.sinodawn.framework.mybatis.mapper.SearchFilter;
import net.sinodawn.framework.restful.data.RestJsonWrapperBean;
import net.sinodawn.framework.security.AccountCategory;
import net.sinodawn.framework.security.authentication.AuthenticationHelper;
import net.sinodawn.framework.utils.BeanUtils;
import net.sinodawn.framework.utils.ServletUtils;
import net.sinodawn.framework.utils.StringUtils;
import net.sinodawn.module.sys.log.bean.CoreLoginLogBean;
import net.sinodawn.module.sys.log.dao.CoreLoginLogDao;
import net.sinodawn.module.sys.log.service.CoreLoginLogService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.lang.Nullable;
import org.springframework.stereotype.Service;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDateTime;
import java.util.List;

@Service
public class CoreLoginLogServiceImpl implements CoreLoginLogService {
   @Autowired
   private CoreLoginLogDao loginLogDao;

   public CoreLoginLogDao getDao() {
      return this.loginLogDao;
   }

   public void insertLoginSuccessLog(String token, String userId, String orgId) {
      CoreLoginLogBean loginLog = new CoreLoginLogBean();
      loginLog.setId(ApplicationContextHelper.getNextIdentity());
      loginLog.setToken(token);
      loginLog.setUserId(userId);
      loginLog.setStatus("1");
      loginLog.setOrgId(orgId);
      loginLog.setLoginIp(ServletUtils.getRemoteIp(ServletUtils.getCurrentRequest()));
      loginLog.setLoginTime(LocalDateTime.now());
      this.setClientCapabilities(loginLog);
      this.getDao().insert(loginLog);
   }

   public void insertLoginFailureLog(String userId, String remark) {
      CoreLoginLogBean loginLog = new CoreLoginLogBean();
      loginLog.setId(ApplicationContextHelper.getNextIdentity());
      loginLog.setUserId(userId);
      loginLog.setStatus("0");
      loginLog.setRemark(remark);
      loginLog.setLoginIp(ServletUtils.getRemoteIp(ServletUtils.getCurrentRequest()));
      loginLog.setLoginTime(LocalDateTime.now());
      this.setClientCapabilities(loginLog);
      this.getDao().insert(loginLog);
   }

   public List<CoreLoginLogBean> selectOnlineUserLoginLogList(@Nullable String userId) {
      CoreLoginLogBean logSearchTerms = (CoreLoginLogBean)BeanUtils.getPropertyListenerProxy(CoreLoginLogBean.class);
      logSearchTerms.setLogoutTime((LocalDateTime)null);
      logSearchTerms.setStatus("1");
      if (!StringUtils.isBlank(userId)) {
         logSearchTerms.setUserId(userId);
      }

      return this.getDao().selectList(logSearchTerms, new Order[0]);
   }

   public CoreLoginLogBean getCurrentLoginLog(String userId) {
      CoreLoginLogBean logSearchTerms = (CoreLoginLogBean)BeanUtils.getPropertyListenerProxy(CoreLoginLogBean.class);
      logSearchTerms.setToken(AuthenticationHelper.getLoginUserToken());
      logSearchTerms.setUserId(userId);
      logSearchTerms.setLogoutTime((LocalDateTime)null);
      logSearchTerms.setStatus("1");
      return (CoreLoginLogBean)this.getDao().selectOneIfPresent(logSearchTerms, new String[0]);
   }

   public Page<CoreLoginLogBean> selectOnlinePagination(RestJsonWrapperBean wrapper) {
      return this.selectPaginationByFilter(SearchFilter.instance().match((String)"LOGOUTTIME", null).filter(MatchPattern.EQ).match((String)"STATUS", "1").filter(MatchPattern.EQ), wrapper);
   }

   public void offline(RestJsonWrapperBean wrapper) {
      List<Long> loginLogIdList = wrapper.parseId(Long.class);
      List<CoreLoginLogBean> loginLogList = this.getDao().selectListByIds(loginLogIdList, new Order[0]);
      if (!loginLogList.isEmpty()) {
         String remark = "强制下线（操作人：" + LocalContextHelper.getLoginUserId() + "）";
         loginLogList.forEach((l) -> {
            AuthenticationHelper.markOffline(l.getToken(), AccountCategory.OFFLINE.name());
         });
         loginLogList.forEach((l) -> {
            AuthenticationHelper.logout(l.getToken(), remark);
         });
      }

   }

   @Transactional
   public void assignLogout(String token, String remark) {
      if (!StringUtils.isBlank(token)) {
         CoreLoginLogBean filter = new CoreLoginLogBean();
         filter.setToken(token);
         CoreLoginLogBean loginLog = (CoreLoginLogBean)BeanUtils.getPropertyListenerProxy((Object)this.getDao().selectOne(filter, new String[0]));
         loginLog.setLogoutTime(LocalDateTime.now());
         loginLog.setRemark(remark);
         this.getDao().update(loginLog, new String[0]);
      }
   }

   private void setClientCapabilities(CoreLoginLogBean loginLog) {
      Capabilities capabilities = ServletUtils.getCurrentRequestCapabilities();
      if (capabilities != null) {
         loginLog.setDevice(capabilities.getDeviceType());
         StringBuilder browser = new StringBuilder();
         if (!StringUtils.isEmpty(capabilities.getBrowser())) {
            browser.append(capabilities.getBrowser());
         }

         if (!StringUtils.isEmpty(capabilities.getBrowserMajorVersion())) {
            if (browser.length() > 0) {
               browser.append(" ");
            }

            browser.append(capabilities.getBrowserMajorVersion());
         }

         if (!StringUtils.isEmpty(capabilities.getBrowserType())) {
            if (browser.length() > 0) {
               browser.append(" ");
            }

            browser.append(capabilities.getBrowserType());
         }

         loginLog.setBrowser(browser.toString());
         StringBuilder platform = new StringBuilder(capabilities.getPlatform());
         if (!StringUtils.isEmpty(capabilities.getPlatformVersion())) {
            platform.append(" ").append(capabilities.getPlatformVersion());
         }

         loginLog.setPlatform(platform.toString());
      }

   }
}
