/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.module.sys.bpmn.service.impl;

import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.context.LocalContextHelper;
import net.sinodawn.framework.data.page.Page;
import net.sinodawn.framework.database.sql.Order;
import net.sinodawn.framework.i18n.I18nHelper;
import net.sinodawn.framework.mybatis.mapper.MatchPattern;
import net.sinodawn.framework.mybatis.mapper.SearchFilter;
import net.sinodawn.framework.restful.data.RestJsonWrapperBean;
import net.sinodawn.framework.support.base.service.GenericService;
import net.sinodawn.framework.support.domain.Persistable;
import net.sinodawn.framework.utils.CollectionUtils;
import net.sinodawn.framework.utils.ConvertUtils;
import net.sinodawn.framework.utils.StringUtils;
import net.sinodawn.module.sys.bpmn.CoreBpmnHelper;
import net.sinodawn.module.sys.bpmn.bean.*;
import net.sinodawn.module.sys.bpmn.diagram.BpmnDiagramHelper;
import net.sinodawn.module.sys.bpmn.diagram.CommentStatus;
import net.sinodawn.module.sys.bpmn.diagram.ProcessStatus;
import net.sinodawn.module.sys.bpmn.diagram.TaskStatus;
import net.sinodawn.module.sys.bpmn.exception.BpmnException;
import net.sinodawn.module.sys.bpmn.service.*;
import org.dom4j.Document;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Repository;

import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

@Repository
public class CoreBpmnExtServiceImpl implements CoreBpmnExtService {
   @Autowired
   private CoreBpmnInstanceService instanceService;
   @Autowired
   private CoreBpmnProcService procService;
   @Autowired
   private CoreBpmnCommentService commentService;
   @Autowired
   private CoreBpmnDiagramService diagramService;
   @Autowired
   private CoreBpmnInstanceTaskService instanceTaskService;
   @Autowired
   private CoreBpmnTargetService targetService;

   public <T extends Persistable<ID>, ID extends Serializable> CoreBpmnRuntimeDTO selectBpmnRuntimeInfo(String table, String id) {
      CoreBpmnRuntimeDTO runtime = new CoreBpmnRuntimeDTO();
      CoreBpmnInstanceBean filter = new CoreBpmnInstanceBean();
      filter.setTargetId(CoreBpmnHelper.getTargetId(table, id));
      CoreBpmnInstanceBean instance = this.instanceService.getDao().selectOneIfPresent(filter);
      if (instance != null) {
         runtime.setInstanceId(instance.getId());
         runtime.setProcId(instance.getProcId());
         runtime.setDiagramId(instance.getDiagramId());
      } else {
         GenericService<T, ID> service = ApplicationContextHelper.getServiceByTable(table);
         String processStatus = null;
         Long procId = null;
         if (service != null) {
            Class<ID> idType = (Class<ID>) service.getDao().getEntityContext().getIdContext().getType();
            ID actualId = (ID) ConvertUtils.convert(id, idType);
            processStatus = service.selectColumnById(actualId, "PROCESSSTATUS");
            if (StringUtils.startsWithIgnoreCase(processStatus, ProcessStatus.DRAFT.name())) {
               procId = service.selectProcId(actualId);
            }
         }

         if (procId == null) {
            CoreBpmnCommentBean comment = (CoreBpmnCommentBean)this.commentService.selectFirstByFilter(SearchFilter.instance().match((String)"TARGETID", CoreBpmnHelper.getTargetId(table, id)).filter(MatchPattern.EQ).match((String)"PROCID", null).filter(MatchPattern.DIFFER), new Order[]{Order.desc("ID")});
            if (comment != null) {
               procId = comment.getProcId();
            }
         }

         runtime.setProcessStatus(processStatus);
         if (procId != null) {
            CoreBpmnProcBean proc = (CoreBpmnProcBean)this.procService.selectById(procId);
            runtime.setProcId(proc.getId());
            runtime.setDiagramId(proc.getDiagramId());
         }
      }

      return runtime;
   }

   public List<CoreBpmnRejectableTaskDTO> selectBpmnRuntimeRejectableTaskList(RestJsonWrapperBean wrapper) {
      List<String> idList = wrapper.parseId(String.class);
      String table = (String)Objects.requireNonNull(wrapper.getParamValue("bpmn_table"));
      String currentStatusCode = wrapper.getParamValue("bpmn_currentStatusCode");
      List<CoreBpmnRejectableTaskDTO> returnRejectableTaskList = new ArrayList();
      List<CoreBpmnCommentBean> commentFilterList = new ArrayList();
      Iterator var7 = idList.iterator();

      while(var7.hasNext()) {
         String id = (String)var7.next();
         CoreBpmnCommentBean commentFilter = new CoreBpmnCommentBean();
         commentFilter.setTargetId(CoreBpmnHelper.getTargetId(table, id));
         commentFilterList.add(commentFilter);
      }

      List<CoreBpmnCommentBean> selectedCommentList = this.commentService.getDao().selectList((List)commentFilterList, (List)Arrays.asList("TARGETID"), (List)CollectionUtils.emptyList(), Order.desc("ID"));
      if (!selectedCommentList.isEmpty()) {
         for(CoreBpmnCommentBean lastComment = (CoreBpmnCommentBean)selectedCommentList.get(0); CommentStatus.CS_APPROVE.name().equalsIgnoreCase(lastComment.getStatus()); lastComment = (CoreBpmnCommentBean)selectedCommentList.get(0)) {
            selectedCommentList.remove(0);
            if (selectedCommentList.isEmpty()) {
               break;
            }
         }
      }

      List<CoreBpmnInstanceBean> instanceFilterList = new ArrayList();
      Iterator var25 = idList.iterator();

      while(var25.hasNext()) {
         String id = (String)var25.next();
         CoreBpmnInstanceBean instanceFilter = new CoreBpmnInstanceBean();
         instanceFilter.setTargetId(CoreBpmnHelper.getTargetId(table, id));
         instanceFilterList.add(instanceFilter);
      }

      List<CoreBpmnInstanceBean> selectedInstanceList = this.instanceService.getDao().selectList((List)instanceFilterList, (List)Arrays.asList("TARGETID"), (List)CollectionUtils.emptyList());
      List<Long> instIdList = (List)selectedInstanceList.stream().map((i) -> {
         return i.getId();
      }).collect(Collectors.toList());
      List<CoreBpmnInstanceTaskBean> selectedInstanceTaskList = this.instanceTaskService.selectAuditableInstanceTaskList(LocalContextHelper.getLoginUserId(), instIdList);
      Iterator var12 = idList.iterator();

      while(var12.hasNext()) {
         String id = (String)var12.next();
         CoreBpmnInstanceBean instance = (CoreBpmnInstanceBean)selectedInstanceList.stream().filter((i) -> {
            return i.getTargetId().contentEquals(CoreBpmnHelper.getTargetId(table, id));
         }).findFirst().get();
         List<CoreBpmnInstanceTaskBean> taskList = (List)selectedInstanceTaskList.stream().filter((t) -> {
            return instance.getId().equals(t.getInstId());
         }).collect(Collectors.toList());
         if (taskList.isEmpty()) {
            throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.NOT_AUTHORIZED");
         }

         if (taskList.size() > 1 && !StringUtils.isEmpty(currentStatusCode)) {
            taskList.removeIf((t) -> {
               return !t.getStatusCode().equals(currentStatusCode);
            });
         }

         Document document = this.diagramService.getDocument(instance.getDiagramId());
         Set<String> previousStatusCodeList = new HashSet();
         Iterator var18 = taskList.iterator();

         while(var18.hasNext()) {
            CoreBpmnInstanceTaskBean task = (CoreBpmnInstanceTaskBean)var18.next();
            previousStatusCodeList.addAll(BpmnDiagramHelper.getPreviousStatusCodeList(document, BpmnDiagramHelper.getProcessElementByStatusCode(document, task.getStatusCode())));
         }

         if (previousStatusCodeList.isEmpty()) {
            CoreBpmnRejectableTaskDTO startEvent = new CoreBpmnRejectableTaskDTO();
            startEvent.setStatusCode(CommentStatus.SUBMIT.name());
            startEvent.setTaskName(I18nHelper.getMessage("CORE.MODULE.SYS.T_CORE_BPMN_DIAGRAM.START_NODE"));
            return Arrays.asList(startEvent);
         }

         List<CoreBpmnRejectableTaskDTO> rejectableTaskList = new ArrayList();
         Iterator var31 = selectedCommentList.iterator();

         while(var31.hasNext()) {
            CoreBpmnCommentBean comment = (CoreBpmnCommentBean)var31.next();
            if (comment.getTargetId().equals(CoreBpmnHelper.getTargetId(table, id))) {
               if (previousStatusCodeList.contains(comment.getStatusCode())) {
                  CoreBpmnRejectableTaskDTO rejectableTask = new CoreBpmnRejectableTaskDTO();
                  rejectableTask.setStatusCode(comment.getStatusCode());
                  rejectableTask.setTaskName(comment.getTaskName());
                  if (StringUtils.startsWithIgnoreCase(rejectableTask.getStatusCode(), ProcessStatus.DRAFT.name()) && StringUtils.isEmpty(rejectableTask.getTaskName())) {
                     rejectableTask.setTaskName(I18nHelper.getMessage("SINO.BPMN.COMMENT.START_NODE"));
                  }

                  if (!rejectableTaskList.contains(rejectableTask)) {
                     rejectableTaskList.add(rejectableTask);
                  }
               }

               if (CommentStatus.SUBMIT.name().equalsIgnoreCase(comment.getStatus())) {
                  break;
               }
            }
         }

         if (returnRejectableTaskList.isEmpty()) {
            returnRejectableTaskList.addAll(rejectableTaskList);
         } else {
            returnRejectableTaskList.removeIf((r) -> {
               return !rejectableTaskList.contains(r);
            });
         }
      }

      return returnRejectableTaskList;
   }

   public String selectBpmnDiagramSvg(Long id, Long instanceId) {
      CoreBpmnDiagramBean diagram = (CoreBpmnDiagramBean)this.diagramService.selectById(id);
      String svg = diagram.getSvg();
      if (instanceId != null) {
         CoreBpmnInstanceBean instance = (CoreBpmnInstanceBean)this.instanceService.selectById(instanceId);
         CoreBpmnCommentBean commentFilter = new CoreBpmnCommentBean();
         commentFilter.setTargetId(instance.getTargetId());
         List<CoreBpmnCommentBean> commentList = this.commentService.selectList(commentFilter, new Order[]{Order.desc("ID")});
         Map<String, CommentStatus> map = new HashMap();
         Iterator var9 = commentList.iterator();

         while(var9.hasNext()) {
            CoreBpmnCommentBean comment = (CoreBpmnCommentBean)var9.next();
            if (!map.containsKey(comment.getTaskId())) {
               map.put(comment.getTaskId(), CommentStatus.valueOf(comment.getStatus()));
            }

            if (CommentStatus.SUBMIT.name().equals(comment.getStatus())) {
               break;
            }
         }

         CoreBpmnInstanceTaskBean taskFilter = new CoreBpmnInstanceTaskBean();
         taskFilter.setInstId(instanceId);
         taskFilter.setStatus(TaskStatus.CREATED.name());
         List<CoreBpmnInstanceTaskBean> taskList = this.instanceTaskService.selectList(taskFilter, new Order[0]);
         Iterator var11 = taskList.iterator();

         while(var11.hasNext()) {
            CoreBpmnInstanceTaskBean task = (CoreBpmnInstanceTaskBean)var11.next();
            map.put(task.getTaskId(), null);
         }

         svg = this.fillColor(svg, map);
      }

      return svg;
   }

   private String fillColor(String svg, Map<String, CommentStatus> map) {
      int dataElementIdPos = 0;

      while(true) {
         dataElementIdPos = svg.indexOf("data-element-id=\"", dataElementIdPos + 1);
         if (dataElementIdPos <= 0) {
            return svg;
         }

         int dataElementIdStartPos = svg.indexOf("\"", dataElementIdPos + 1);
         int dataElementIdEndPos = svg.indexOf("\"", dataElementIdStartPos + 1);
         String dataElementId = svg.substring(dataElementIdStartPos + 1, dataElementIdEndPos);
         if (!StringUtils.startsWithIgnoreCase(dataElementId, "SequenceFlow")) {
            svg = this.updateBackgroundColor(svg, dataElementId, (CommentStatus)map.get(dataElementId), map.containsKey(dataElementId));
            svg = this.updateStrokeColor(svg, dataElementId, (CommentStatus)map.get(dataElementId), map.containsKey(dataElementId));
            svg = this.updateTextColor(svg, dataElementId, (CommentStatus)map.get(dataElementId), map.containsKey(dataElementId));
         }
      }
   }

   private String updateBackgroundColor(String svg, String dataElementId, CommentStatus status, boolean todo) {
      int elementPos = svg.indexOf("data-element-id=\"" + dataElementId + "\"");
      if (elementPos <= 0) {
         return svg;
      } else {
         int fillPos = svg.indexOf("fill:", elementPos);
         if (fillPos <= 0) {
            return svg;
         } else {
            int semicolonPos = svg.indexOf(";", fillPos);
            if (semicolonPos <= 0) {
               return svg;
            } else {
               if (!CommentStatus.SUBMIT.equals(status)) {
                  if (!CommentStatus.APPROVE.equals(status) && !CommentStatus.CS_END.equals(status)) {
                     if (CommentStatus.REJECT.equals(status)) {
                        svg = svg.substring(0, fillPos) + "fill:#ffecec" + svg.substring(semicolonPos);
                     } else if (status == null) {
                        if (todo) {
                           svg = svg.substring(0, fillPos) + "fill:#e7f4ff" + svg.substring(semicolonPos);
                        } else {
                           svg = svg.substring(0, fillPos) + "fill:#f4f4f5" + svg.substring(semicolonPos);
                        }
                     }
                  } else {
                     svg = svg.substring(0, fillPos) + "fill:#e7faef" + svg.substring(semicolonPos);
                  }
               }

               return svg;
            }
         }
      }
   }

   private String updateStrokeColor(String svg, String dataElementId, CommentStatus status, boolean todo) {
      int elementPos = svg.indexOf("data-element-id=\"" + dataElementId + "\"");
      if (elementPos <= 0) {
         return svg;
      } else {
         int strokePos = svg.indexOf("stroke:", elementPos);
         if (strokePos <= 0) {
            return svg;
         } else {
            int semicolonPos = svg.indexOf(";", strokePos);
            if (semicolonPos <= 0) {
               return svg;
            } else {
               if (CommentStatus.SUBMIT.equals(status)) {
                  svg = svg.substring(0, strokePos) + "stroke:#a1ebc2" + svg.substring(semicolonPos);
               } else if (!CommentStatus.APPROVE.equals(status) && !CommentStatus.CS_END.equals(status)) {
                  if (CommentStatus.REJECT.equals(status)) {
                     svg = svg.substring(0, strokePos) + "stroke:#ffb6b6" + svg.substring(semicolonPos);
                  } else if (status == null) {
                     if (todo) {
                        svg = svg.substring(0, strokePos) + "stroke:#a3d3ff" + svg.substring(semicolonPos);
                     } else {
                        svg = svg.substring(0, strokePos) + "stroke:#d3d4d6" + svg.substring(semicolonPos);
                     }
                  }
               } else {
                  svg = svg.substring(0, strokePos) + "stroke:#a1ebc2" + svg.substring(semicolonPos);
               }

               return svg;
            }
         }
      }
   }

   private String updateTextColor(String svg, String dataElementId, CommentStatus status, boolean todo) {
      int elementPos = svg.indexOf("data-element-id=\"" + dataElementId + "\"");
      if (elementPos <= 0) {
         return svg;
      } else {
         int textPos = svg.indexOf("<text ", elementPos);
         if (textPos <= 0) {
            return svg;
         } else {
            int fillPos = svg.indexOf("fill:", textPos);
            if (fillPos <= 0) {
               return svg;
            } else {
               int semicolonPos = svg.indexOf(";", fillPos);
               if (!CommentStatus.SUBMIT.equals(status)) {
                  if (!CommentStatus.APPROVE.equals(status) && !CommentStatus.CS_END.equals(status)) {
                     if (CommentStatus.REJECT.equals(status)) {
                        svg = svg.substring(0, fillPos) + "fill:#ff4949" + svg.substring(semicolonPos);
                     } else if (status == null) {
                        if (todo) {
                           svg = svg.substring(0, fillPos) + "fill:#1890ff" + svg.substring(semicolonPos);
                        } else {
                           svg = svg.substring(0, fillPos) + "fill:#000; fill-opacity: 0.65" + svg.substring(semicolonPos);
                        }
                     }
                  } else {
                     svg = svg.substring(0, fillPos) + "fill:#13ce66" + svg.substring(semicolonPos);
                  }
               }

               return svg;
            }
         }
      }
   }

   public Page<CoreBpmnTargetBean> selectBpmnRuntimeTargetList(RestJsonWrapperBean wrapper) {
      return this.targetService.selectPaginationByFilter(SearchFilter.instance().match((String)"PROCESSSTATUS", "APPROVE").filter(MatchPattern.SB), wrapper);
   }
}
