/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.module.sys.bpmn.service.impl;

import net.sinodawn.framework.at.annotation.AuditTrailEntry;
import net.sinodawn.framework.at.annotation.AuditTrailType;
import net.sinodawn.framework.audit.aunnotation.Audit;
import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.data.page.Page;
import net.sinodawn.framework.database.context.instance.EntityHelper;
import net.sinodawn.framework.database.sql.Order;
import net.sinodawn.framework.mybatis.mapper.MapperParameter;
import net.sinodawn.framework.mybatis.mapper.MatchPattern;
import net.sinodawn.framework.mybatis.mapper.SearchFilter;
import net.sinodawn.framework.restful.data.RestJsonWrapperBean;
import net.sinodawn.framework.utils.BeanUtils;
import net.sinodawn.module.sys.bpmn.bean.*;
import net.sinodawn.module.sys.bpmn.dao.CoreBpmnDraftDao;
import net.sinodawn.module.sys.bpmn.service.*;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

@Repository
public class CoreBpmnDraftServiceImpl implements CoreBpmnDraftService {
   @Autowired
   private CoreBpmnDraftDao draftDao;
   @Autowired
   private CoreBpmnDraftOrgService draftOrgService;
   @Autowired
   private CoreBpmnDiagramService diagramService;
   @Autowired
   @Lazy
   private CoreBpmnProcService procService;
   @Autowired
   @Lazy
   private CoreBpmnProcOrgService procOrgService;

   public CoreBpmnDraftDao getDao() {
      return this.draftDao;
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.INSERT)
   @Audit("AUDIT.INSERT")
   public Long insert(RestJsonWrapperBean wrapper) {
      CoreBpmnDiagramBean diagram = new CoreBpmnDiagramBean();
      diagram.setId(ApplicationContextHelper.getNextIdentity());
      this.diagramService.getDao().insert(diagram);
      CoreBpmnDraftBean draft = wrapper.parseUnique(CoreBpmnDraftBean.class);
      draft.setId(ApplicationContextHelper.getNextIdentity());
      draft.setDiagramId(diagram.getId());
      this.getDao().insert(draft);
      return draft.getId();
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.INSERT)
   @Audit("AUDIT.INSERT")
   public Long insertByProcId(Long procId) {
      CoreBpmnProcBean proc = (CoreBpmnProcBean)this.procService.selectById(procId);
      CoreBpmnDiagramBean procDiagram = (CoreBpmnDiagramBean)this.diagramService.selectById(proc.getDiagramId());
      CoreBpmnProcOrgBean procOrgFilter = new CoreBpmnProcOrgBean();
      procOrgFilter.setProcId(procId);
      List<CoreBpmnProcOrgBean> procOrgList = this.procOrgService.selectList(procOrgFilter, new Order[]{Order.asc("ID")});
      CoreBpmnDiagramBean diagram = new CoreBpmnDiagramBean();
      BeanUtils.copyProperties((Object)procDiagram, diagram);
      diagram.setId(ApplicationContextHelper.getNextIdentity());
      EntityHelper.assignCreatedElement(diagram);
      this.diagramService.getDao().insert(diagram);
      CoreBpmnDraftBean draft = new CoreBpmnDraftBean();
      BeanUtils.copyProperties((Object)proc, draft);
      draft.setId(ApplicationContextHelper.getNextIdentity());
      draft.setProcVersion(draft.getProcVersion() + 1L);
      draft.setDiagramId(diagram.getId());
      EntityHelper.assignCreatedElement(draft);
      this.getDao().insert(draft);
      if (!procOrgList.isEmpty()) {
         List<CoreBpmnDraftOrgBean> draftOrgList = new ArrayList();
         Iterator var9 = procOrgList.iterator();

         while(var9.hasNext()) {
            CoreBpmnProcOrgBean procOrg = (CoreBpmnProcOrgBean)var9.next();
            CoreBpmnDraftOrgBean draftOrg = new CoreBpmnDraftOrgBean();
            draftOrg.setId(ApplicationContextHelper.getNextIdentity());
            draftOrg.setDraftId(draft.getId());
            draftOrg.setUsedOrgId(procOrg.getUsedOrgId());
            draftOrgList.add(draftOrg);
         }

         this.draftOrgService.getDao().insert((List)draftOrgList);
      }

      return draft.getId();
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.INSERT)
   @Audit("AUDIT.INSERT")
   public void submit(Long id) {
      CoreBpmnDraftBean draft = this.selectById(id);
      CoreBpmnDraftOrgBean draftOrgFiltre = new CoreBpmnDraftOrgBean();
      draftOrgFiltre.setDraftId(id);
      List<CoreBpmnDraftOrgBean> draftOrgList = this.draftOrgService.selectList(draftOrgFiltre, new Order[]{Order.asc("ID")});
      CoreBpmnProcBean proc = new CoreBpmnProcBean();
      BeanUtils.copyProperties(draft, proc);
      proc.setId(ApplicationContextHelper.getNextIdentity());
      EntityHelper.assignCreatedElement(proc);
      this.procService.getDao().insert(proc);
      if (!draftOrgList.isEmpty()) {
         List<CoreBpmnProcOrgBean> procOrgList = new ArrayList();
         Iterator var7 = draftOrgList.iterator();

         while(var7.hasNext()) {
            CoreBpmnDraftOrgBean draftOrg = (CoreBpmnDraftOrgBean)var7.next();
            CoreBpmnProcOrgBean procOrg = new CoreBpmnProcOrgBean();
            procOrg.setId(ApplicationContextHelper.getNextIdentity());
            procOrg.setProcId(proc.getId());
            procOrg.setUsedOrgId(draftOrg.getUsedOrgId());
            procOrgList.add(procOrg);
         }

         this.procOrgService.getDao().insert((List)procOrgList);
      }

      this.getDao().delete(draft.getId());
      this.draftOrgService.getDao().deleteBy(draftOrgList);
      this.procService.cacheEvictLatestProc(draft.getProcCode());
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.DELETE)
   @Audit("AUDIT.DELETE")
   public void delete(RestJsonWrapperBean wrapper) {
      List<CoreBpmnDraftBean> deleteList = wrapper.parse(CoreBpmnDraftBean.class);
      if (!deleteList.isEmpty()) {
         List<Long> diagramIdList = (List)deleteList.stream().map((d) -> {
            return d.getDiagramId();
         }).collect(Collectors.toList());
         diagramIdList.stream().filter((i) -> {
            return i != null;
         }).forEach((i) -> {
            this.diagramService.delete(i);
         });
         this.getDao().deleteBy(deleteList, new String[0]);
      }
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.IMPORT)
   @Audit("AUDIT.IMPORT")
   public void importBpmn(List<CoreBpmnImportDTO> bpmnImportList) {
      List<String> procCodeList = (List)bpmnImportList.stream().map((i) -> {
         return i.getProcCode();
      }).collect(Collectors.toList());
      List<CoreBpmnDraftBean> selectedDraftList = this.getDao().selectListByOneColumnValues(procCodeList, "PROCCODE", new Order[0]);
      this.getDao().deleteBy(selectedDraftList, new String[0]);
      List<Long> diagramIdList = (List)selectedDraftList.stream().map((d) -> {
         return d.getDiagramId();
      }).collect(Collectors.toList());
      diagramIdList.stream().filter((i) -> {
         return i != null;
      }).forEach((i) -> {
         this.diagramService.delete(i);
      });
      List<CoreBpmnDiagramBean> insertDiagramList = new ArrayList();
      List<CoreBpmnDraftBean> insertDraftList = new ArrayList();
      List<CoreBpmnDraftOrgBean> insertDraftOrgList = new ArrayList();

      CoreBpmnDraftBean draft;
      for(Iterator var8 = bpmnImportList.iterator(); var8.hasNext(); insertDraftList.add(draft)) {
         CoreBpmnImportDTO bpmnImport = (CoreBpmnImportDTO)var8.next();
         CoreBpmnDiagramBean diagram = new CoreBpmnDiagramBean();
         diagram.setId(ApplicationContextHelper.getNextIdentity());
         diagram.setDiagram(bpmnImport.getDiagram());
         diagram.setSvg(bpmnImport.getSvg());
         insertDiagramList.add(diagram);
         draft = new CoreBpmnDraftBean();
         draft.setId(ApplicationContextHelper.getNextIdentity());
         draft.setProcCode(bpmnImport.getProcCode());
         draft.setServiceId(bpmnImport.getServiceId());
         draft.setExpression(bpmnImport.getExpression());
         draft.setDiagramId(diagram.getId());
         if (bpmnImport.getUsedOrgIdList() != null) {
            Iterator var12 = bpmnImport.getUsedOrgIdList().iterator();

            while(var12.hasNext()) {
               String usedOrgId = (String)var12.next();
               CoreBpmnDraftOrgBean draftOrg = new CoreBpmnDraftOrgBean();
               draftOrg.setId(ApplicationContextHelper.getNextIdentity());
               draftOrg.setDraftId(draft.getId());
               draftOrg.setUsedOrgId(usedOrgId);
               insertDraftOrgList.add(draftOrg);
            }
         }

         MapperParameter parameter = new MapperParameter();
         parameter.setFilter(SearchFilter.instance().match((String)"PROCCODE", bpmnImport.getProcCode()).filter(MatchPattern.SEQ));
         parameter.put("service", "modifiable");
         List<CoreBpmnProcBean> procList = this.procService.selectList(parameter);
         if (!procList.isEmpty()) {
            draft.setProcVersion(((CoreBpmnProcBean)procList.get(0)).getProcVersion() + 1L);
         }
      }

      this.diagramService.getDao().insert((List)insertDiagramList);
      this.getDao().insert(insertDraftList);
      this.draftOrgService.getDao().insert((List)insertDraftOrgList);
   }

   public Page<CoreBpmnDraftOrgBean> selectDraftOrgPagination(Long id, RestJsonWrapperBean wrapper) {
      return this.draftOrgService.selectPaginationByFilter(SearchFilter.instance().match((String)"DRAFTID", id).filter(MatchPattern.EQ), wrapper);
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.INSERT)
   @Audit("AUDIT.INSERT")
   public void insertDraftOrg(Long id, RestJsonWrapperBean wrapper) {
      List<CoreBpmnDraftOrgBean> draftOrgList = wrapper.parse(CoreBpmnDraftOrgBean.class);
      draftOrgList.forEach((o) -> {
         o.setId(ApplicationContextHelper.getNextIdentity());
         o.setDraftId(id);
      });
      this.draftOrgService.getDao().insert(draftOrgList);
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.DELETE)
   @Audit("AUDIT.DELETE")
   public void deleteDraftOrg(Long id, RestJsonWrapperBean wrapper) {
      this.draftOrgService.delete(wrapper);
   }
}
