/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.module.sys.bpmn.service.impl;

import net.sinodawn.framework.cache.redis.annotation.SinoCacheable;
import net.sinodawn.framework.exception.UnsupportedException;
import net.sinodawn.framework.i18n.I18nHelper;
import net.sinodawn.framework.utils.*;
import net.sinodawn.module.sys.bpmn.CoreBpmnHelper;
import net.sinodawn.module.sys.bpmn.bean.CoreBpmnDiagramBean;
import net.sinodawn.module.sys.bpmn.bean.CoreBpmnDiagramSequenceFlowConditionDTO;
import net.sinodawn.module.sys.bpmn.bean.CoreBpmnDiagramTaskCandidatorDTO;
import net.sinodawn.module.sys.bpmn.dao.CoreBpmnDiagramDao;
import net.sinodawn.module.sys.bpmn.diagram.BpmnDiagramHelper;
import net.sinodawn.module.sys.bpmn.diagram.ProcessElementType;
import net.sinodawn.module.sys.bpmn.service.CoreBpmnDiagramService;
import org.dom4j.Document;
import org.dom4j.Element;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

@Repository
public class CoreBpmnDiagramServiceImpl implements CoreBpmnDiagramService {
   @Autowired
   private CoreBpmnDiagramDao diagramDao;
   @Autowired
   @Lazy
   private CoreBpmnDiagramService proxyInstance;

   public CoreBpmnDiagramDao getDao() {
      return this.diagramDao;
   }

   public String selectColumnById(Long id, String column) {
      CoreBpmnDiagramBean diagram = (CoreBpmnDiagramBean)this.proxyInstance.selectById(id);
      Method method = ReflectionUtils.findReadMethod(this.getDao().getType(), column);
      return method != null ? (String)ConvertUtils.convert(ReflectionUtils.invokeMethod(method, diagram), String.class) : diagram.getExt$Item(column);
   }

   @SinoCacheable({"T_CORE_BPMN_DIAGRAM.DOCUMENT"})
   public Document getDocument(Long id) {
      return XmlUtils.read(CoreBpmnHelper.getCorrectedDiagram(this.selectColumnById(id, "DIAGRAM")));
   }

   @SinoCacheable({"T_CORE_BPMN_DIAGRAM.TASKCANDIDATOR"})
   public List<CoreBpmnDiagramTaskCandidatorDTO> selectTaskCandidatorList(Long id) {
      List<CoreBpmnDiagramTaskCandidatorDTO> taskCandidatorList = new ArrayList();
      Document document = this.proxyInstance.getDocument(id);
      if (document == null) {
         return CollectionUtils.emptyList();
      } else {
         List<Element> userTaskList = BpmnDiagramHelper.getProcessElementList(document, ProcessElementType.USER_TASK);

         CoreBpmnDiagramTaskCandidatorDTO taskCandidator;
         for(Iterator var5 = userTaskList.iterator(); var5.hasNext(); taskCandidatorList.add(taskCandidator)) {
            Element userTask = (Element)var5.next();
            taskCandidator = new CoreBpmnDiagramTaskCandidatorDTO();
            taskCandidator.setStatusCode(XmlUtils.getAttributeValue(userTask, "statusCode"));
            taskCandidator.setTaskName(XmlUtils.getAttributeValue(userTask, "name"));
            taskCandidator.setInitiator(StringUtils.isEmpty(BpmnDiagramHelper.getUserTaskInitiatorValue(userTask)) ? "0" : "1");
            taskCandidator.setDynamic(BpmnDiagramHelper.getUserTaskDynamicValue(userTask));
            Map<String, String> userMap = BpmnDiagramHelper.getUserTaskCandidatorValue(userTask);
            if (!userMap.isEmpty()) {
               taskCandidator.setUser(StringUtils.join(userMap.values(), ","));
            }

            Map<Long, String> roleMap = BpmnDiagramHelper.getUserTaskRoleValue(userTask);
            if (!roleMap.isEmpty()) {
               taskCandidator.setRole(StringUtils.join(roleMap.values(), ","));
            }
         }

         return taskCandidatorList;
      }
   }

   @SinoCacheable({"T_CORE_BPMN_DIAGRAM.SEQUENCEFLOWCONDITION"})
   public List<CoreBpmnDiagramSequenceFlowConditionDTO> selectSequenceFlowConditionList(Long id) {
      List<CoreBpmnDiagramSequenceFlowConditionDTO> conditionList = new ArrayList();
      Document document = this.proxyInstance.getDocument(id);
      if (document == null) {
         return CollectionUtils.emptyList();
      } else {
         List<Element> sequenceFlowList = BpmnDiagramHelper.getProcessElementList(document, ProcessElementType.SEQUENCE_FLOW);
         Iterator var5 = sequenceFlowList.iterator();

         while(var5.hasNext()) {
            Element sequenceFlow = (Element)var5.next();
            String conditionExpression = BpmnDiagramHelper.getConditionExpression(sequenceFlow);
            if (!StringUtils.isEmpty(conditionExpression)) {
               CoreBpmnDiagramSequenceFlowConditionDTO condition = new CoreBpmnDiagramSequenceFlowConditionDTO();
               condition.setConditionExpression(conditionExpression);
               Element sourceElement = BpmnDiagramHelper.getSourceElement(document, sequenceFlow);
               condition.setSourceTask(this.getElementDesc(sourceElement));
               Element targetElement = BpmnDiagramHelper.getTargetElement(document, sequenceFlow);
               condition.setTargetTask(this.getElementDesc(targetElement));
               conditionList.add(condition);
            }
         }

         return conditionList;
      }
   }

   private String getElementDesc(Element element) {
      String name = XmlUtils.getAttributeValue(element, "name");
      StringBuilder sb = new StringBuilder();
      if (!StringUtils.isEmpty(name)) {
         sb.append(name).append("(");
      }

      if (ProcessElementType.START_EVENT.getName().equals(element.getName())) {
         sb.append(I18nHelper.getMessage("CORE.MODULE.SYS.T_CORE_BPMN_DIAGRAM.START_NODE"));
      } else if (ProcessElementType.EXCLUSIVE_GATEWAY.getName().equals(element.getName())) {
         sb.append(I18nHelper.getMessage("CORE.MODULE.SYS.T_CORE_BPMN_DIAGRAM.EXCLUSIVE_GATEWAY"));
      } else if (ProcessElementType.PARALLEL_GATEWAY.getName().equals(element.getName())) {
         sb.append(I18nHelper.getMessage("CORE.MODULE.SYS.T_CORE_BPMN_DIAGRAM.PARALLEL_GATEWAY"));
      } else if (ProcessElementType.END_EVENT.getName().equals(element.getName())) {
         sb.append(I18nHelper.getMessage("CORE.MODULE.SYS.T_CORE_BPMN_DIAGRAM.END_NODE"));
      } else {
         if (!ProcessElementType.USER_TASK.getName().equals(element.getName())) {
            throw new UnsupportedException(element.asXML());
         }

         sb.append(I18nHelper.getMessage("CORE.MODULE.SYS.T_CORE_BPMN_DIAGRAM.USER_TASK"));
      }

      if (!StringUtils.isEmpty(name)) {
         sb.append(")");
      }

      return sb.toString();
   }
}
