/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.module.sys.bpmn.engine.support;

import com.alibaba.fastjson.JSON;
import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.context.LocalContextHelper;
import net.sinodawn.framework.context.SinoBeanContext;
import net.sinodawn.framework.exception.checked.CheckedException;
import net.sinodawn.framework.executor.ExecutorHelper;
import net.sinodawn.framework.i18n.I18nHelper;
import net.sinodawn.framework.mybatis.mapper.GlobalMapper;
import net.sinodawn.framework.security.bean.LoginUser;
import net.sinodawn.framework.support.base.service.GenericService;
import net.sinodawn.framework.support.domain.Auditable;
import net.sinodawn.framework.utils.*;
import net.sinodawn.module.mdm.user.bean.CoreUserBean;
import net.sinodawn.module.mdm.user.service.CoreUserService;
import net.sinodawn.module.sys.bpmn.bean.*;
import net.sinodawn.module.sys.bpmn.diagram.BpmnDiagramHelper;
import net.sinodawn.module.sys.bpmn.diagram.CommentStatus;
import net.sinodawn.module.sys.bpmn.diagram.ProcessStatus;
import net.sinodawn.module.sys.bpmn.diagram.TaskStatus;
import net.sinodawn.module.sys.bpmn.diagram.attribute.CandidatorFilterStrategy;
import net.sinodawn.module.sys.bpmn.diagram.attribute.CounterSignStrategy;
import net.sinodawn.module.sys.bpmn.engine.CoreBpmnRuntimeService;
import net.sinodawn.module.sys.bpmn.engine.CoreBpmnRuntimeSource;
import net.sinodawn.module.sys.bpmn.engine.cache.BpmnRuntimeCacheProvider;
import net.sinodawn.module.sys.bpmn.engine.cache.BpmnRuntimeData;
import net.sinodawn.module.sys.bpmn.event.CoreBpmnNotificationEvent;
import net.sinodawn.module.sys.bpmn.event.CoreBpmnUpdateStatusEvent;
import net.sinodawn.module.sys.bpmn.exception.BpmnException;
import net.sinodawn.module.sys.bpmn.service.*;
import net.sinodawn.module.sys.role.bean.CoreRoleBean;
import net.sinodawn.module.sys.role.service.CoreRoleService;
import org.dom4j.Document;
import org.dom4j.Element;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.context.ApplicationContext;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.util.*;
import java.util.stream.Collectors;

public abstract class AbstractCoreBpmnRuntimeProcessSupport<T extends Auditable<ID>, ID extends Serializable> {
   private static final Logger logger = LoggerFactory.getLogger(AbstractCoreBpmnRuntimeProcessSupport.class);
   protected static final String BPMN_VALIDATOR_SQL_SN = "SN_";
   protected static final String BPMN_VALIDATOR_SQL_TYPE = "TYPE_";
   protected static final String BPMN_VALIDATOR_SQL_DESC = "DESC_";
   protected static final String BPMN_VALIDATOR_SQL_TABLE = "TABLE_";
   protected final GlobalMapper globalMapper = (GlobalMapper)ApplicationContextHelper.getBean(GlobalMapper.class);
   protected final CoreBpmnProcService procService = (CoreBpmnProcService)ApplicationContextHelper.getBean(CoreBpmnProcService.class);
   protected final CoreBpmnDiagramService diagramService = (CoreBpmnDiagramService)ApplicationContextHelper.getBean(CoreBpmnDiagramService.class);
   protected final CoreBpmnInstanceService instanceService = (CoreBpmnInstanceService)ApplicationContextHelper.getBean(CoreBpmnInstanceService.class);
   protected final CoreBpmnInstanceTaskService instanceTaskService = (CoreBpmnInstanceTaskService)ApplicationContextHelper.getBean(CoreBpmnInstanceTaskService.class);
   protected final CoreBpmnInstanceTaskUserService instanceTaskUserService = (CoreBpmnInstanceTaskUserService)ApplicationContextHelper.getBean(CoreBpmnInstanceTaskUserService.class);
   protected final CoreBpmnInstanceTaskRoleService instanceTaskRoleService = (CoreBpmnInstanceTaskRoleService)ApplicationContextHelper.getBean(CoreBpmnInstanceTaskRoleService.class);
   protected final CoreBpmnCommentService commentService = (CoreBpmnCommentService)ApplicationContextHelper.getBean(CoreBpmnCommentService.class);
//   protected final CoreMenuExtService menuExtService = (CoreMenuExtService)ApplicationContextHelper.getBean(CoreMenuExtService.class);
   protected final CoreBpmnTargetService bpmnTargetService = (CoreBpmnTargetService)ApplicationContextHelper.getBean(CoreBpmnTargetService.class);
   protected final CoreBpmnInstanceTaskHisService taskHisService = (CoreBpmnInstanceTaskHisService)ApplicationContextHelper.getBean(CoreBpmnInstanceTaskHisService.class);

   public List<CoreBpmnInstanceStatusDTO<ID>> apply(List<CoreBpmnRuntimeSource<T, ID>> sourceList) {
      List var7;
      try {
         Collections.sort(sourceList, (s1, s2) -> {
            return ObjectUtils.toString(s1.getId()).compareTo(ObjectUtils.toString(s2.getId()));
         });
         BpmnRuntimeCacheProvider.init(sourceList);
         this.doValidate(sourceList);
         this.validateBySql();
         List<CoreBpmnInstanceStatusDTO<ID>> statusList = this.doProcess(sourceList);
         this.processData();
         String tableName = sourceList.get(0).getTableName();
         this.updateProcessStatus(tableName, statusList);
         CoreBpmnUpdateStatusEvent event = new CoreBpmnUpdateStatusEvent(tableName, statusList);
         ApplicationContextHelper.getApplicationContext().publishEvent(event);
         Map<String, List<ID>> callback = BpmnRuntimeCacheProvider.getBpmnRuntimeCallback();
         if (!CollectionUtils.isEmpty(callback)) {
            this.doCallback(callback);
         }

         Map<String, List<ID>> endCallback = BpmnRuntimeCacheProvider.getBpmnRuntimeEndCallback();
         if (!CollectionUtils.isEmpty(endCallback)) {
            this.doCallback(endCallback);
         }

         if (BpmnRuntimeCacheProvider.getProcessData().isAsync()) {
            ExecutorHelper.submitAfterCommitTransaction(() -> {
               this.sendNotification(sourceList, statusList);
            });
         } else {
            this.sendNotification(sourceList, statusList);
         }

         var7 = statusList;
      } finally {
         BpmnRuntimeCacheProvider.remove();
      }

      return var7;
   }

   protected void doValidate(List<CoreBpmnRuntimeSource<T, ID>> sourceList) {
   }

   protected List<CoreBpmnInstanceStatusDTO<ID>> doProcess(List<CoreBpmnRuntimeSource<T, ID>> sourceList) {
      return Collections.emptyList();
   }

   protected void sendNotification(List<CoreBpmnRuntimeSource<T, ID>> sourceList, List<CoreBpmnInstanceStatusDTO<ID>> statusList) {
      ApplicationContext applicationContext = ApplicationContextHelper.getApplicationContext();
      List<CoreBpmnInstanceStatusDTO<ID>> auditingInstanceStatusList = new ArrayList();
      Iterator var5 = statusList.iterator();

      while(true) {
         while(true) {
            CoreBpmnInstanceStatusDTO instanceStatus;
            do {
               do {
                  if (!var5.hasNext()) {
                     if (!auditingInstanceStatusList.isEmpty()) {
                        List<CoreBpmnRuntimeSource<T, ID>> auditingSourceList = (List)sourceList.stream().filter((s) -> {
                           return auditingInstanceStatusList.stream().anyMatch((a) -> {
                              return a.getId().equals(s.getId());
                           });
                        }).collect(Collectors.toList());
                        List<CoreBpmnInstanceBean> instanceList = this.getBpmnRuntimeInstanceListFromDatabase(auditingSourceList);
                        List<Long> idList = (List)instanceList.stream().map((i) -> {
                           return i.getId();
                        }).collect(Collectors.toList());
                        List<CoreBpmnInstanceTaskBean> taskList = this.instanceTaskService.getDao().selectListByOneColumnValues(idList, "INSTID");
                        List<CoreBpmnInstanceTaskCandidatorDTO> instanceTaskCandidatorList = this.instanceService.selectInstanceTaskAuditorList(idList);
                        Iterator var10 = taskList.iterator();

                        while(var10.hasNext()) {
                           CoreBpmnInstanceTaskBean task = (CoreBpmnInstanceTaskBean)var10.next();
                           CoreBpmnInstanceBean instance = (CoreBpmnInstanceBean)instanceList.stream().filter((i) -> {
                              return i.getId().equals(task.getInstId());
                           }).findAny().get();
                           CoreBpmnRuntimeSource<T, ID> source = (CoreBpmnRuntimeSource)sourceList.stream().filter((s) -> {
                              return s.getTargetId().equals(instance.getTargetId());
                           }).findAny().get();
                           CoreBpmnInstanceStatusDTO<ID> status = (CoreBpmnInstanceStatusDTO)statusList.stream().filter((s) -> {
                              return s.getId().equals(source.getId());
                           }).findAny().get();
                           List<CoreBpmnInstanceTaskCandidatorDTO> candidatorList = (List)instanceTaskCandidatorList.stream().filter((i) -> {
                              return i.getInstTaskId().equals(task.getId());
                           }).collect(Collectors.toList());
                           if (!candidatorList.isEmpty()) {
                              CoreBpmnNotificationEvent event = new CoreBpmnNotificationEvent(status.getProcessStatus(), I18nHelper.getMessage(source.getTableName().toUpperCase() + ".SERVICE_NAME"), source.getDesc(), task.getTaskName(), LocalContextHelper.getLoginUserId(), source.getComment(), (List)candidatorList.stream().map((c) -> {
                                 return c.getCandidatorId();
                              }).distinct().collect(Collectors.toList()));
                              applicationContext.publishEvent(event);
                           }
                        }
                     }

                     return;
                  }

                  instanceStatus = (CoreBpmnInstanceStatusDTO)var5.next();
               } while(ProcessStatus.DONE.equals(instanceStatus.getProcessStatus()));
            } while(ProcessStatus.DRAFT_UNDO.equals(instanceStatus.getProcessStatus()));

            CoreBpmnRuntimeSource source;
            CoreBpmnNotificationEvent event;
            if (ProcessStatus.DRAFT_WITHDRAWED.equals(instanceStatus.getProcessStatus())) {
               CoreBpmnInstanceStatusDTO finalInstanceStatus = instanceStatus;
               source = sourceList.stream().filter((s) -> {
                  return s.getId().equals(finalInstanceStatus.getId());
               }).findAny().get();
               event = new CoreBpmnNotificationEvent(instanceStatus.getProcessStatus(), I18nHelper.getMessage(source.getTableName().toUpperCase() + ".SERVICE_NAME"), source.getDesc(), I18nHelper.getMessage("CORE.MODULE.SYS.T_CORE_BPMN_DIAGRAM.START_NODE"), LocalContextHelper.getLoginUserId(), source.getComment(), source.getCandidatorList());
               applicationContext.publishEvent(event);
            } else if (!ProcessStatus.DRAFT.equals(instanceStatus.getProcessStatus()) && !ProcessStatus.DRAFT_REJECTED.equals(instanceStatus.getProcessStatus())) {
               auditingInstanceStatusList.add(instanceStatus);
            } else {
               CoreBpmnInstanceStatusDTO finalInstanceStatus1 = instanceStatus;
               source = sourceList.stream().filter((s) -> {
                  return s.getId().equals(finalInstanceStatus1.getId());
               }).findAny().get();
               event = new CoreBpmnNotificationEvent(instanceStatus.getProcessStatus(), I18nHelper.getMessage(source.getTableName().toUpperCase() + ".SERVICE_NAME"), source.getDesc(), I18nHelper.getMessage("CORE.MODULE.SYS.T_CORE_BPMN_DIAGRAM.START_NODE"), LocalContextHelper.getLoginUserId(), source.getComment(), Arrays.asList(source.getRuntimeInitiator()));
               applicationContext.publishEvent(event);
            }
         }
      }
   }

   protected void insertComment(String targetId, Element taskElement, CommentStatus commentStatus, CoreBpmnRuntimeSource<T, ID> source, String route) {
      CoreBpmnCommentBean bpmnComment = new CoreBpmnCommentBean();
      bpmnComment.setId(ApplicationContextHelper.getNextIdentity());
      bpmnComment.setTargetId(targetId);
      bpmnComment.setProcId(source.getRuntimeProcId());
      if (!StringUtils.isEmpty(source.getRuntimeTaskName())) {
         bpmnComment.setTaskName(source.getRuntimeTaskName());
      } else {
         if (taskElement != null) {
            bpmnComment.setTaskId(XmlUtils.getAttributeValue(taskElement, "id"));
            bpmnComment.setTaskName(XmlUtils.getAttributeValue(taskElement, "name"));
            bpmnComment.setStatusCode(XmlUtils.getAttributeValue(taskElement, "statusCode"));
         }

         if (StringUtils.isEmpty(bpmnComment.getTaskName())) {
            if (CommentStatus.SUBMIT.equals(commentStatus)) {
               bpmnComment.setTaskName(I18nHelper.getMessage("SINO.BPMN.TASK.START"));
            } else if (CommentStatus.UNDO.equals(commentStatus)) {
               bpmnComment.setTaskName(I18nHelper.getMessage("SINO.BPMN.TASK.END"));
            }
         }
      }

      bpmnComment.setAttachmentId(source.getAttachmentId());
      bpmnComment.setUserId(LocalContextHelper.getLoginUserId());
      bpmnComment.setRoleId(LocalContextHelper.getLoginRoleId());
      bpmnComment.setStatus(commentStatus.name());
      bpmnComment.setComment(source.getComment());
      bpmnComment.setRoute(route);
      if (source.getNextCandidateRoleList().isEmpty() && source.getNextCandidatorList().isEmpty()) {
         if (!StringUtils.isBlank(source.getTransferCandidator())) {
            CoreUserService userService = (CoreUserService)ApplicationContextHelper.getBean(CoreUserService.class);
            CoreUserBean transferCandidator = (CoreUserBean)userService.selectById(source.getTransferCandidator());
            StringBuilder sb = (new StringBuilder("(")).append(I18nHelper.getMessage("SINO.BPMN.COMMENT.TRANSFER", LocalContextHelper.getLoginUserName(), transferCandidator.getUserName())).append(")");
            bpmnComment.setComment(bpmnComment.getComment() + sb.toString());
         }
      } else {
         StringBuilder sb;
         boolean start;
         List userList;
         Iterator var11;
         if (!source.getNextCandidateRoleList().isEmpty()) {
            sb = (new StringBuilder("(")).append(I18nHelper.getMessage("SINO.BPMN.COMMENT.SELECTED_CANDIDATE_ROLE")).append(":");
            start = true;
            CoreRoleService roleService = (CoreRoleService)ApplicationContextHelper.getBean(CoreRoleService.class);
            userList = roleService.selectListByIds(source.getNextCandidateRoleList());

            CoreRoleBean role;
            for(var11 = userList.iterator(); var11.hasNext(); sb.append(role.getRoleName())) {
               role = (CoreRoleBean)var11.next();
               if (start) {
                  start = false;
               } else {
                  sb.append(",");
               }
            }

            sb.append(")");
            bpmnComment.setComment(bpmnComment.getComment() + sb.toString());
         }

         if (!source.getNextCandidatorList().isEmpty()) {
            sb = (new StringBuilder("(")).append(I18nHelper.getMessage("SINO.BPMN.COMMENT.SELECTED_CANDIDATOR")).append(":");
            start = true;
            CoreUserService userService = (CoreUserService)ApplicationContextHelper.getBean(CoreUserService.class);
            userList = userService.selectListByIds(source.getNextCandidatorList());

            CoreUserBean user;
            for(var11 = userList.iterator(); var11.hasNext(); sb.append(user.getUserName())) {
               user = (CoreUserBean)var11.next();
               if (start) {
                  start = false;
               } else {
                  sb.append(",");
               }
            }

            sb.append(")");
            bpmnComment.setComment(bpmnComment.getComment() + sb.toString());
         }
      }

      BpmnRuntimeCacheProvider.getProcessData().addInsertComment(bpmnComment);
   }

   protected void prepareValidateSql(int serialNumber, String sql, ID id) {
      this.getValidateSqlMap(serialNumber + 1, sql, id).forEach((k, v) -> {
         BpmnRuntimeCacheProvider.cacheBpmnRuntimeValidateSql(k, v);
      });
   }

   protected Map<String, String> getValidateSqlMap(int serialNumber, String validatorSql, Object id) {
      if (StringUtils.isBlank(validatorSql)) {
         return CollectionUtils.emptyMap();
      } else {
         Map<String, String> sqlMap = new HashMap();
         String[] sqls = StringUtils.split(validatorSql, ";");
         String[] var6 = sqls;
         int var7 = sqls.length;

         for(int var8 = 0; var8 < var7; ++var8) {
            String sql = var6[var8];
            if (!StringUtils.isBlank(sql)) {
               String replacedSql = this.getReplacedValidatorSql(sql, id);
               sqlMap.put(sql, "SELECT '" + serialNumber + "' AS " + "SN_" + ", " + StringUtils.removeStartIgnoreCase(StringUtils.trim(replacedSql), "select"));
            }
         }

         return sqlMap;
      }
   }

   protected void validateBySql() {
      Map<String, List<String>> cacheMap = BpmnRuntimeCacheProvider.getBpmnRuntimeValidateSql();
      if (cacheMap != null && !cacheMap.isEmpty()) {
         cacheMap.values().forEach((sql) -> {
            List<Map<String, Object>> mapList = this.globalMapper.selectListBySqlList(sql);
            Map<Integer, String> invalidContainer = new HashMap();
            Iterator var4 = mapList.iterator();

            while(true) {
               Map map;
               ArrayList invalidColumnList;
               do {
                  if (!var4.hasNext()) {
                     if (!invalidContainer.isEmpty()) {
                        StringBuilder sbx = new StringBuilder();
                        invalidContainer.forEach((k, v) -> {
                           if (sbx.length() > 0) {
                              sbx.append(";");
                           }

                           sbx.append(v);
                        });
                        throw new CheckedException(sbx.toString());
                     }

                     return;
                  }

                  map = (Map)var4.next();
                  CollectionUtils.upperCaseKey(map);
                  invalidColumnList = new ArrayList();
                  ArrayList finalInvalidColumnList = invalidColumnList;
                  map.forEach((k, v) -> {
                     if ("0".equals(v)) {
                        finalInvalidColumnList.add(k);
                     }

                  });
               } while(invalidColumnList.isEmpty());

               int serialNumber = NumberUtils.parseInt(map.get("SN_"));
               String type = (String)map.get("TYPE_");
               String table = (String)map.get("TABLE_");
               String desc = (String)map.get("DESC_");

               String message;
               for(Iterator var11 = invalidColumnList.iterator(); var11.hasNext(); invalidContainer.put(serialNumber, message)) {
                  String column = (String)var11.next();
                  message = (String)invalidContainer.get(serialNumber);
                  if (StringUtils.isEmpty(message)) {
                     StringBuilder sb = new StringBuilder();
                     if (StringUtils.isEmpty(desc)) {
                        sb.append(I18nHelper.getMessage("SINO.VALIDATOR.INVALID_ROWNUMBER")).append(" ").append(serialNumber);
                     } else {
                        sb.append(I18nHelper.getMessage("SINO.VALIDATOR.CHECKED_DATA")).append(" ").append(desc);
                     }

                     sb.append(" :<br>").append(I18nHelper.getMessage(table + "." + column)).append(" ").append(I18nHelper.getMessage(type));
                     message = sb.toString();
                  } else {
                     message = message + "<br>" + I18nHelper.getMessage(table + "." + column) + " " + I18nHelper.getMessage(type);
                  }
               }
            }
         });
      }

   }

   protected void prepareCallback(CallbackCategory callbackCateogry, Element element, ID id) {
      String callback = null;
      if (CallbackCategory.pass.equals(callbackCateogry)) {
         callback = BpmnDiagramHelper.getPassCallback(element);
      } else if (CallbackCategory.reject.equals(callbackCateogry)) {
         callback = BpmnDiagramHelper.getRejectCallback(element);
      } else if (CallbackCategory.end.equals(callbackCateogry)) {
         callback = BpmnDiagramHelper.getEndCallback(element);
      }

      if (!StringUtils.isBlank(callback)) {
         BpmnRuntimeCacheProvider.cacheBpmnRuntimeCallback(callback, id);
      }

   }

   protected void prepareEndCallback(CallbackCategory callbackCateogry, Element element, ID id) {
      String endCallback = null;
      if (CallbackCategory.pass.equals(callbackCateogry)) {
         endCallback = BpmnDiagramHelper.getPassCallback(element);
      } else if (CallbackCategory.reject.equals(callbackCateogry)) {
         endCallback = BpmnDiagramHelper.getRejectCallback(element);
      } else if (CallbackCategory.end.equals(callbackCateogry)) {
         endCallback = BpmnDiagramHelper.getEndCallback(element);
      }

      if (!StringUtils.isBlank(endCallback)) {
         BpmnRuntimeCacheProvider.cacheBpmnRuntimeEndCallback(endCallback, id);
      }

   }

   protected void insertNextInstanceTask(Document diagram, CoreBpmnInstanceBean instance, List<Element> nextTaskElementList, CoreBpmnRuntimeSource<T, ID> source) {
      CoreBpmnInstanceTaskHisBean taskHis = BpmnRuntimeCacheProvider.getBpmnRuntimeTaskHisMap() != null ? (CoreBpmnInstanceTaskHisBean)BpmnRuntimeCacheProvider.getBpmnRuntimeTaskHisMap().get(source.getId()) : null;
      Iterator var6 = nextTaskElementList.iterator();

      while(var6.hasNext()) {
         Element nextTaskElement = (Element)var6.next();
         Long taskId = ApplicationContextHelper.getNextIdentity();
         ArrayList roleList;
         String createdByOrgId;
         if (taskHis == null) {
            List<String> userList = new ArrayList();
            roleList = new ArrayList();
            if (!StringUtils.isEmpty(BpmnDiagramHelper.getUserTaskInitiatorValue(nextTaskElement)) && !StringUtils.isEmpty(instance.getInitiator())) {
               userList.add(instance.getInitiator());
            }

            boolean nextCandidatorSelected = !source.getNextCandidatorList().isEmpty() || !source.getNextCandidateRoleList().isEmpty();
            if (nextCandidatorSelected) {
               userList.addAll(source.getNextCandidatorList());
               roleList.addAll(source.getNextCandidateRoleList());
            } else {
               createdByOrgId = BpmnDiagramHelper.getUserTaskDynamicValue(nextTaskElement);
               if (!StringUtils.isEmpty(createdByOrgId)) {
                  String[] dynamicValues = StringUtils.split(createdByOrgId, ",");
                  String[] var14 = dynamicValues;
                  int var15 = dynamicValues.length;

                  for(int var16 = 0; var16 < var15; ++var16) {
                     String value = var14[var16];
                     String role;
                     if (StringUtils.startsWith(value, "u:")) {
                        role = (String)source.getVars().get(StringUtils.removeStart(value, "u:"));
                        if (role == null) {
                           throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.DYNAMIC_USER_NOT_EXISTS");
                        }

                        userList.add(role);
                     } else if (StringUtils.startsWith(value, "r:")) {
                        role = ObjectUtils.toString(source.getVars().get(StringUtils.removeStart(value, "r:")));
                        if (StringUtils.isEmpty(role)) {
                           throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.DYNAMIC_ROLE_NOT_EXISTS");
                        }

                        roleList.add(NumberUtils.parseLong(role));
                     }
                  }
               }

               Map<String, String> userTaskCandidatorMap = BpmnDiagramHelper.getUserTaskCandidatorValue(nextTaskElement);
               if (!userTaskCandidatorMap.isEmpty()) {
                  userList.addAll(userTaskCandidatorMap.keySet());
               }

               Map<Long, String> userTaskRoleMap = BpmnDiagramHelper.getUserTaskRoleValue(nextTaskElement);
               if (!userTaskRoleMap.isEmpty()) {
                  roleList.addAll(userTaskRoleMap.keySet());
               }
            }

            List taskRoleList;
            if (BpmnDiagramHelper.isFilterPreviousCandidatorsTask(nextTaskElement)) {
               taskRoleList = BpmnDiagramHelper.getIncomingUserTaskElementList(diagram, nextTaskElement);
               if (!taskRoleList.isEmpty()) {
                  Iterator var28 = taskRoleList.iterator();

                  while(var28.hasNext()) {
                     Element userTaskElement = (Element)var28.next();
                     List<String> auditedByList = this.commentService.selectUserTaskAuditedByList(source.getTargetId(), BpmnDiagramHelper.getUserTaskStatusCode(userTaskElement));
                     if (auditedByList != null) {
                        userList.removeIf((u) -> {
                           return auditedByList.contains(u);
                        });
                     }
                  }
               }

               userList.remove(LocalContextHelper.getLoginUserId());
            }

            if (userList.isEmpty() && roleList.isEmpty()) {
               throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.CANDIDATOR_AND_ROLE_NOT_EXISTS");
            }

            if (!userList.isEmpty()) {
               if (CandidatorFilterStrategy.sameDept.equals(BpmnDiagramHelper.getCandidatorFilterStrategy(nextTaskElement))) {
                  createdByOrgId = (String)CollectionUtils.getValueIgnorecase(source.getVars(), "createdByOrgId");
                  if (!StringUtils.isBlank(createdByOrgId)) {
                     CoreUserService userService = (CoreUserService)ApplicationContextHelper.getBean(CoreUserService.class);
                     List<CoreUserBean> selectedUserList = userService.selectListByIds(userList);
                     String finalCreatedByOrgId = createdByOrgId;
                     userList.removeIf((u) -> {
                        return selectedUserList.stream().filter((s) -> {
                           return s.getId().equals(u);
                        }).noneMatch((s) -> {
                           return finalCreatedByOrgId.equals(s.getOrgId());
                        });
                     });
                  } else {
                     logger.warn("No createdByOrgId column in (" + source.getTableName() + "), ignore candidator filter strategy.");
                  }
               }

               taskRoleList = (List)userList.stream().distinct().map((u) -> {
                  CoreBpmnInstanceTaskUserBean taskUser = new CoreBpmnInstanceTaskUserBean();
                  taskUser.setId(ApplicationContextHelper.getNextIdentity());
                  taskUser.setInstId(instance.getId());
                  taskUser.setInstTaskId(taskId);
                  taskUser.setUserId(u);
                  return taskUser;
               }).collect(Collectors.toList());
               BpmnRuntimeCacheProvider.getProcessData().addInsertInstanceTaskUser(taskRoleList);
            }

            if (!roleList.isEmpty()) {
               taskRoleList = (List)roleList.stream().distinct().map((r) -> {
                  CoreBpmnInstanceTaskRoleBean taskRole = new CoreBpmnInstanceTaskRoleBean();
                  taskRole.setId(ApplicationContextHelper.getNextIdentity());
                  taskRole.setInstId(instance.getId());
                  taskRole.setInstTaskId(taskId);
                  taskRole.setRoleId((Long) r);
                  return taskRole;
               }).collect(Collectors.toList());
               BpmnRuntimeCacheProvider.getProcessData().addInsertInstanceTaskRole(taskRoleList);
            }

            if (userList.isEmpty() && roleList.isEmpty()) {
               throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.CANDIDATOR_AND_ROLE_NOT_EXISTS");
            }
         } else {
            CoreBpmnInstanceTaskHisCandidatorDTO taskHisCandidator = (CoreBpmnInstanceTaskHisCandidatorDTO)JSON.parseObject(taskHis.getCandidatorJson(), CoreBpmnInstanceTaskHisCandidatorDTO.class);
            Iterator var11;
            if (!CollectionUtils.isEmpty((Collection)taskHisCandidator.getUserIdList())) {
               roleList = new ArrayList();
               var11 = taskHisCandidator.getUserIdList().iterator();

               while(var11.hasNext()) {
                  createdByOrgId = (String)var11.next();
                  CoreBpmnInstanceTaskUserBean taskUser = new CoreBpmnInstanceTaskUserBean();
                  taskUser.setId(ApplicationContextHelper.getNextIdentity());
                  taskUser.setInstId(instance.getId());
                  taskUser.setInstTaskId(taskId);
                  taskUser.setUserId(createdByOrgId);
                  roleList.add(taskUser);
               }

               BpmnRuntimeCacheProvider.getProcessData().addInsertInstanceTaskUser(roleList);
            }

            if (!CollectionUtils.isEmpty((Collection)taskHisCandidator.getRoleIdList())) {
               roleList = new ArrayList();
               var11 = taskHisCandidator.getRoleIdList().iterator();

               while(var11.hasNext()) {
                  Long roleId = (Long)var11.next();
                  CoreBpmnInstanceTaskRoleBean taskRole = new CoreBpmnInstanceTaskRoleBean();
                  taskRole.setId(ApplicationContextHelper.getNextIdentity());
                  taskRole.setInstId(instance.getId());
                  taskRole.setInstTaskId(taskId);
                  taskRole.setRoleId(roleId);
                  roleList.add(taskRole);
               }

               BpmnRuntimeCacheProvider.getProcessData().addInsertInstanceTaskRole(roleList);
            }
         }

         CoreBpmnInstanceTaskBean task = new CoreBpmnInstanceTaskBean();
         task.setId(taskId);
         task.setInstId(instance.getId());
         task.setTaskId(XmlUtils.getAttributeValue(nextTaskElement, "id"));
         task.setTaskName(XmlUtils.getAttributeValue(nextTaskElement, "name"));
         task.setStatusCode(XmlUtils.getAttributeValue(nextTaskElement, "statusCode"));
         if (CounterSignStrategy.none.equals(BpmnDiagramHelper.getCounterSignStrategy(nextTaskElement))) {
            task.setCounterSign("0");
         } else {
            task.setCounterSign("1");
         }

         task.setStatus(TaskStatus.CREATED.name());
         BpmnRuntimeCacheProvider.getProcessData().addInsertInstanceTask(task);
      }

   }

   protected CoreBpmnTargetBean getBpmnTarget(CoreBpmnRuntimeSource<T, ID> source, ProcessStatus status) {
      CoreBpmnTargetBean target = new CoreBpmnTargetBean();
      target.setId(source.getTargetId());
      target.setTargetId(ObjectUtils.toString(source.getId()));
      target.setTargetTable(source.getTableName());
      target.setTargetDesc(source.getDesc());
      target.setProcessStatus(status.name());
      return target;
   }

   protected String getRoute(String sourceName, String targetName) {
      return sourceName + " -> " + (String)Optional.ofNullable(targetName).orElse(I18nHelper.getMessage("SINO.BPMN.TASK.GATEWAY"));
   }

   protected String getElementName(Document diagram, String statusCodes) {
      if (StringUtils.isBlank(statusCodes)) {
         return null;
      } else {
         String[] codes = StringUtils.split(statusCodes, ",");
         StringBuilder sb = new StringBuilder();
         String[] var5 = codes;
         int var6 = codes.length;

         for(int var7 = 0; var7 < var6; ++var7) {
            String code = var5[var7];
            if (sb.length() > 0) {
               sb.append(",");
            }

            sb.append(XmlUtils.getAttributeValue(BpmnDiagramHelper.getProcessElementByStatusCode(diagram, code), "name"));
         }

         return sb.toString();
      }
   }

   protected void updateProcessStatus(String table, List<CoreBpmnInstanceStatusDTO<ID>> statusList) {
      GenericService<T, ID> service = SinoBeanContext.getServiceByTable(table);
      BpmnRuntimeData.BpmnProcessData data = BpmnRuntimeCacheProvider.getProcessData();
      List<T> itemList = new ArrayList();

      Auditable item;
      for(Iterator var6 = statusList.iterator(); var6.hasNext(); itemList.add((T) item)) {
         CoreBpmnInstanceStatusDTO<ID> status = (CoreBpmnInstanceStatusDTO)var6.next();
         item = (Auditable)ClassUtils.newInstance(service.getDao().getType());
         item.setId(status.getId());
         if (data.isAsync()) {
            item.setProcessStatus("waiting$" + status.getProcessStatus().name().toLowerCase());
         } else {
            item.setProcessStatus(status.getProcessStatus().name().toLowerCase());
         }
      }

      if (data.isAsync()) {
         List<T> asyncItemList = new ArrayList();

         for (CoreBpmnInstanceStatusDTO<ID> idCoreBpmnInstanceStatusDTO : statusList) {
            CoreBpmnInstanceStatusDTO<ID> status = (CoreBpmnInstanceStatusDTO) idCoreBpmnInstanceStatusDTO;
            item = ClassUtils.newInstance(service.getDao().getType());
            item.setId(status.getId());
            item.setProcessStatus(status.getProcessStatus().name().toLowerCase());
            asyncItemList.add((T) item);
         }

         data.setUpdateProcessStatusRunnable(() -> {
            service.getDao().update(asyncItemList, "PROCESSSTATUS");
         });
      }

      service.getDao().update(itemList, "PROCESSSTATUS");
   }

   protected void doCallback(Map<String, List<ID>> callbackMap) {
      callbackMap.forEach((k, v) -> {
         int lastIndexOf = k.lastIndexOf(".");
         String className = k.substring(0, lastIndexOf);
         String methodName = k.substring(lastIndexOf + 1);
         Class<?> clazz = ClassUtils.getClass(className);
         Object bean = ApplicationContextHelper.getBean(clazz);
         Method method = ReflectionUtils.findMethod(clazz, methodName, List.class);
         ReflectionUtils.invokeMethod(method, bean, v);
      });
   }

   protected void processData() {
      BpmnRuntimeData.BpmnProcessData data = BpmnRuntimeCacheProvider.getProcessData();
      CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
      int limit = (Integer)ApplicationContextHelper.getEnvironment().getProperty("sino.bpmn.process-data-limit", Integer.class, 50);
      if (limit <= 0 || data.getInsertInstanceList().size() <= limit && data.getInsertInstanceTaskList().size() <= limit) {
         runtimeService.processData(data);
      } else {
         data.setAsync(true);
         LoginUser loginUser = (LoginUser)ObjectUtils.clone(LocalContextHelper.getLoginPrincipal());
         ExecutorHelper.submitAfterCommitTransaction(() -> {
            try {
               if (loginUser != null) {
                  LocalContextHelper.setUserLogin(loginUser);
               }

               runtimeService.processData(data);
            } finally {
               if (loginUser != null) {
                  LocalContextHelper.removeUserLogin();
               }

            }

         });
      }

   }

   private List<CoreBpmnInstanceBean> getBpmnRuntimeInstanceListFromDatabase(List<CoreBpmnRuntimeSource<T, ID>> sourceList) {
      List<String> targetIdList = (List)sourceList.stream().map(CoreBpmnRuntimeSource::getTargetId).collect(Collectors.toList());
      return this.instanceService.getDao().selectListByOneColumnValues(targetIdList, "TARGETID");
   }

   private String getReplacedValidatorSql(String sql, Object id) {
      sql = StringUtils.replace(sql, "${id}", ObjectUtils.toString(id));
      sql = StringUtils.replace(sql, "${loginUser.id}", LocalContextHelper.getLoginUserId());
      sql = StringUtils.replace(sql, "${loginUser.userName}", LocalContextHelper.getLoginUserName());
      sql = StringUtils.replace(sql, "${loginUser.orgId}", LocalContextHelper.getLoginUser().getOrgId());
      return StringUtils.replace(sql, "${loginUser.orgName}", LocalContextHelper.getLoginUser().getOrgName());
   }

   protected static enum CallbackCategory {
      pass,
      reject,
      end;
   }
}
