/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.module.sys.bpmn.engine.cache;

import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.context.LocalContextHelper;
import net.sinodawn.framework.context.SinoBeanContext;
import net.sinodawn.framework.data.Pair;
import net.sinodawn.framework.support.auditable.service.GenericAuditableService;
import net.sinodawn.framework.support.domain.Auditable;
import net.sinodawn.framework.utils.CollectionUtils;
import net.sinodawn.framework.utils.StringUtils;
import net.sinodawn.framework.utils.XmlUtils;
import net.sinodawn.module.mdm.user.bean.CoreUserBean;
import net.sinodawn.module.mdm.user.service.CoreUserService;
import net.sinodawn.module.sys.bpmn.CoreBpmnHelper;
import net.sinodawn.module.sys.bpmn.bean.*;
import net.sinodawn.module.sys.bpmn.diagram.BpmnDiagramHelper;
import net.sinodawn.module.sys.bpmn.diagram.ProcessElementType;
import net.sinodawn.module.sys.bpmn.diagram.ProcessStatus;
import net.sinodawn.module.sys.bpmn.diagram.attribute.CandidatorFilterStrategy;
import net.sinodawn.module.sys.bpmn.diagram.attribute.CounterSignStrategy;
import net.sinodawn.module.sys.bpmn.engine.CoreBpmnRuntimeSource;
import net.sinodawn.module.sys.bpmn.exception.BpmnException;
import net.sinodawn.module.sys.bpmn.service.*;
import org.dom4j.Document;
import org.dom4j.Element;

import java.io.Serializable;
import java.util.*;
import java.util.Map.Entry;
import java.util.stream.Collectors;

public class BpmnRuntimeCacheProvider {
   private static final ThreadLocal<List<BpmnRuntimeData>> BPMN_CACHE = new ThreadLocal();

   public static <T extends Auditable<ID>, ID extends Serializable> BpmnRuntimeData<T, ID> getRuntimeData() {
      List<BpmnRuntimeData> dataList = (List)BPMN_CACHE.get();
      if (dataList == null) {
         dataList = new ArrayList();
         BPMN_CACHE.set(dataList);
         BpmnRuntimeData<T, ID> runtimeData = new BpmnRuntimeData();
         ((List)dataList).add(runtimeData);
      }

      return (BpmnRuntimeData)((List)dataList).get(((List)dataList).size() - 1);
   }

   public static BpmnRuntimeData.BpmnProcessData getProcessData() {
      return getRuntimeData().getProcessData();
   }

   public static <T extends Auditable<ID>, ID extends Serializable> void init(List<CoreBpmnRuntimeSource<T, ID>> sourceList) {
      List<BpmnRuntimeData> dataList = (List)BPMN_CACHE.get();
      if (dataList == null) {
         dataList = new ArrayList();
         BPMN_CACHE.set(dataList);
      }

      BpmnRuntimeData<T, ID> runtimeData = new BpmnRuntimeData();
      ((List)dataList).add(runtimeData);
      runtimeData.setSourceList(sourceList);
   }

   public static void remove() {
      List<BpmnRuntimeData> dataList = (List)BPMN_CACHE.get();
      if (dataList != null && dataList.size() > 1) {
         dataList.remove(dataList.size() - 1);
      } else {
         BPMN_CACHE.remove();
      }

   }

   public static <T extends Auditable<ID>, ID extends Serializable> List<CoreBpmnRuntimeSource<T, ID>> getBpmnRuntimeSourceList() {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      return runtimeData.getSourceList();
   }

   public static <T extends Auditable<ID>, ID extends Serializable> List<CoreBpmnInstanceBean> getBpmnRuntimeInstanceList() {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      List<CoreBpmnInstanceBean> instanceList = runtimeData.getInstanceList();
      if (instanceList == null) {
         CoreBpmnInstanceService instanceService = (CoreBpmnInstanceService)ApplicationContextHelper.getBean(CoreBpmnInstanceService.class);
         List<String> targetIdList = (List)getBpmnRuntimeSourceList().stream().map(CoreBpmnRuntimeSource::getTargetId).collect(Collectors.toList());
         instanceList = instanceService.getDao().selectListByOneColumnValues(targetIdList, "TARGETID");
         runtimeData.setInstanceList(instanceList);
      }

      return instanceList;
   }

   public static <T extends Auditable<ID>, ID extends Serializable> List<CoreBpmnInstanceTaskBean> getBpmnRuntimeInstanceTaskList() {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      List<CoreBpmnInstanceTaskBean> instanceTaskList = runtimeData.getInstanceTaskList();
      if (instanceTaskList == null) {
         instanceTaskList = new ArrayList();
         List<CoreBpmnRuntimeSource<T, ID>> sourceList = getBpmnRuntimeSourceList();
         List<CoreBpmnInstanceBean> instanceList = getBpmnRuntimeInstanceList();
         CoreBpmnInstanceTaskService instanceTaskService = (CoreBpmnInstanceTaskService)ApplicationContextHelper.getBean(CoreBpmnInstanceTaskService.class);
         List<Long> permissionRequiredBpmnInstanceIdList = (List)instanceList.stream().filter((i) -> {
            return sourceList.stream().anyMatch((s) -> {
               return s.getTargetId().equals(i.getTargetId()) && s.isRuntimeCheckPermission();
            });
         }).map((i) -> {
            return i.getId();
         }).collect(Collectors.toList());
         List<Long> permissionIgnoredBpmnInstanceIdList = (List)instanceList.stream().filter((i) -> {
            return sourceList.stream().anyMatch((s) -> {
               return s.getTargetId().equals(i.getTargetId()) && !s.isRuntimeCheckPermission();
            });
         }).map((i) -> {
            return i.getId();
         }).collect(Collectors.toList());
         if (LocalContextHelper.isUserLogin()) {
            if (!permissionRequiredBpmnInstanceIdList.isEmpty()) {
               ((List)instanceTaskList).addAll(instanceTaskService.selectAuditableInstanceTaskList(LocalContextHelper.getLoginUserId(), permissionRequiredBpmnInstanceIdList));
            }

            if (!permissionIgnoredBpmnInstanceIdList.isEmpty()) {
               ((List)instanceTaskList).addAll(instanceTaskService.selectAuditableInstanceTaskList(null, permissionIgnoredBpmnInstanceIdList));
            }
         } else if (!permissionIgnoredBpmnInstanceIdList.isEmpty()) {
            ((List)instanceTaskList).addAll(instanceTaskService.selectAuditableInstanceTaskList(null, permissionIgnoredBpmnInstanceIdList));
         }

         runtimeData.setInstanceTaskList((List)instanceTaskList);
      }

      return (List)instanceTaskList;
   }

   public static <T extends Auditable<ID>, ID extends Serializable> List<CoreBpmnInstanceTaskBean> getBpmnRuntimeUnlimitedInstanceTaskList() {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      List<CoreBpmnInstanceTaskBean> unlimitedInstanceTaskList = runtimeData.getUnlimitedInstanceTaskList();
      if (unlimitedInstanceTaskList == null) {
         List<CoreBpmnInstanceBean> instanceList = getBpmnRuntimeInstanceList();
         CoreBpmnInstanceTaskService instanceTaskService = (CoreBpmnInstanceTaskService)ApplicationContextHelper.getBean(CoreBpmnInstanceTaskService.class);
         List<Long> unlimitedBpmnInstanceIdList = (List)instanceList.stream().map((i) -> {
            return i.getId();
         }).collect(Collectors.toList());
         unlimitedInstanceTaskList = instanceTaskService.selectAuditableInstanceTaskList(null, unlimitedBpmnInstanceIdList);
         runtimeData.setUnlimitedInstanceTaskList(unlimitedInstanceTaskList);
      }

      return unlimitedInstanceTaskList;
   }

   public static <T extends Auditable<ID>, ID extends Serializable> List<String> getBpmnPreviousTaskAuditedByList(CoreBpmnRuntimeSource<T, ID> source) {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      List<String> previousTaskAuditedByList = runtimeData.getPreviousTaskAuditedByList();
      if (previousTaskAuditedByList == null) {
         Element taskElement = getBpmnRuntimeTaskElement(source);
         previousTaskAuditedByList = new ArrayList();
         if (BpmnDiagramHelper.isFilterPreviousCandidatorsTask(taskElement)) {
            List<Element> userTaskElementList = BpmnDiagramHelper.getIncomingUserTaskElementList(getBpmnRuntimeDocument(source), taskElement);
            if (!userTaskElementList.isEmpty()) {
               Iterator var5 = userTaskElementList.iterator();

               while(var5.hasNext()) {
                  Element userTaskElement = (Element)var5.next();
                  List<String> auditedByList = ((CoreBpmnCommentService)ApplicationContextHelper.getBean(CoreBpmnCommentService.class)).selectUserTaskAuditedByList(source.getTargetId(), BpmnDiagramHelper.getUserTaskStatusCode(userTaskElement));
                  if (auditedByList != null && auditedByList.contains(LocalContextHelper.getLoginUserId())) {
                     throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.FILTER_PREVIOUS_CANDIDATORS");
                  }

                  ((List)previousTaskAuditedByList).addAll(auditedByList);
               }
            }
         }

         runtimeData.setPreviousTaskAuditedByList((List)previousTaskAuditedByList);
      }

      return (List)previousTaskAuditedByList;
   }

   public static <T extends Auditable<ID>, ID extends Serializable> List<CoreBpmnInstanceCandidatorDTO> getBpmnRuntimeInstanceCandidatorList() {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      List<CoreBpmnInstanceCandidatorDTO> instanceCandidatorList = runtimeData.getInstanceCandidatorList();
      if (instanceCandidatorList == null) {
         List<CoreBpmnInstanceBean> instanceList = getBpmnRuntimeInstanceList();
         List<Long> idList = (List)instanceList.stream().map((i) -> {
            return i.getId();
         }).collect(Collectors.toList());
         CoreBpmnInstanceService instanceService = (CoreBpmnInstanceService)ApplicationContextHelper.getBean(CoreBpmnInstanceService.class);
         instanceCandidatorList = instanceService.selectInstanceCandidatorList(idList);
         runtimeData.setInstanceCandidatorList(instanceCandidatorList);
      }

      return instanceCandidatorList;
   }

   public static <T extends Auditable<ID>, ID extends Serializable> CoreBpmnInstanceBean getBpmnRuntimeInstance(CoreBpmnRuntimeSource<T, ID> source) {
      List<CoreBpmnInstanceBean> instanceList = getBpmnRuntimeInstanceList();
      return (CoreBpmnInstanceBean)instanceList.stream().filter((i) -> {
         return i.getTargetId().equals(source.getTargetId());
      }).findAny().orElse(null);
   }

   public static <T extends Auditable<ID>, ID extends Serializable> CoreBpmnInstanceTaskBean getBpmnRuntimeInstanceTask(CoreBpmnRuntimeSource<T, ID> source) {
      CoreBpmnInstanceBean instance = getBpmnRuntimeInstance(source);
      List<CoreBpmnInstanceTaskBean> runtimeTaskList = getBpmnRuntimeInstanceTaskList();
      List<CoreBpmnInstanceTaskBean> instanceTaskList = (List)runtimeTaskList.stream().filter((t) -> {
         boolean flag = t.getInstId().equals(instance.getId());
         if (flag && !StringUtils.isEmpty(source.getCurrentStatusCode())) {
            flag = source.getCurrentStatusCode().equals(t.getStatusCode());
         }

         return flag;
      }).collect(Collectors.toList());
      if (instanceTaskList.isEmpty()) {
         throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.NOT_AUTHORIZED");
      } else if (instanceTaskList.size() > 1) {
         throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.MULTIPLE_TASK_ASSIGNED");
      } else {
         return (CoreBpmnInstanceTaskBean)instanceTaskList.get(0);
      }
   }

   public static <T extends Auditable<ID>, ID extends Serializable> List<CoreBpmnInstanceTaskBean> getBpmnRuntimeInstanceTaskList(CoreBpmnRuntimeSource<T, ID> source) {
      CoreBpmnInstanceBean instance = getBpmnRuntimeInstance(source);
      List<CoreBpmnInstanceTaskBean> runtimeTaskList = getBpmnRuntimeInstanceTaskList();
      return (List)runtimeTaskList.stream().filter((t) -> {
         boolean flag = t.getInstId().equals(instance.getId());
         if (flag && !StringUtils.isEmpty(source.getCurrentStatusCode())) {
            flag = source.getCurrentStatusCode().equals(t.getStatusCode());
         }

         return flag;
      }).collect(Collectors.toList());
   }

   public static <T extends Auditable<ID>, ID extends Serializable> CoreBpmnProcBean getBpmnProc(CoreBpmnRuntimeSource<T, ID> source) {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      CoreBpmnProcBean proc = (CoreBpmnProcBean)runtimeData.getProcMap().get(source.getId());
      if (proc == null) {
         CoreBpmnProcService procService = (CoreBpmnProcService)ApplicationContextHelper.getBean(CoreBpmnProcService.class);
         if (source.getProcId() != null) {
            proc = (CoreBpmnProcBean)procService.selectById(source.getProcId());
         } else {
            CoreBpmnInstanceBean instance = getBpmnRuntimeInstance(source);
            proc = (CoreBpmnProcBean)procService.selectByIdIfPresent(instance.getProcId());
         }

         runtimeData.getProcMap().put(source.getId(), proc);
      }

      return proc;
   }

   public static <T extends Auditable<ID>, ID extends Serializable> Document getBpmnRuntimeDocument(CoreBpmnRuntimeSource<T, ID> source) {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      Document document = (Document)runtimeData.getDocumentMap().get(source.getId());
      if (document == null) {
         CoreBpmnProcBean proc = getBpmnProc(source);
         Objects.requireNonNull(proc, "CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.PROCESS_NOT_EXISTS");
         CoreBpmnDiagramService diagramService = (CoreBpmnDiagramService)ApplicationContextHelper.getBean(CoreBpmnDiagramService.class);
         document = diagramService.getDocument(proc.getDiagramId());
         runtimeData.getDocumentMap().put(source.getId(), document);
      }

      return document;
   }

   public static <T extends Auditable<ID>, ID extends Serializable> Element getBpmnRuntimeTaskElement(CoreBpmnRuntimeSource<T, ID> source) {
      Document diagram = getBpmnRuntimeDocument(source);
      if (isStartTask(source)) {
         return BpmnDiagramHelper.getProcessElement(diagram, ProcessElementType.START_EVENT);
      } else {
         CoreBpmnInstanceTaskBean instanceTask = getBpmnRuntimeInstanceTask(source);
         return BpmnDiagramHelper.getProcessElement(diagram, instanceTask.getTaskId());
      }
   }

   public static <T extends Auditable<ID>, ID extends Serializable> void cacheBpmnRuntimeCallback(String callback, ID id) {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      List<ID> idList = (List)runtimeData.getCallbackMap().get(callback);
      if (idList == null) {
         idList = new ArrayList();
         runtimeData.getCallbackMap().put(callback, idList);
      }

      if (!((List)idList).contains(id)) {
         ((List)idList).add(id);
      }

   }

   public static <T extends Auditable<ID>, ID extends Serializable> Map<String, List<ID>> getBpmnRuntimeCallback() {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      return runtimeData.getCallbackMap();
   }

   public static <T extends Auditable<ID>, ID extends Serializable> void removeBpmnRuntimeCallback(ID id) {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      Iterator iterator = runtimeData.getCallbackMap().entrySet().iterator();

      while(iterator.hasNext()) {
         Entry<String, List<ID>> entry = (Entry)iterator.next();
         if (((List)entry.getValue()).contains(id)) {
            if (((List)entry.getValue()).size() == 1) {
               iterator.remove();
            } else {
               ((List)entry.getValue()).remove(id);
            }
         }
      }

   }

   public static <T extends Auditable<ID>, ID extends Serializable> void cacheBpmnRuntimeEndCallback(String endCallback, ID id) {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      List<ID> idList = (List)runtimeData.getEndCallbackMap().get(endCallback);
      if (idList == null) {
         idList = new ArrayList();
         runtimeData.getEndCallbackMap().put(endCallback, idList);
      }

      if (!((List)idList).contains(id)) {
         ((List)idList).add(id);
      }

   }

   public static <T extends Auditable<ID>, ID extends Serializable> void cacheBpmnRuntimeTaskHisMap(Map<ID, CoreBpmnInstanceTaskHisBean> taskHisMap) {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      if (!taskHisMap.isEmpty()) {
         runtimeData.setTaskHisMap(taskHisMap);
      }

   }

   public static <T extends Auditable<ID>, ID extends Serializable> Map<ID, CoreBpmnInstanceTaskHisBean> getBpmnRuntimeTaskHisMap() {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      return runtimeData.getTaskHisMap();
   }

   public static <T extends Auditable<ID>, ID extends Serializable> Map<String, List<ID>> getBpmnRuntimeEndCallback() {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      return runtimeData.getEndCallbackMap();
   }

   public static <T extends Auditable<ID>, ID extends Serializable> void cacheBpmnRuntimeValidateSql(String rawSql, String validateSql) {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      List<String> validateSqlList = (List)runtimeData.getValidateSqlMap().get(rawSql);
      if (validateSqlList == null) {
         validateSqlList = new ArrayList();
         runtimeData.getValidateSqlMap().put(rawSql, validateSqlList);
      }

      if (!((List)validateSqlList).contains(validateSql)) {
         ((List)validateSqlList).add(validateSql);
      }

   }

   public static <T extends Auditable<ID>, ID extends Serializable> Map<String, List<String>> getBpmnRuntimeValidateSql() {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      return runtimeData.getValidateSqlMap();
   }

   public static <T extends Auditable<ID>, ID extends Serializable> List<CoreBpmnInstanceNextTaskElementDTO<ID>> getNextTaskList() {
      BpmnRuntimeData<T, ID> runtimeData = getRuntimeData();
      List<CoreBpmnInstanceNextTaskElementDTO<ID>> nextTaskElementList = runtimeData.getNextTaskElementList();
      if (nextTaskElementList == null) {
         nextTaskElementList = new ArrayList();
         List<CoreBpmnRuntimeSource<T, ID>> sourceList = getBpmnRuntimeSourceList();
         List<CoreBpmnRuntimeSource<T, ID>> draftSourceList = (List)sourceList.stream().filter((s) -> {
            return StringUtils.startsWithIgnoreCase(s.getOldItem().getProcessStatus(), ProcessStatus.DRAFT.name());
         }).collect(Collectors.toList());
         List<CoreBpmnRuntimeSource<T, ID>> approveSourceList = (List)sourceList.stream().filter((s) -> {
            return StringUtils.startsWithIgnoreCase(s.getOldItem().getProcessStatus(), ProcessStatus.APPROVE.name());
         }).collect(Collectors.toList());
         if (!draftSourceList.isEmpty()) {
            CoreBpmnProcService procService = (CoreBpmnProcService)ApplicationContextHelper.getBean(CoreBpmnProcService.class);
            CoreBpmnDiagramService diagramService = (CoreBpmnDiagramService)ApplicationContextHelper.getBean(CoreBpmnDiagramService.class);
            GenericAuditableService<T, ID> service = (GenericAuditableService)SinoBeanContext.getServiceByTable(((CoreBpmnRuntimeSource)draftSourceList.get(0)).getTableName());
            List<Pair<ID, Long>> pairList = service.getBpmnProcIdList((List)draftSourceList.stream().map((s) -> {
               return s.getOldItem();
            }).collect(Collectors.toList()));
            Iterator var9 = draftSourceList.iterator();

            while(var9.hasNext()) {
               CoreBpmnRuntimeSource<T, ID> draftSource = (CoreBpmnRuntimeSource)var9.next();
               Long procId = (Long)((Pair)pairList.stream().filter((p) -> {
                  return ((Serializable)p.getFirst()).equals(draftSource.getOldItem().getId());
               }).findFirst().get()).getSecond();
               CoreBpmnProcBean proc = (CoreBpmnProcBean)procService.selectById(procId);
               if (proc == null) {
                  throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.PROCESS_NOT_EXISTS");
               }

               Document diagram = diagramService.getDocument(proc.getDiagramId());
               Element startElement = BpmnDiagramHelper.getProcessElement(diagram, ProcessElementType.START_EVENT);
               CoreBpmnInstanceNextTaskElementDTO<ID> nextTaskElement = new CoreBpmnInstanceNextTaskElementDTO();
               nextTaskElement.setId(draftSource.getId());
               nextTaskElement.setSourceElement(startElement);
               nextTaskElement.setNextTaskElementList(getNextTaskElementList(draftSource, startElement));
               ((List)nextTaskElementList).add(nextTaskElement);
            }
         }

         if (!approveSourceList.isEmpty()) {
            Iterator var16 = approveSourceList.iterator();

            while(var16.hasNext()) {
               CoreBpmnRuntimeSource<T, ID> approveSource = (CoreBpmnRuntimeSource)var16.next();
               Element taskElement = getBpmnRuntimeTaskElement(approveSource);
               CoreBpmnInstanceNextTaskElementDTO<ID> nextTaskElement = new CoreBpmnInstanceNextTaskElementDTO();
               nextTaskElement.setId(approveSource.getId());
               nextTaskElement.setSourceElement(taskElement);
               nextTaskElement.setNextTaskElementList(getNextTaskElementList(approveSource, taskElement));
               ((List)nextTaskElementList).add(nextTaskElement);
            }
         }

         runtimeData.setNextTaskElementList((List)nextTaskElementList);
      }

      return (List)nextTaskElementList;
   }

   public static <T extends Auditable<ID>, ID extends Serializable> CoreBpmnInstanceNextTaskElementDTO<ID> getNextTask(CoreBpmnRuntimeSource<T, ID> source) {
      List<CoreBpmnInstanceNextTaskElementDTO<ID>> nextTaskElementList = getNextTaskList();
      return (CoreBpmnInstanceNextTaskElementDTO)nextTaskElementList.parallelStream().filter((t) -> {
         return t.getId().equals(source.getId());
      }).findAny().orElse(null);
   }

   public static <T extends Auditable<ID>, ID extends Serializable> boolean isLastTaskCandidator(CoreBpmnRuntimeSource<T, ID> source) {
      Element currentElement = getBpmnRuntimeTaskElement(source);
      CounterSignStrategy counterSignStrategy = BpmnDiagramHelper.getCounterSignStrategy(currentElement);
      if (CounterSignStrategy.none.equals(counterSignStrategy)) {
         return true;
      } else {
         List<CoreBpmnInstanceCandidatorDTO> instanceCandidatorList = getBpmnRuntimeInstanceCandidatorList();
         CoreBpmnInstanceTaskBean instanceTask = getBpmnRuntimeInstanceTask(source);
         List<CoreBpmnInstanceCandidatorDTO> candidatorList = (List)instanceCandidatorList.stream().filter((c) -> {
            return c.getInstTaskId().equals(instanceTask.getId());
         }).collect(Collectors.toList());
         if (candidatorList.isEmpty()) {
            throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.CANDIDATOR_AND_ROLE_NOT_EXISTS");
         } else if (CounterSignStrategy.all.equals(counterSignStrategy)) {
            return candidatorList.stream().allMatch((c) -> {
               return LocalContextHelper.getLoginUserId().equals(c.getCandidatorId());
            });
         } else {
            return candidatorList.stream().filter((c) -> {
               return c.getRoleId() != null;
            }).count() == candidatorList.stream().filter((c) -> {
               return c.getRoleId() != null && LocalContextHelper.getLoginUserId().equals(c.getCandidatorId());
            }).count();
         }
      }
   }

   private static <T extends Auditable<ID>, ID extends Serializable> boolean isStartTask(CoreBpmnRuntimeSource<T, ID> source) {
      return StringUtils.startsWithIgnoreCase(source.getOldItem().getProcessStatus(), ProcessStatus.DRAFT.name());
   }

   private static <T extends Auditable<ID>, ID extends Serializable> List<Element> getNextTaskElementList(CoreBpmnRuntimeSource<T, ID> source, Element currentTask) {
      Document diagram = getBpmnRuntimeDocument(source);
      if (!StringUtils.isBlank(source.getNextStatusCode())) {
         return Arrays.asList(BpmnDiagramHelper.getProcessElementByStatusCode(diagram, source.getNextStatusCode()));
      } else {
         CounterSignStrategy counterSignStrategy = BpmnDiagramHelper.getCounterSignStrategy(currentTask);
         List reachableTaskList;
         if (!CounterSignStrategy.none.equals(counterSignStrategy)) {
            List<CoreBpmnInstanceCandidatorDTO> instanceCandidatorList = getBpmnRuntimeInstanceCandidatorList();
            CoreBpmnInstanceTaskBean instanceTask = getBpmnRuntimeInstanceTask(source);
            if (!instanceTask.getTaskId().equals(XmlUtils.getAttributeValue(currentTask, "id"))) {
               return CollectionUtils.emptyList();
            }

            List<String> previousTaskAuditedByList = getBpmnPreviousTaskAuditedByList(source);
            List<CoreBpmnInstanceCandidatorDTO> currentInstanceCandidatorList = instanceCandidatorList.stream().filter((c) -> {
               return c.getInstTaskId().equals(instanceTask.getId()) && !previousTaskAuditedByList.contains(c.getCandidatorId());
            }).collect(Collectors.toList());
            if (CandidatorFilterStrategy.sameDept.equals(BpmnDiagramHelper.getCandidatorFilterStrategy(currentTask))) {
               String createdByOrgId = (String)CollectionUtils.getValueIgnorecase(source.getVars(), "createdByOrgId");
               if (!StringUtils.isBlank(createdByOrgId)) {
                  CoreUserService userService = ApplicationContextHelper.getBean(CoreUserService.class);
                  reachableTaskList = userService.selectListByIds(currentInstanceCandidatorList.stream().map((c) -> c.getCandidatorId()).collect(Collectors.toList()));
                  List<CoreUserBean> finalReachableTaskList = reachableTaskList;
                  currentInstanceCandidatorList.removeIf((c) -> finalReachableTaskList.stream().filter((s) -> s.getId().equals(c.getCandidatorId())).noneMatch((s) -> createdByOrgId.equals(s.getOrgId())));
               }
            }

            if (CounterSignStrategy.all.equals(counterSignStrategy)) {
               if (currentInstanceCandidatorList.stream().anyMatch((c) -> {
                  return !c.getCandidatorId().equals(LocalContextHelper.getLoginUserId());
               })) {
                  return CollectionUtils.emptyList();
               }
            } else if (CounterSignStrategy.role.equals(counterSignStrategy)) {
               List<Long> roleIdList = (List)currentInstanceCandidatorList.stream().filter((c) -> {
                  return c.getTaskRoleId() != null && c.getCandidatorId().equals(LocalContextHelper.getLoginUserId());
               }).map((c) -> {
                  return c.getRoleId();
               }).distinct().collect(Collectors.toList());
               if (currentInstanceCandidatorList.stream().anyMatch((c) -> {
                  return c.getTaskRoleId() != null && !roleIdList.contains(c.getRoleId());
               })) {
                  return CollectionUtils.emptyList();
               }
            }
         }

         List<Element> resultList = new ArrayList();
         List<Element> outgoingSequenceFlowList = BpmnDiagramHelper.getOutgoingSequenceFlowList(diagram, currentTask);
         Iterator var16 = outgoingSequenceFlowList.iterator();

         while(var16.hasNext()) {
            Element outgoingSequenceFlow = (Element)var16.next();
            if (CoreBpmnHelper.evalSequenceFlowConditionExpression(outgoingSequenceFlow, source.getVars())) {
               Element targetElement = BpmnDiagramHelper.getTargetElement(diagram, outgoingSequenceFlow);
               ProcessElementType targetElementType = BpmnDiagramHelper.getProcessElementType(targetElement);
               if (ProcessElementType.USER_TASK.equals(targetElementType)) {
                  resultList.add(targetElement);
               } else if (ProcessElementType.END_EVENT.equals(targetElementType)) {
                  resultList.add(targetElement);
               } else if (ProcessElementType.PARALLEL_GATEWAY.equals(targetElementType)) {
                  reachableTaskList = BpmnDiagramHelper.getIncomingSequenceFlowList(diagram, targetElement);
                  if (reachableTaskList.size() == 1) {
                     resultList.addAll(getNextTaskElementList(source, targetElement));
                  } else {
                     CoreBpmnInstanceBean instance = getBpmnRuntimeInstance(source);
                     List<CoreBpmnInstanceTaskBean> instanceTaskCacheList = getBpmnRuntimeUnlimitedInstanceTaskList();
                     List<CoreBpmnInstanceTaskBean> instanceTaskList = (List)instanceTaskCacheList.stream().filter((i) -> {
                        return i.getInstId().equals(instance.getId());
                     }).collect(Collectors.toList());
                     if ((long)instanceTaskList.size() == 1L && ((CoreBpmnInstanceTaskBean)instanceTaskList.get(0)).getTaskId().equals(XmlUtils.getAttributeValue(currentTask, "id"))) {
                        resultList.addAll(getNextTaskElementList(source, targetElement));
                     }
                  }
               } else if (ProcessElementType.EXCLUSIVE_GATEWAY.equals(targetElementType)) {
                  reachableTaskList = getNextTaskElementList(source, targetElement);
                  if (reachableTaskList.size() > 1) {
                     throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.EXCLUSIVE_GATEWAY_OUTGOING_MULTIPLE_TASK");
                  }

                  resultList.addAll(reachableTaskList);
               }
            }
         }

         return resultList;
      }
   }
}
