/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.module.sys.bpmn.diagram;

import net.sinodawn.framework.support.domain.Auditable;
import net.sinodawn.framework.utils.NumberUtils;
import net.sinodawn.framework.utils.ObjectUtils;
import net.sinodawn.framework.utils.StringUtils;
import net.sinodawn.framework.utils.XmlUtils;
import net.sinodawn.module.sys.bpmn.bean.CoreBpmnTaskCandidatorsDTO;
import net.sinodawn.module.sys.bpmn.diagram.attribute.AttachmentStrategy;
import net.sinodawn.module.sys.bpmn.diagram.attribute.CandidatorFilterStrategy;
import net.sinodawn.module.sys.bpmn.diagram.attribute.CounterSignStrategy;
import net.sinodawn.module.sys.bpmn.diagram.attribute.NextCandidatorOptStrategy;
import net.sinodawn.module.sys.bpmn.engine.CoreBpmnRuntimeSource;
import net.sinodawn.module.sys.bpmn.exception.BpmnException;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;
import org.dom4j.Document;
import org.dom4j.Element;

public abstract class BpmnDiagramHelper {
   public static final ProcessElementType getProcessElementType(Element processElement) {
      String name = processElement.getName();
      return ProcessElementType.nameOf(name);
   }

   public static final Element getProcessElement(Document diagram, String id) {
      if (StringUtils.isBlank(id)) {
         return null;
      } else {
         Element processElement = getProcessElement(diagram);
         List<Element> elementList = processElement.elements();
         Iterator var4 = elementList.iterator();

         Element element;
         do {
            if (!var4.hasNext()) {
               return null;
            }

            element = (Element)var4.next();
         } while(!id.equals(XmlUtils.getAttributeValue(element, "id")));

         return element;
      }
   }

   public static final Element getProcessElement(Document diagram, ProcessElementType type) {
      Element element = getProcessElement(diagram);
      return element.element(type.getName());
   }

   public static final Element getProcessElementByStatusCode(Document diagram, String statusCode) {
      Element processElement = getProcessElement(diagram);
      List<Element> elementList = processElement.elements();
      Iterator var4 = elementList.iterator();

      Element element;
      do {
         if (!var4.hasNext()) {
            return null;
         }

         element = (Element)var4.next();
      } while(!statusCode.equals(getUserTaskStatusCode(element)));

      return element;
   }

   public static final List<Element> getProcessElementList(Document diagram, ProcessElementType processElementType) {
      Element element = getProcessElement(diagram);
      return element.elements(processElementType.getName());
   }

   public static final List<Element> getOutgoingSequenceFlowList(Document diagram, Element taskElement) {
      List<Element> taskSubOutgoingElementList = taskElement.elements(TaskElementType.outgoing.name());
      return (List)taskSubOutgoingElementList.stream().map((e) -> {
         return getProcessElement(diagram, e.getStringValue());
      }).collect(Collectors.toList());
   }

   public static final List<Element> getOutgoingSequenceFlowList(Document diagram, String taskId) {
      Element taskElement = getProcessElement(diagram, taskId);
      return getOutgoingSequenceFlowList(diagram, taskElement);
   }

   public static final List<String> getOutgoingSequenceFlowIdList(Document diagram, Element taskElement) {
      List<Element> taskSubOutgoingElementList = taskElement.elements(TaskElementType.outgoing.name());
      return (List)taskSubOutgoingElementList.stream().map((e) -> {
         return e.getStringValue();
      }).collect(Collectors.toList());
   }

   public static final List<String> getOutgoingSequenceFlowIdList(Document diagram, String taskId) {
      Element taskElement = getProcessElement(diagram, taskId);
      return getOutgoingSequenceFlowIdList(diagram, taskElement);
   }

   public static final boolean isLastUserTask(Document diagram, Element userTaskElement) {
      List<Element> outgoingSequenceFlowList = getOutgoingSequenceFlowList(diagram, userTaskElement);
      if (outgoingSequenceFlowList.isEmpty()) {
         return true;
      } else if (outgoingSequenceFlowList.size() > 1) {
         return false;
      } else {
         Element sequenceFlow = (Element)outgoingSequenceFlowList.get(0);
         Element targetElement = getTargetElement(diagram, sequenceFlow);
         ProcessElementType targetType = getProcessElementType(targetElement);
         return ProcessElementType.END_EVENT.equals(targetType);
      }
   }

   public static final List<Element> getIncomingSequenceFlowList(Document diagram, String taskId) {
      Element taskElement = getProcessElement(diagram, taskId);
      return getIncomingSequenceFlowList(diagram, taskElement);
   }

   public static final List<Element> getIncomingSequenceFlowList(Document diagram, Element taskElement) {
      List<Element> taskSubIncomingElementList = taskElement.elements(TaskElementType.incoming.name());
      return (List)taskSubIncomingElementList.stream().map((e) -> {
         return getProcessElement(diagram, e.getStringValue());
      }).collect(Collectors.toList());
   }

   public static final List<String> getIncomingSequenceFlowIdList(Document diagram, Element taskElement) {
      List<Element> taskSubIncomingElementList = taskElement.elements(TaskElementType.incoming.name());
      return (List)taskSubIncomingElementList.stream().map((e) -> {
         return e.getStringValue();
      }).collect(Collectors.toList());
   }

   public static final List<Element> getIncomingUserTaskElementList(Document diagram, Element taskElement) {
      List<Element> userTaskElementList = new ArrayList();
      List<Element> taskSubIncomingElementList = getIncomingSequenceFlowList(diagram, taskElement);
      Iterator var4 = taskSubIncomingElementList.iterator();

      while(true) {
         Element sourceElement;
         ProcessElementType sourceType;
         do {
            while(true) {
               do {
                  if (!var4.hasNext()) {
                     return userTaskElementList;
                  }

                  Element sequenceFlow = (Element)var4.next();
                  sourceElement = getSourceElement(diagram, sequenceFlow);
               } while(sourceElement == null);

               sourceType = getProcessElementType(sourceElement);
               if (!ProcessElementType.USER_TASK.equals(sourceType) && !ProcessElementType.START_EVENT.equals(sourceType)) {
                  break;
               }

               userTaskElementList.add(sourceElement);
            }
         } while(!ProcessElementType.PARALLEL_GATEWAY.equals(sourceType) && !ProcessElementType.EXCLUSIVE_GATEWAY.equals(sourceType) && !ProcessElementType.SEQUENCE_FLOW.equals(sourceType));

         userTaskElementList.addAll(getIncomingUserTaskElementList(diagram, sourceElement));
      }
   }

   public static final List<String> getIncomingSequenceFlowIdList(Document diagram, String taskId) {
      Element taskElement = getProcessElement(diagram, taskId);
      return getIncomingSequenceFlowIdList(diagram, taskElement);
   }

   public static final Element getTargetElement(Document diagram, String sequenceFlowId) {
      Element element = getProcessElement(diagram, sequenceFlowId);
      return getTargetElement(diagram, element);
   }

   public static final Element getTargetElement(Document diagram, Element sequenceFlowElement) {
      String targetId = XmlUtils.getAttributeValue(sequenceFlowElement, "targetRef");
      return getProcessElement(diagram, targetId);
   }

   public static final Element getSourceElement(Document diagram, String sequenceFlowId) {
      Element element = getProcessElement(diagram, sequenceFlowId);
      return getSourceElement(diagram, element);
   }

   public static final Element getSourceElement(Document diagram, Element sequenceFlowElement) {
      String targetId = XmlUtils.getAttributeValue(sequenceFlowElement, "sourceRef");
      return getProcessElement(diagram, targetId);
   }

   public static final String getConditionExpression(Document diagram, String sequenceFlowId) {
      Element sequenceFlowElement = getProcessElement(diagram, sequenceFlowId);
      return getConditionExpression(sequenceFlowElement);
   }

   public static final String getConditionExpression(Element sequenceFlowElement) {
      Element conditionExpressionElement = sequenceFlowElement.element(SequenceElementType.conditionExpression.name());
      return conditionExpressionElement == null ? null : conditionExpressionElement.getStringValue();
   }

   public static final String getUserTaskInitiatorValue(Element userTask) {
      return XmlUtils.getAttributeValue(userTask, "initiator");
   }

   public static final String getUserTaskDynamicValue(Element userTask) {
      return XmlUtils.getAttributeValue(userTask, "dynamic");
   }

   public static final Map<String, String> getUserTaskCandidatorValue(Element userTask) {
      Map<String, String> map = new HashMap();
      Element candidateUsersElement = userTask.element(TaskElementType.candidateUsersElements.name());
      if (candidateUsersElement != null) {
         Element propertiesElement = candidateUsersElement.element("properties");
         propertiesElement.elements("property").forEach((e) -> {
            String var10000 = (String)map.put(XmlUtils.getAttributeValue(e, "value"), XmlUtils.getAttributeValue(e, "name"));
         });
      }

      return map;
   }

   public static final Map<Long, String> getUserTaskRoleValue(Element userTask) {
      Map<Long, String> map = new HashMap();
      Element candidateUsersElement = userTask.element(TaskElementType.candidateRolesElements.name());
      if (candidateUsersElement != null) {
         Element propertiesElement = candidateUsersElement.element("properties");
         propertiesElement.elements("property").forEach((e) -> {
            String var10000 = (String)map.put(NumberUtils.parseLong(XmlUtils.getAttributeValue(e, "value")), XmlUtils.getAttributeValue(e, "name"));
         });
      }

      return map;
   }

   public static final String getUserTaskStatusCode(Element userTask) {
      return XmlUtils.getAttributeValue(userTask, "statusCode");
   }

   public static final boolean isAuthRequiredUserTask(Element userTask) {
      return "true".equalsIgnoreCase(XmlUtils.getAttributeValue(userTask, "authRequired"));
   }

   public static final boolean isCommentRequiredTask(Element startEventOrUserTask) {
      return "true".equalsIgnoreCase(XmlUtils.getAttributeValue(startEventOrUserTask, "commentRequired"));
   }

   public static final boolean isFilterPreviousCandidatorsTask(Element userTask) {
      return "true".equalsIgnoreCase(XmlUtils.getAttributeValue(userTask, "filterPreviousCandidators"));
   }

   public static final boolean isTransferTask(Element userTask) {
      return "true".equalsIgnoreCase(XmlUtils.getAttributeValue(userTask, "transfer"));
   }

   public static final AttachmentStrategy getAttachmentStrategy(Element startEventOrUserTask) {
      String attachmentStrategy = XmlUtils.getAttributeValue(startEventOrUserTask, "attachmentStrategy");
      return StringUtils.isBlank(attachmentStrategy) ? AttachmentStrategy.none : AttachmentStrategy.valueOf(attachmentStrategy);
   }

   public static final CounterSignStrategy getCounterSignStrategy(Element userTask) {
      String counterSignStrategy = XmlUtils.getAttributeValue(userTask, "counterSignStrategy");
      return StringUtils.isBlank(counterSignStrategy) ? CounterSignStrategy.none : CounterSignStrategy.valueOf(counterSignStrategy);
   }

   public static final NextCandidatorOptStrategy getNextCandidatorOptStrategy(Element startEventOrUserTask) {
      String nextCandidatorOptStrategy = XmlUtils.getAttributeValue(startEventOrUserTask, "nextCandidatorOptStrategy");
      return StringUtils.isBlank(nextCandidatorOptStrategy) ? NextCandidatorOptStrategy.none : NextCandidatorOptStrategy.valueOf(nextCandidatorOptStrategy);
   }

   public static final CandidatorFilterStrategy getCandidatorFilterStrategy(Element userTask) {
      String candidatorFilterStrategy = XmlUtils.getAttributeValue(userTask, "candidatorFilterStrategy");
      return StringUtils.isBlank(candidatorFilterStrategy) ? CandidatorFilterStrategy.none : CandidatorFilterStrategy.valueOf(candidatorFilterStrategy);
   }

   public static final String getPassValidatorSql(Element startEventOrUserTask) {
      return XmlUtils.getAttributeValue(startEventOrUserTask, "passValidator");
   }

   public static final String getRejectValidatorSql(Element userTask) {
      return XmlUtils.getAttributeValue(userTask, "rejectValidator");
   }

   public static final String getPassCallback(Element startEventOrUserTask) {
      return XmlUtils.getAttributeValue(startEventOrUserTask, "passCallback");
   }

   public static final String getEndCallback(Element startEventOrUserTask) {
      return XmlUtils.getAttributeValue(startEventOrUserTask, "endCallback");
   }

   public static final String getRejectCallback(Element userTask) {
      return XmlUtils.getAttributeValue(userTask, "rejectCallback");
   }

   public static List<String> getPreviousStatusCodeList(Document document, Element element) {
      List<String> statusCodeList = new ArrayList();
      pickPreviousStatusCodeList(statusCodeList, document, element);
      return (List)statusCodeList.stream().distinct().collect(Collectors.toList());
   }

   public static final <T extends Auditable<ID>, ID extends Serializable> CoreBpmnTaskCandidatorsDTO getUserTaskCandidators(CoreBpmnRuntimeSource<T, ID> source, Element userTaskElement, String initiator) {
      List<String> candidatorIdList = new ArrayList();
      List<Long> candidateRoleIdList = new ArrayList();
      if (!StringUtils.isEmpty(getUserTaskInitiatorValue(userTaskElement))) {
         candidatorIdList.add(initiator);
      }

      String dynamic = getUserTaskDynamicValue(userTaskElement);
      if (!StringUtils.isEmpty(dynamic)) {
         String[] dynamicValues = StringUtils.split(dynamic, ",");
         String[] var7 = dynamicValues;
         int var8 = dynamicValues.length;

         for(int var9 = 0; var9 < var8; ++var9) {
            String value = var7[var9];
            String role;
            if (StringUtils.startsWith(value, "u:")) {
               role = (String)source.getVars().get(StringUtils.removeStart(value, "u:"));
               if (role == null) {
                  throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.DYNAMIC_USER_NOT_EXISTS");
               }

               candidatorIdList.add(role);
            } else if (StringUtils.startsWith(value, "r:")) {
               role = ObjectUtils.toString(source.getVars().get(StringUtils.removeStart(value, "r:")));
               if (StringUtils.isEmpty(role)) {
                  throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.DYNAMIC_ROLE_NOT_EXISTS");
               }

               candidateRoleIdList.add(NumberUtils.parseLong(role));
            }
         }
      }

      Map<String, String> userTaskCandidatorMap = getUserTaskCandidatorValue(userTaskElement);
      if (!userTaskCandidatorMap.isEmpty()) {
         candidatorIdList.addAll(userTaskCandidatorMap.keySet());
      }

      Map<Long, String> userTaskRoleMap = getUserTaskRoleValue(userTaskElement);
      if (!userTaskRoleMap.isEmpty()) {
         candidateRoleIdList.addAll(userTaskRoleMap.keySet());
      }

      CoreBpmnTaskCandidatorsDTO candidators = new CoreBpmnTaskCandidatorsDTO();
      candidators.setCandidatorIdList((List)candidatorIdList.stream().distinct().collect(Collectors.toList()));
      candidators.setCandidateRoleIdList((List)candidateRoleIdList.stream().distinct().collect(Collectors.toList()));
      return candidators;
   }

   private static final Element getProcessElement(Document diagram) {
      return diagram.getRootElement().element("process");
   }

   private static void pickPreviousStatusCodeList(List<String> pickedStatusCodeList, Document document, Element element) {
      List<Element> sequenceFlowList = getIncomingSequenceFlowList(document, element);
      Iterator var4 = sequenceFlowList.iterator();

      while(true) {
         Element source;
         ProcessElementType type;
         while(true) {
            if (!var4.hasNext()) {
               return;
            }

            Element sequenceFlowElement = (Element)var4.next();
            source = getSourceElement(document, sequenceFlowElement);
            type = getProcessElementType(source);
            if (!ProcessElementType.USER_TASK.equals(type) && !ProcessElementType.START_EVENT.equals(type)) {
               break;
            }

            String statusCode = getUserTaskStatusCode(source);
            if (!pickedStatusCodeList.contains(statusCode)) {
               pickedStatusCodeList.add(getUserTaskStatusCode(source));
               break;
            }
         }

         if (!ProcessElementType.START_EVENT.equals(type)) {
            pickPreviousStatusCodeList(pickedStatusCodeList, document, source);
         }
      }
   }
}
