/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.module.mdm.user.service.impl;

import net.sinodawn.framework.at.annotation.AuditTrailEntry;
import net.sinodawn.framework.at.annotation.AuditTrailType;
import net.sinodawn.framework.audit.aunnotation.Audit;
import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.context.LocalContextHelper;
import net.sinodawn.framework.data.page.Page;
import net.sinodawn.framework.database.sql.Order;
import net.sinodawn.framework.exception.InvalidDataException;
import net.sinodawn.framework.io.file.FileScope;
import net.sinodawn.framework.mybatis.mapper.MapperParameter;
import net.sinodawn.framework.mybatis.mapper.MatchPattern;
import net.sinodawn.framework.mybatis.mapper.SearchFilter;
import net.sinodawn.framework.mybatis.page.PageRowBounds;
import net.sinodawn.framework.restful.data.RestHttpTextEntity;
import net.sinodawn.framework.restful.data.RestJsonWrapperBean;
import net.sinodawn.framework.security.authentication.AuthenticationHelper;
import net.sinodawn.framework.security.service.DefaultUserDetailsChecker;
import net.sinodawn.framework.utils.*;
import net.sinodawn.module.item.file.bean.CoreFileBean;
import net.sinodawn.module.item.file.manager.CoreFileManager;
import net.sinodawn.module.item.file.service.CoreFileService;
import net.sinodawn.module.mdm.org.service.CoreOrgUserService;
import net.sinodawn.module.mdm.user.bean.CoreUserBean;
import net.sinodawn.module.mdm.user.bean.CoreUserPermissionBean;
import net.sinodawn.module.mdm.user.bean.CoreUserSignatureDTO;
import net.sinodawn.module.mdm.user.dao.CoreUserDao;
import net.sinodawn.module.mdm.user.service.CoreUserPermissionService;
import net.sinodawn.module.mdm.user.service.CoreUserService;
import net.sinodawn.module.sys.addomain.bean.CoreAdDomainUserBean;
import net.sinodawn.module.sys.addomain.service.CoreAdDomainUserService;
import net.sinodawn.module.sys.config.bean.CoreLoginConfigBean;
import net.sinodawn.module.sys.config.bean.CoreSystemConfigBean;
import net.sinodawn.module.sys.config.helper.SystemConfigHelper;
import net.sinodawn.module.sys.password.bean.CorePasswordHisBean;
import net.sinodawn.module.sys.password.service.CorePasswordHisService;
import net.sinodawn.module.sys.password.service.CorePasswordPolicyService;
import net.sinodawn.module.sys.role.service.CoreRoleUserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.context.annotation.Lazy;
import org.springframework.http.HttpStatus;
import org.springframework.security.crypto.password.PasswordEncoder;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

@Repository
public class CoreUserServiceImpl implements CoreUserService {
   private static final List<String> INVALID_USERID_LIST = Arrays.asList("DEFAULT", "SYSTEM", "ADMIN", "ROOT");
   @Autowired
   private CoreUserDao userDao;
   @Autowired
   @Lazy
   private CoreUserService proxyInstance;
   @Autowired
   @Lazy
   private CoreRoleUserService roleUserService;
   @Autowired
   @Lazy
   private CoreOrgUserService orgUserService;
   @Autowired
   private CoreUserPermissionService userPermissionService;
   @Autowired
   @Lazy
   private CorePasswordPolicyService passwordPolicyService;
   @Autowired
   @Lazy
   private CorePasswordHisService passwordHisService;
   @Autowired
   @Lazy
   private CoreFileService coreFileService;
   @Autowired
   @Lazy
   private PasswordEncoder passwordEncoder;
   @Autowired
   @Lazy
   private CoreAdDomainUserService coreAdDomainUserService;
   @Autowired
   @Lazy
   private CoreFileManager fileManager;

   public CoreUserDao getDao() {
      return this.userDao;
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.INSERT)
   @Audit("AUDIT.INSERT")
   public String insert(RestJsonWrapperBean wrapper) {
      CoreUserBean user = (CoreUserBean)wrapper.parseUnique(CoreUserBean.class);
      if (CollectionUtils.containsIgnoreCase(INVALID_USERID_LIST, user.getId())) {
         throw new InvalidDataException("SINO.USER.INVALID_USER_ID");
      } else {
         user.setPassword(this.getDefaultPassword(user.getId()));
         LocalDateTime now = LocalDateTime.now();
         user.setPasswordUpdatedTime(now);
         user.setStatus("locked");
         long expiryInterval = NumberUtils.parseLong(ApplicationContextHelper.getConstantValue("ACCOUNT_EXPIRY_INTERVAL"));
         if (expiryInterval > 0L) {
            user.setExpiryDate(now.plusDays(expiryInterval));
         }

         CoreLoginConfigBean loginConfig = SystemConfigHelper.getLoginConfig();
         if ("adDomain".equals(loginConfig.getLoginAuthMethod()) || "systemOrAdDomain".equals(loginConfig.getLoginAuthMethod())) {
            CoreAdDomainUserBean adDomainUser = new CoreAdDomainUserBean();
            adDomainUser.setId(user.getId());
            if ("1".equals(loginConfig.getAdUserIdIgnoreCase())) {
               adDomainUser.setAdUserId(StringUtils.lowerCase(user.getId()));
            } else {
               adDomainUser.setAdUserId(user.getId());
            }

            this.coreAdDomainUserService.getDao().insert(adDomainUser);
         }

         this.getDao().insert(user);
         return user.getId();
      }
   }

   @AuditTrailEntry(AuditTrailType.UPDATE)
   @Audit("AUDIT.SAVE")
   public void postUpdate(List<String> updatedColumnNameList, List<String> idList) {
      if (CollectionUtils.containsIgnoreCase(updatedColumnNameList, "status")) {
         List<CoreUserBean> updatedUserList = this.selectListByIds(idList);
         List<CoreUserBean> updatedStatusUserList = (List)updatedUserList.stream().filter((u) -> {
            return "activated".equals(u.getStatus());
         }).map((u) -> {
            u.setExpiredPasswordUses(0);
            u.setFailedLoginAttempts(0);
            return u;
         }).collect(Collectors.toList());
         if (!updatedStatusUserList.isEmpty()) {
            this.getDao().update(updatedStatusUserList, new String[]{"EXPIREDPASSWORDUSES", "FAILEDLOGINATTEMPTS"});
         }
      }

   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.DELETE)
   @Audit("AUDIT.DELETE")
   public void delete(RestJsonWrapperBean wrapper) {
      wrapper.parse(CoreUserBean.class).forEach((u) -> {
         this.proxyInstance.delete(u.getId());
      });
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.DELETE)
   @Audit("AUDIT.DELETE")
   public void delete(String id) {
      this.getDao().delete(id);
      this.roleUserService.deleteByUserId(id);
   }

   public CoreUserBean selectById(String id) {
      return (CoreUserBean)this.getDao().selectByIdIfPresent(id);
   }

   public CoreUserBean selectByLoginId(String loginId) {
      String loginMode = ApplicationContextHelper.getConstantValue("LOGIN.LOGINMODE");
      CoreUserBean search = new CoreUserBean();
      if (!StringUtils.isEmpty(loginMode) && loginMode.length() == 5) {
         if ("1".equals(loginMode.substring(0, 1))) {
            search.setId(loginId);
         }

         if ("1".equals(loginMode.substring(1, 2))) {
            search.setMobile(loginId);
         }

         if ("1".equals(loginMode.substring(2, 3))) {
            search.setIdCard(loginId);
         }

         if ("1".equals(loginMode.substring(3, 4))) {
            search.setEmail(loginId);
         }

         if ("1".equals(loginMode.substring(4, 5))) {
            search.setUserNo(loginId);
         }

         return (CoreUserBean)this.getDao().selectOneUnionIfPresent(search, new Order[0]);
      } else {
         return (CoreUserBean)this.proxyInstance.selectById(loginId);
      }
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.UPDATE)
   @Audit("AUDIT.SAVE")
   public void resetPassword(RestJsonWrapperBean wrapper) {
      wrapper.parse(CoreUserBean.class).forEach((u) -> {
         this.proxyInstance.resetPassword(u.getId());
      });
   }

   public void resetPassword(String id) {
      CoreUserBean user = (CoreUserBean)BeanUtils.getPropertyListenerProxy((Object)this.getDao().selectById(id));
      String oldPassword = user.getPassword();
      user.setId(id);
      user.setPassword(this.getDefaultPassword(id));
      user.setFailedLoginAttempts(0);
      user.setExpiredPasswordUses(0);
      user.setPasswordUpdatedTime(LocalDateTime.now());
      this.getDao().update(user);
      CorePasswordHisBean passwordHis = new CorePasswordHisBean();
      passwordHis.setId(ApplicationContextHelper.getNextIdentity());
      passwordHis.setUserId(user.getId());
      passwordHis.setPassword(oldPassword);
      this.passwordHisService.getDao().insert(passwordHis);
   }

   @AuditTrailEntry(AuditTrailType.UPDATE)
   @Audit("AUDIT.SAVE")
   public RestHttpTextEntity changePassword(RestJsonWrapperBean wrapper) {
      String oldPassword = AuthenticationHelper.getRawPassword(wrapper.getParamValue("oldPassword"));
      String newRawPassword = AuthenticationHelper.getRawPassword(wrapper.getParamValue("newRawPassword"));
      if (!StringUtils.isEmpty(oldPassword) && !StringUtils.isEmpty(newRawPassword)) {
         CoreUserBean user = (CoreUserBean)BeanUtils.getPropertyListenerProxy((Object)LocalContextHelper.getLoginUser());
         String encodedOldPassword = this.passwordEncoder.encode(oldPassword);
         if (!user.getPassword().equals(encodedOldPassword)) {
            ServletUtils.responseError("SINO.USER.PASSWORD.INVALID_OLD_PASSWORD");
            return new RestHttpTextEntity(HttpStatus.FORBIDDEN);
         } else {
            return this.changePassword(user.getId(), newRawPassword);
         }
      } else {
         ServletUtils.responseError("SINO.USER.PASSWORD.REQUIRE_NOT_EMPTY");
         return new RestHttpTextEntity(HttpStatus.BAD_REQUEST);
      }
   }

   @AuditTrailEntry(AuditTrailType.UPDATE)
   @Audit("AUDIT.SAVE")
   public RestHttpTextEntity changeNewPassword(RestJsonWrapperBean wrapper) {
      String newRawPassword = AuthenticationHelper.getRawPassword(wrapper.getParamValue("newRawPassword"));
      if (StringUtils.isEmpty(newRawPassword)) {
         ServletUtils.responseError("SINO.USER.PASSWORD.REQUIRE_NOT_EMPTY");
         return new RestHttpTextEntity(HttpStatus.BAD_REQUEST);
      } else {
         return this.changePassword(LocalContextHelper.getLoginUserId(), newRawPassword);
      }
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.UPDATE)
   @Audit("AUDIT.SAVE")
   public RestHttpTextEntity changePassword(String id, RestJsonWrapperBean wrapper) {
      String newRawPassword = AuthenticationHelper.getRawPassword(wrapper.getParamValue("newRawPassword"));
      if (StringUtils.isEmpty(newRawPassword)) {
         ServletUtils.responseError("SINO.USER.PASSWORD.REQUIRE_NOT_EMPTY");
         return new RestHttpTextEntity(HttpStatus.BAD_REQUEST);
      } else {
         return this.changePassword(id, newRawPassword);
      }
   }

   public Page<CoreUserBean> selectChoosablePagination(RestJsonWrapperBean wrapper) {
      MapperParameter parameter = wrapper.extractMapFilter();
      PageRowBounds rowBounds = wrapper.extractPageRowBounds();
      parameter.setSuspendedFlag("0");
      parameter.setChoosableQueries();
      return this.selectPagination(parameter, rowBounds);
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.UPDATE)
   @Audit("AUDIT.SAVE")
   public void deprecate(RestJsonWrapperBean wrapper) {
      List<CoreUserBean> userList = wrapper.parse(CoreUserBean.class);
      List<String> idList = (List)userList.stream().map((e) -> {
         return e.getId();
      }).collect(Collectors.toList());
      CoreUserBean user = new CoreUserBean();
      user.setStatus("deprecated");
      this.getDao().updateByIds(user, idList, new String[0]);
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.INSERT)
   @Audit("AUDIT.INSERT")
   public void insertUserPermission(String id, String permissionTypeId, RestJsonWrapperBean wrapper) {
      this.userPermissionService.insert(id, permissionTypeId, wrapper);
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.DELETE)
   @Audit("AUDIT.DELETE")
   public void deleteUserPermission(String id, String permissionTypeId, RestJsonWrapperBean wrapper) {
      List<String> idList = wrapper.parseId(String.class);
      Iterator var6;
      if (idList.stream().allMatch((i) -> {
         return NumberUtils.isNumber(i);
      })) {
         List<CoreUserPermissionBean> permissionList = wrapper.parse(CoreUserPermissionBean.class);
         var6 = permissionList.iterator();

         while(var6.hasNext()) {
            CoreUserPermissionBean permission = (CoreUserPermissionBean)var6.next();
            permission.setPermissionTypeId(permissionTypeId);
            permission.setUserId(id);
         }

         if (permissionList.stream().allMatch((p) -> {
            return StringUtils.isEmpty(p.getTargetId());
         })) {
            permissionList.forEach((p) -> {
               p.setTargetId("" + p.getId());
            });
         }

         this.userPermissionService.getDao().deleteBy(permissionList, "USERID", "TARGETID", "PERMISSIONTYPEID");
      } else {
         List<CoreUserPermissionBean> permissionList = new ArrayList();
         var6 = idList.iterator();

         while(var6.hasNext()) {
            String strId = (String)var6.next();
            CoreUserPermissionBean permission = new CoreUserPermissionBean();
            permission.setTargetId(strId);
            permission.setPermissionTypeId(permissionTypeId);
            permission.setUserId(id);
            permissionList.add(permission);
         }

         this.userPermissionService.getDao().deleteBy((List)permissionList, "USERID", "TARGETID", "PERMISSIONTYPEID");
      }

   }

   public boolean defaultPassword() {
      String userId = LocalContextHelper.getLoginUserId();
      return ((CoreUserBean)this.proxyInstance.selectById(userId)).getPassword().equals(this.getEncodedPassword(userId));
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.UPDATE)
   @Audit("AUDIT.SAVE")
   public void setDefaultUserPermission(String id, String permissionTypeId, RestJsonWrapperBean wrapper) {
      List<CoreUserPermissionBean> updateUserPermissionList = new ArrayList();
      CoreUserPermissionBean filter = new CoreUserPermissionBean();
      filter.setUserId(id);
      filter.setPermissionTypeId(permissionTypeId);
      filter.setDefaultFlag("1");
      List<CoreUserPermissionBean> userPermissionList = this.userPermissionService.selectList(filter, new Order[0]);
      CoreUserPermissionBean defaultUserPermission;
      if (userPermissionList.size() > 0) {
         defaultUserPermission = (CoreUserPermissionBean)userPermissionList.get(0);
         defaultUserPermission.setDefaultFlag("0");
         updateUserPermissionList.add(defaultUserPermission);
      }

      defaultUserPermission = wrapper.parseUnique(CoreUserPermissionBean.class);
      defaultUserPermission.setDefaultFlag("1");
      updateUserPermissionList.add(defaultUserPermission);
      this.userPermissionService.getDao().update(updateUserPermissionList, "DEFAULTFLAG");
   }

   public CoreUserSignatureDTO sign(RestJsonWrapperBean wrapper) {
      String username = wrapper.getParamValue("username");
      String password = wrapper.getParamValue("password");
      DefaultUserDetailsChecker.checkPassword(username, password);
      CoreUserSignatureDTO userSignature = new CoreUserSignatureDTO();
      CoreUserBean signUser = this.selectById(username);
      userSignature.setSignUserId(signUser.getId());
      userSignature.setSignUserName(signUser.getUserName());
      userSignature.setSignDate(LocalDate.now());
      userSignature.setSignTime(LocalDateTime.now());
      userSignature.setSignPicUrl(this.selectSignatureUrl(username));
      return userSignature;
   }

   public String selectSignatureUrl(String userId) {
      CoreFileBean signatureFile = (CoreFileBean)this.coreFileService.selectFirstByFilter(SearchFilter.instance().match((String)"TARGETID", this.getDao().getTable() + "$" + userId).filter(MatchPattern.EQ).match((String)"BIZCATEGORY", "SIGNATURE").filter(MatchPattern.EQ), new Order[]{Order.desc("ID")});
      return signatureFile == null ? null : this.fileManager.getDownloadUrl(signatureFile);
   }

   public String selectAvatarUrl(String userId) {
      CoreFileBean avatarFile = (CoreFileBean)this.coreFileService.selectFirstByFilter(SearchFilter.instance().match((String)"TARGETID", this.getDao().getTable() + "$" + userId).filter(MatchPattern.EQ).match((String)"SCOPE", FileScope.avatar).filter(MatchPattern.EQ), new Order[]{Order.desc("ID")});
      return avatarFile == null ? null : this.fileManager.getDownloadUrl(avatarFile);
   }

   protected RestHttpTextEntity changePassword(String userId, String newRawPassword) {
      CoreUserBean user = (CoreUserBean)BeanUtils.getPropertyListenerProxy((Object)this.selectById(userId));
      String encoedNewPassword = this.passwordEncoder.encode(newRawPassword);
      if (user.getPassword().equals(encoedNewPassword)) {
         ServletUtils.responseError("SINO.USER.PASSWORD.NEW_OLD_PASSWORD_SAME");
         return new RestHttpTextEntity(HttpStatus.FORBIDDEN);
      } else {
         String oldPassword = user.getPassword();
         this.passwordPolicyService.checkPassword(user.getId(), newRawPassword);
         user.setPassword(encoedNewPassword);
         user.setFailedLoginAttempts(0);
         user.setExpiredPasswordUses(0);
         user.setPasswordUpdatedTime(LocalDateTime.now());
         this.proxyInstance.update(user);
         CorePasswordHisBean passwordHis = new CorePasswordHisBean();
         passwordHis.setId(ApplicationContextHelper.getNextIdentity());
         passwordHis.setUserId(user.getId());
         passwordHis.setPassword(oldPassword);
         this.passwordHisService.getDao().insert(passwordHis);
         return new RestHttpTextEntity(HttpStatus.OK);
      }
   }

   private String getEncodedPassword(String rawPassword) {
      return this.passwordEncoder.encode(rawPassword);
   }

   private String getDefaultPassword(String userId) {
      CoreSystemConfigBean systemConfig = SystemConfigHelper.getSystemConfig();
      return "customPassword".equals(systemConfig.getInitPasswordPolicy()) && !StringUtils.isEmpty(systemConfig.getDefaultPassword()) ? this.getEncodedPassword(systemConfig.getDefaultPassword()) : this.getEncodedPassword(userId);
   }
}
