/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.module.mdm.org.service.impl;

import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.data.page.Page;
import net.sinodawn.framework.data.page.Pagination;
import net.sinodawn.framework.mybatis.mapper.MapperParameter;
import net.sinodawn.framework.mybatis.page.MybatisPageHelper;
import net.sinodawn.framework.mybatis.page.PageRowBounds;
import net.sinodawn.framework.restful.data.RestJsonWrapperBean;
import net.sinodawn.framework.support.PersistableHelper;
import net.sinodawn.framework.support.tree.TreeDescriptor;
import net.sinodawn.framework.support.tree.TreeHelper;
import net.sinodawn.framework.utils.CollectionUtils;
import net.sinodawn.framework.utils.ObjectUtils;
import net.sinodawn.module.mdm.org.bean.CoreOrgBean;
import net.sinodawn.module.mdm.org.bean.CoreOrgDTO;
import net.sinodawn.module.mdm.org.bean.CoreOrgUserBean;
import net.sinodawn.module.mdm.org.dao.CoreOrgDao;
import net.sinodawn.module.mdm.org.service.CoreOrgService;
import net.sinodawn.module.mdm.org.service.CoreOrgUserService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.annotation.Cacheable;
import org.springframework.context.annotation.Lazy;
import org.springframework.stereotype.Repository;
import org.springframework.transaction.annotation.Transactional;

import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.stream.Collectors;

@Repository
public class CoreOrgServiceImpl implements CoreOrgService {
   @Autowired
   private CoreOrgDao orgDao;
   @Autowired
   @Lazy
   private CoreOrgUserService orgUserService;
   @Autowired
   @Lazy
   private CoreOrgService proxyInstance;

   public CoreOrgDao getDao() {
      return this.orgDao;
   }

   @Cacheable(
      value = {"T_CORE_ORG"},
      key = "'ALL'"
   )
   public List<CoreOrgBean> selectAll() {
      return this.getDao().selectAll();
   }

   @Transactional
   public String insert(RestJsonWrapperBean wrapper) {
      CoreOrgBean coreOrg = (CoreOrgBean)wrapper.parseUnique(CoreOrgBean.class);
      coreOrg.setOrderNo(ApplicationContextHelper.getNextOrderNo());
      this.getDao().insert(coreOrg);
      return coreOrg.getId();
   }

   @Transactional
   public void insertUser(String id, RestJsonWrapperBean wrapper) {
      List<CoreOrgUserBean> orgUserList = wrapper.parse(CoreOrgUserBean.class);
      List<Long> idList = ApplicationContextHelper.getNextIdentityList(orgUserList.size());

      for(int index = 0; index < idList.size(); ++index) {
         ((CoreOrgUserBean)orgUserList.get(index)).setId((Long)idList.get(index));
      }

      this.orgUserService.getDao().insert(orgUserList);
   }

   @Transactional
   public void deleteUser(String id, RestJsonWrapperBean wrapper) {
      this.orgUserService.delete(wrapper);
   }

   public List<CoreOrgDTO> selectTree(MapperParameter parameter) {
      List<CoreOrgBean> orgList = this.getDao().selectTreeNodeList(parameter);
      TreeDescriptor<CoreOrgBean> descriptor = new TreeDescriptor("id", "parentId", "orgName", "orderNo");
      descriptor.setParseTreeNodeParentIdFunc((t) -> {
         String id = t.getId();
         return id.length() > 2 ? id.substring(0, id.length() - 2) : null;
      });
      return TreeHelper.parseTreeNode(orgList, descriptor, CoreOrgDTO.class);
   }

   public Page<CoreOrgBean> selectChoosablePagination(RestJsonWrapperBean wrapper) {
      MapperParameter parameter = wrapper.extractMapFilter();
      parameter.setOrgAuthority();
      parameter.setChoosableQueries();
      PageRowBounds rowBounds = wrapper.extractPageRowBounds();
      return this.selectPagination(parameter, rowBounds);
   }

   public Page<CoreOrgBean> selectRoleOrgList(Long roleId, RestJsonWrapperBean wrapper) {
      MapperParameter parameter = wrapper.extractMapFilter();
      parameter.put("roleId", roleId);
      PageRowBounds rowBounds = wrapper.extractPageRowBounds();
      Page<Map<String, Object>> mapPage = MybatisPageHelper.get(rowBounds, () -> {
         return this.getDao().selectRoleOrgList(parameter);
      });
      List<CoreOrgBean> itemList = (List)mapPage.getRows().stream().map((m) -> {
         return (CoreOrgBean)PersistableHelper.mapToPersistable(m, CoreOrgBean.class);
      }).collect(Collectors.toList());
      return new Pagination(mapPage, itemList);
   }

   public void insertOrgUser(String orgId, RestJsonWrapperBean wrapper) {
      CoreOrgBean org = (CoreOrgBean)this.getDao().selectById(orgId);
      List<CoreOrgUserBean> orgUserList = wrapper.parse(CoreOrgUserBean.class);
      List<Long> idList = ApplicationContextHelper.getNextIdentityList(orgUserList.size());

      for(int i = 0; i < idList.size(); ++i) {
         CoreOrgUserBean orgUser = (CoreOrgUserBean)orgUserList.get(i);
         orgUser.setId((Long)idList.get(i));
         orgUser.setOrgId(orgId);
         orgUser.setOrgName(org.getOrgName());
      }

      this.orgUserService.getDao().insert(orgUserList);
   }

   public List<CoreOrgBean> selectChildOrgList(String orgId) {
      List<CoreOrgBean> allOrgList = this.proxyInstance.selectAll();
      CoreOrgBean org = (CoreOrgBean)allOrgList.stream().filter((o) -> {
         return o.getId().equals(orgId);
      }).findFirst().orElse(null);
      return org == null ? CollectionUtils.emptyList() : this.selectChildOrgList(allOrgList, org);
   }

   public List<CoreOrgBean> selectParentOrgList(String orgId) {
      List<CoreOrgBean> allOrgList = this.proxyInstance.selectAll();
      CoreOrgBean org = (CoreOrgBean)allOrgList.stream().filter((o) -> {
         return o.getId().equals(orgId);
      }).findFirst().orElse(null);
      return org == null ? CollectionUtils.emptyList() : this.selectParentOrgList(allOrgList, org);
   }

   private List<CoreOrgBean> selectChildOrgList(List<CoreOrgBean> allList, CoreOrgBean currOrg) {
      List<CoreOrgBean> childOrgList = new ArrayList();
      Iterator var4 = allList.iterator();

      while(var4.hasNext()) {
         CoreOrgBean org = (CoreOrgBean)var4.next();
         if (ObjectUtils.equals(org.getParentId(), currOrg.getId())) {
            childOrgList.add(org);
            childOrgList.addAll(this.selectChildOrgList(allList, org));
         }
      }

      return childOrgList;
   }

   private List<CoreOrgBean> selectParentOrgList(List<CoreOrgBean> allList, CoreOrgBean currOrg) {
      List<CoreOrgBean> parentOrgList = new ArrayList();
      Iterator var4 = allList.iterator();

      while(var4.hasNext()) {
         CoreOrgBean org = (CoreOrgBean)var4.next();
         if (ObjectUtils.equals(org.getId(), currOrg.getParentId())) {
            parentOrgList.add(org);
            parentOrgList.addAll(this.selectParentOrgList(allList, org));
         }
      }

      return parentOrgList;
   }
}
