/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.utils;

import net.sinodawn.framework.exception.FileException;
import net.sinodawn.framework.exception.UnsupportedException;

import java.io.IOException;
import java.io.InputStream;
import java.io.UnsupportedEncodingException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.nio.charset.StandardCharsets;
import java.util.*;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

@SuppressWarnings({"unused", "rawtypes"})
public class StringUtils {
    public StringUtils() {
    }

    public static boolean isBlank(final String target) {
        return target == null || target.length() == 0 || trim(target).length() == 0;
    }

    public static boolean isEmpty(final String target) {
        return target == null || target.length() == 0;
    }

    public static boolean contains(String str, String searchPattern) {
        if (isEmpty(str)) {
            return false;
        } else {
            return isEmpty(searchPattern) || str.contains(searchPattern);
        }
    }

    public static boolean containsIgnoreCase(String str, String searchPattern) {
        if (isEmpty(str)) {
            return false;
        } else {
            return isEmpty(searchPattern) || str.toLowerCase().contains(searchPattern.toLowerCase());
        }
    }

    public static String capitalize(String str) {
        return changeFirstCharacterCase(str, true);
    }

    public static String uncapitalize(String str) {
        return changeFirstCharacterCase(str, false);
    }

    public static boolean startsWith(String str, String prefix) {
        if (str != null && prefix != null) {
            return prefix.length() <= str.length() && str.startsWith(prefix);
        } else {
            return false;
        }
    }

    public static boolean startsWithIgnoreCase(String str, String prefix) {
        if (str != null && prefix != null) {
            return prefix.length() <= str.length() && str.substring(0, prefix.length()).equalsIgnoreCase(prefix);
        } else {
            return false;
        }
    }

    public static boolean endsWith(String str, String suffix) {
        if (str != null && suffix != null) {
            return suffix.length() <= str.length() && str.endsWith(suffix);
        } else {
            return false;
        }
    }

    public static boolean endsWithIgnoreCase(String str, String suffix) {
        if (str != null && suffix != null) {
            return suffix.length() <= str.length() && str.substring(str.length() - suffix.length()).equalsIgnoreCase(suffix);
        } else {
            return false;
        }
    }

    public static String trim(final String target) {
        if (target == null) {
            return null;
        } else {
            char[] chars = target.toCharArray();
            int start = 0;
            int end = chars.length;
            int var5 = chars.length;

            for (char c : chars) {
                if (!Character.isWhitespace(c)) {
                    break;
                }

                ++start;
            }

            for(int i = chars.length - 1; i > 0 && Character.isWhitespace(chars[i]); --i) {
                --end;
            }

            return end > start ? String.copyValueOf(chars, start, end - start) : "";
        }
    }

    public static <V> String replace(String target, String searchPattern, V replacement) {
        if (!isEmpty(target) && !isEmpty(searchPattern)) {
            int start = 0;
            int end = target.indexOf(searchPattern, start);
            if (end == -1) {
                return target;
            } else {
                StringBuilder sb = new StringBuilder(target.substring(0, start));

                for(int searchPatternLen = searchPattern.length(); end != -1; end = target.indexOf(searchPattern, start)) {
                    sb.append(target, start, end).append(replacement);
                    start = end + searchPatternLen;
                }

                sb.append(target.substring(start));
                return sb.toString();
            }
        } else {
            return target;
        }
    }

    public static <V> String replaceFirst(String target, String searchPattern, V replacement) {
        if (!isEmpty(target) && !isEmpty(searchPattern)) {
            int pos = target.indexOf(searchPattern);
            return pos == -1 ? target : target.substring(0, pos) + replacement + target.substring(pos + searchPattern.length());
        } else {
            return target;
        }
    }

    public static <V> String replaceLast(String target, String searchPattern, V replacement) {
        if (!isEmpty(target) && !isEmpty(searchPattern)) {
            int pos = target.lastIndexOf(searchPattern);
            return pos == -1 ? target : target.substring(0, pos) + replacement + target.substring(pos + searchPattern.length());
        } else {
            return target;
        }
    }

    public static <V> String replaceIgnoreCase(String target, String searchPattern, V replacement) {
        if (!isEmpty(target) && !isEmpty(searchPattern)) {
            int targetLength = target.length();
            int searchPatternLength = searchPattern.length();
            StringBuilder sb = new StringBuilder();
            int start = 0;
            int lastEnd = 0;

            while(start + searchPatternLength <= targetLength) {
                String next = target.substring(start, start + searchPatternLength);
                if (next.equalsIgnoreCase(searchPattern)) {
                    sb.append(target, lastEnd, start).append(replacement);
                    lastEnd = start + searchPatternLength;
                    start = lastEnd;
                } else {
                    ++start;
                }
            }

            sb.append(target.substring(lastEnd));
            return sb.toString();
        } else {
            return target;
        }
    }

    public static String removeStart(String target, String searchPattern) {
        if (!isEmpty(target) && !isEmpty(searchPattern)) {
            return target.startsWith(searchPattern) ? target.substring(searchPattern.length()) : target;
        } else {
            return target;
        }
    }

    public static String removeStartIgnoreCase(String target, String searchPattern) {
        if (!isEmpty(target) && !isEmpty(searchPattern)) {
            return target.toLowerCase().startsWith(searchPattern.toLowerCase()) ? target.substring(searchPattern.length()) : target;
        } else {
            return target;
        }
    }

    public static String removeEnd(String target, String searchPattern) {
        if (!isEmpty(target) && !isEmpty(searchPattern)) {
            return target.endsWith(searchPattern) ? target.substring(0, target.length() - searchPattern.length()) : target;
        } else {
            return target;
        }
    }

    public static String removeEndIgnoreCase(String target, String searchPattern) {
        if (!isEmpty(target) && !isEmpty(searchPattern)) {
            return target.toLowerCase().endsWith(searchPattern.toLowerCase()) ? target.substring(0, target.length() - searchPattern.length()) : target;
        } else {
            return target;
        }
    }

    public static String trimAllWhitespace(String str) {
        if (isEmpty(str)) {
            return str;
        } else {
            int len = str.length();
            StringBuilder sb = new StringBuilder(str.length());

            for(int i = 0; i < len; ++i) {
                char c = str.charAt(i);
                if (!Character.isSpaceChar(c) && !Character.isWhitespace(c)) {
                    sb.append(c);
                }
            }

            return sb.toString();
        }
    }

    // TODO 待测试
    public static String replaceBetween(final String target, final String openTag, final String closeTag, final Map<String, ?> map) {
        if (map != null && !map.isEmpty()) {
            List<String> substrings = substringsBetween(target, openTag, closeTag);
            if (substrings.isEmpty()) {
                return target;
            } else {
                String replaceTarget = target;

                for (String substring : substrings) {
                    Object replacementValue = map.get(substring);
                    if (replacementValue != null) {
                        String replacement = ConvertUtils.convert(map.get(substring), String.class);
                        replaceTarget = replace(replaceTarget, openTag + substring + closeTag, replacement);
                    }
                }

                return replaceTarget;
            }
        } else {
            return target;
        }
    }

    public static String replaceBetweenSpan(final String target, final Map<String, Object> map) {
        if (map.isEmpty()) {
            return target;
        } else {
            String replaceTarget = target;

            Map.Entry entry;
            Matcher m;
            for(Iterator iterator = map.entrySet().iterator(); iterator.hasNext(); replaceTarget = m.replaceAll("<span id='" + entry.getKey() + "'>" + entry.getValue() + "</span>")) {
                entry = (Map.Entry)iterator.next();
                String regex = "<span\\s+id\\s*=\\s*(['\"])" + entry.getKey() + "(['\"])\\s*>(.|\\s)*?</span>";
                Pattern p = Pattern.compile(regex, Pattern.CASE_INSENSITIVE);
                m = p.matcher(replaceTarget);
            }

            return replaceTarget;
        }
    }

    public static String repeat(String source, int repeat) {
        if (repeat > 0 && !isEmpty(source)) {
            return source.repeat(repeat);
        } else {
            return "";
        }
    }

    public static int countOccurrencesOf(String str, String sub) {
        if (str != null && sub != null && str.length() != 0 && sub.length() != 0) {
            int count = 0;
            int subLength = sub.length();
            int i = 0;

            while(i <= str.length() - subLength) {
                if (sub.equals(str.substring(i, i + subLength))) {
                    ++count;
                    i += subLength;
                } else {
                    ++i;
                }
            }

            return count;
        } else {
            return 0;
        }
    }

    public static String[] split(String str, String delimiter) {
        if (str == null) {
            return new String[0];
        } else if (isEmpty(delimiter)) {
            return new String[]{str};
        } else {
            List<String> container = new ArrayList<>();
            int lastIndex = 0;

            int currentIndex;
            for(int delimiterLength = delimiter.length(); (currentIndex = str.indexOf(delimiter, lastIndex)) > -1; lastIndex = currentIndex + delimiterLength) {
                container.add(str.substring(lastIndex, currentIndex));
            }

            if (lastIndex <= str.length()) {
                container.add(str.substring(lastIndex));
            }

            return container.toArray(new String[0]);
        }
    }

    public static String[] regexSplit(String str, String regex) {
        if (str == null) {
            return new String[0];
        } else {
            return isEmpty(regex) ? new String[]{str} : str.split(regex);
        }
    }

    public static String decode(String str, String fromCharsetName, String toCharsetName) {
        try {
            return new String(str.getBytes(fromCharsetName), toCharsetName);
        } catch (UnsupportedEncodingException var4) {
            throw new UnsupportedException(var4);
        }
    }

    public static String unionArithmetic(String target1, String target2, int length) {
        if (!isEmpty(target1) && !isEmpty(target2)) {
            String temp = Integer.toBinaryString(NumberUtils.toBinaryNumber(target1) | NumberUtils.toBinaryNumber(target2));
            return leftPad(temp, length);
        } else {
            return "";
        }
    }

    public static String leftPad(String source, int length) {
        return pad(source, length, true);
    }

    public static String rightPad(String source, int length) {
        return pad(source, length, false);
    }

    public static List<String> substringsBetween(final String target, final String openTag, final String closeTag) {
        List<String> subStrings = new ArrayList<>();
        if (!isEmpty(target) && !isEmpty(openTag) && !isEmpty(closeTag)) {
            int strLen = target.length();
            if (strLen != 0) {
                int closeLen = closeTag.length();
                int openLen = openTag.length();

                int end;
                for (int pos = 0; pos < strLen - closeLen; pos = end + closeLen) {
                    int start = target.indexOf(openTag, pos);
                    if (start < 0) {
                        break;
                    }

                    start += openLen;
                    end = target.indexOf(closeTag, start);
                    if (end < 0) {
                        break;
                    }

                    subStrings.add(target.substring(start, end));
                }

            }
        }
        return subStrings;
    }

    public static String randomUUID(int size) {
        return randomUUID().substring(0, size);
    }

    public static String randomUUID() {
        UUID uuid = UUID.randomUUID();
        return replace(uuid.toString(), "-", "");
    }

    private static String changeFirstCharacterCase(String str, boolean capitalize) {
        if (str != null && str.length() != 0) {
            StringBuilder sb = new StringBuilder(str.length());
            if (capitalize) {
                sb.append(Character.toUpperCase(str.charAt(0)));
            } else {
                sb.append(Character.toLowerCase(str.charAt(0)));
            }

            sb.append(str.substring(1));
            return sb.toString();
        } else {
            return str;
        }
    }

    private static String pad(String source, int length, boolean leftPosition) {
        if (isEmpty(source)) {
            return repeat("0", length);
        } else if (length > 0 && source.length() < length) {
            return leftPosition ? repeat("0", length - source.length()) + source : source + repeat("0", length - source.length());
        } else {
            return source;
        }
    }

    public static boolean equalsIgnoreCase(String v1, String v2) {
        if (v1 == null) {
            return v2 == null;
        } else if (v2 == null) {
            return false;
        } else {
            return v1.equalsIgnoreCase(v2);
        }
    }

    public static boolean equals(String v1, String v2) {
        if (v1 == null) {
            return v2 == null;
        } else if (v2 == null) {
            return false;
        } else {
            return v1.equals(v2);
        }
    }

    public static String encodeURL(String url) {
        if (url.contains(" ")) {
            String[] strs = split(url, " ");
            StringBuilder sb = new StringBuilder();
            int var4 = strs.length;

            for (String s : strs) {
                if (sb.length() > 0) {
                    sb.append("%20");
                }

                if (!isEmpty(s)) {
                    sb.append(URLEncoder.encode(s, StandardCharsets.UTF_8));
                }
            }

            return sb.toString();
        } else {
            return URLEncoder.encode(url, StandardCharsets.UTF_8);
        }
    }

    public static String decodeURL(String url) {
        return URLDecoder.decode(url, StandardCharsets.UTF_8);
    }

    public static String reverse(String str) {
        return str == null ? null : (new StringBuilder(str)).reverse().toString();
    }

    public static String lowerCase(String str) {
        return str == null ? null : str.toLowerCase();
    }

    public static Object lowerCaseIfPossible(Object value) {
        return value instanceof String ? ((String)value).toLowerCase() : value;
    }

    public static String upperCase(String value) {
        return value != null ? value.toUpperCase() : null;
    }

    public static String join(Collection<?> itemList, String separator) {
        StringBuilder sb = new StringBuilder();

        Object item;
        for(Iterator var3 = itemList.iterator(); var3.hasNext(); sb.append(ConvertUtils.convert(item, String.class, ""))) {
            item = var3.next();
            if (sb.length() > 0) {
                sb.append(separator);
            }
        }

        return sb.toString();
    }

    public static <T> String join(T[] itemList, String separator) {
        StringBuilder sb = new StringBuilder();
        int var4 = itemList.length;

        for(int var5 = 0; var5 < var4; ++var5) {
            Object item = ((Object[]) itemList)[var5];
            if (sb.length() > 0) {
                sb.append(separator);
            }

            sb.append(ConvertUtils.convert(item, String.class, ""));
        }

        return sb.toString();
    }

    public static final String read(InputStream is) {
        try {
            byte[] bytes = null;
            byte[] buf = new byte[2048];

            int i;
            while((i = is.read(buf)) != -1) {
                if (bytes == null) {
                    bytes = new byte[i];
                    System.arraycopy(buf, 0, bytes, 0, i);
                } else {
                    byte[] temp = new byte[bytes.length + i];
                    System.arraycopy(bytes, 0, temp, 0, bytes.length);
                    System.arraycopy(buf, 0, temp, bytes.length, i);
                    bytes = temp;
                }
            }

            if (bytes == null) {
                return "";
            } else {
                return new String(bytes, EncodingUtils.detect(bytes));
            }
        } catch (IOException var5) {
            throw new FileException(var5);
        }
    }

    public static String abbreviate(String text) {
        if (isBlank(text)) {
            return text;
        } else {
            int length = text.length();
            return length > 240 ? text.substring(0, 240) + "..." : text;
        }
    }

    public static boolean containsNumber(String text) {
        return !isBlank(text) && text.matches(".*\\d+.*");
    }

    public static boolean containsLowercaseLetter(String text) {
        return !isBlank(text) && text.matches(".*[a-z]+.*");
    }

    public static boolean containsCapitalLetter(String text) {
        return !isBlank(text) && text.matches(".*[A-Z]+.*");
    }

    public static boolean containsSpecialCharacter(String text) {
        return !isBlank(text) && text.matches(".*[`~!@#$%^&*()_+\\-={}\\[\\]|:;\"'<>,.?/\\\\]+.*");
    }
}
