/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.utils;


import com.blueconic.browscap.Capabilities;
import com.blueconic.browscap.ParseException;
import com.blueconic.browscap.UserAgentParser;
import com.blueconic.browscap.UserAgentService;
import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.context.LocalContextHelper;
import net.sinodawn.framework.exception.UnexpectedException;
import net.sinodawn.framework.exception.core.HttpErrorMessage;
import org.springframework.web.context.ContextLoader;
import org.springframework.web.context.WebApplicationContext;
import org.springframework.web.context.request.RequestContextHolder;
import org.springframework.web.context.request.ServletRequestAttributes;

import javax.servlet.ServletContext;
import javax.servlet.ServletResponse;
import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.servlet.http.HttpSession;
import java.io.IOException;
import java.io.PrintWriter;
import java.util.HashMap;
import java.util.Map;

public abstract class ServletUtils {
    private static final UserAgentParser userAgentParser;
    private static String contextPath;

    public ServletUtils() {
    }

    public static Capabilities getCurrentRequestCapabilities() {
        HttpServletRequest request = getCurrentRequest();
        return request == null ? null : userAgentParser.parse(request.getHeader("user-agent"));
    }

    public static String getRemoteIp(HttpServletRequest request) {
        if (request == null) {
            return "";
        } else {
            String ip = request.getHeader("x-forwarded-for");
            if (isBlankOrUnknownIp(ip)) {
                ip = request.getHeader("Proxy-Client-IP");
            }

            if (isBlankOrUnknownIp(ip)) {
                ip = request.getHeader("WL-Proxy-Client-IP");
            }

            if (isBlankOrUnknownIp(ip)) {
                ip = request.getHeader("HTTP_CLIENT_IP");
            }

            if (isBlankOrUnknownIp(ip)) {
                ip = request.getHeader("HTTP_X_FORWARDED_FOR");
            }

            if (isBlankOrUnknownIp(ip)) {
                ip = request.getRemoteAddr();
            }

            return ip;
        }
    }

    public static HttpServletRequest getCurrentRequest() {
        ServletRequestAttributes requestAttributes = (ServletRequestAttributes) RequestContextHolder.getRequestAttributes();
        if (requestAttributes != null) {
            HttpServletRequest request = requestAttributes.getRequest();
            if (request != null) {
                return request;
            }
        }

        return LocalContextHelper.getHttpServletRequest();
    }

    public static HttpServletResponse getCurrentResponse() {
        ServletRequestAttributes requestAttributes = (ServletRequestAttributes)RequestContextHolder.getRequestAttributes();
        if (requestAttributes != null) {
            HttpServletResponse response = requestAttributes.getResponse();
            if (response != null) {
                return response;
            }
        }

        return LocalContextHelper.getHttpServletResponse();
    }

    public static HttpSession getCurrentSession() {
        HttpServletRequest request = getCurrentRequest();
        return request == null ? null : request.getSession(false);
    }

    public static String getCurrentRawSessionId() {
        HttpSession session = getCurrentSession();
        return session == null ? null : getRawSessionId(session.getId());
    }

    public static ServletContext getServletContext() {
        WebApplicationContext webApplicationContext = ContextLoader.getCurrentWebApplicationContext();
        return webApplicationContext.getServletContext();
    }

    public static String getServletRootPath() {
        return getServletContext().getRealPath("/");
    }

    public static String getRawSessionId(final String sessionId) {
        return sessionId.contains("!") ? sessionId.substring(0, sessionId.indexOf("!")) : sessionId;
    }

    public static Map<String, String> getCurrentRequestParamMap() {
        Map<String, String> map = new HashMap<>();
        HttpServletRequest request = getCurrentRequest();
        if (request != null) {
            request.getParameterMap().forEach((k, v) -> {
                String var10000 = (String)map.put(k, v[0]);
            });
        }

        return map;
    }

    /** @deprecated */
    @Deprecated
    public static void responseError(String error) {
        responseError(400, error);
    }

    public static void responseError(HttpErrorMessage errorMessage) {
        responseError(errorMessage.getStatus(), errorMessage.toString());
    }

    public static void responseError(int httpStatus, String error) {
        HttpServletResponse response = getCurrentResponse();
        if (response != null && !response.isCommitted()) {
            response.reset();
            response.setStatus(httpStatus);
            response.setCharacterEncoding("UTF-8");

            try {
                PrintWriter pw = response.getWriter();
                Throwable var4 = null;

                try {
                    pw.print(error);
                    pw.flush();
                } catch (Throwable var14) {
                    var4 = var14;
                    throw var14;
                } finally {
                    if (pw != null) {
                        if (var4 != null) {
                            try {
                                pw.close();
                            } catch (Throwable var13) {
                                var4.addSuppressed(var13);
                            }
                        } else {
                            pw.close();
                        }
                    }

                }
            } catch (IOException var16) {
                throw new UnexpectedException(var16);
            }
        }

    }

    public static void responseMessage(ServletResponse response, String message) {
        if (response != null && !response.isCommitted()) {
            response.reset();
            response.setCharacterEncoding("UTF-8");

            try {
                PrintWriter writer = response.getWriter();
                Throwable var3 = null;

                try {
                    writer.write(message);
                    writer.flush();
                } catch (Throwable var13) {
                    var3 = var13;
                    throw var13;
                } finally {
                    if (writer != null) {
                        if (var3 != null) {
                            try {
                                writer.close();
                            } catch (Throwable var12) {
                                var3.addSuppressed(var12);
                            }
                        } else {
                            writer.close();
                        }
                    }

                }
            } catch (IOException var15) {
                throw new UnexpectedException(var15);
            }
        }

    }

    public static void responseMessage(ServletResponse response, int httpStatus, String message) {
        if (response != null && !response.isCommitted()) {
            response.reset();
            ((HttpServletResponse)response).setStatus(httpStatus);
            response.setCharacterEncoding("UTF-8");

            try {
                PrintWriter writer = response.getWriter();
                Throwable var4 = null;

                try {
                    writer.write(message);
                    writer.flush();
                } catch (Throwable var14) {
                    var4 = var14;
                    throw var14;
                } finally {
                    if (writer != null) {
                        if (var4 != null) {
                            try {
                                writer.close();
                            } catch (Throwable var13) {
                                var4.addSuppressed(var13);
                            }
                        } else {
                            writer.close();
                        }
                    }

                }
            } catch (IOException var16) {
                throw new UnexpectedException(var16);
            }
        }

    }

    public static String getRequestUri() {
        return getRequestUri(getCurrentRequest());
    }

    public static String getContextPath() {
        if (contextPath == null) {
            contextPath = ApplicationContextHelper.getEnvironment().getProperty("server.servlet.context-path", "");
        }

        return contextPath;
    }

    public static String getRequestUri(HttpServletRequest request) {
        if (request == null) {
            return "";
        } else {
            return StringUtils.isBlank(getContextPath()) ? request.getRequestURI() : StringUtils.removeStart(request.getRequestURI(), getContextPath());
        }
    }

    public static String getRequestUrl() {
        return getRequestUrl(getCurrentRequest());
    }

    public static String getRequestUrl(HttpServletRequest request) {
        if (request == null) {
            return "";
        } else {
            return StringUtils.isBlank(getContextPath()) ? request.getRequestURL().toString() : StringUtils.removeStart(request.getRequestURL().toString(), getContextPath());
        }
    }

    public static String getUrl(String url, String replativeUrl) {
        StringBuilder sb = new StringBuilder();
        if (!StringUtils.isEmpty(url)) {
            sb.append(StringUtils.removeEnd(url, "/"));
        }

        return sb.append(replativeUrl).toString();
    }

    public static void addCookie(HttpServletRequest request, HttpServletResponse response, String cookieName, String cookieValue) {
        addCookie(request, response, cookieName, cookieValue, -1);
    }

    public static void addCookie(HttpServletRequest request, HttpServletResponse response, String cookieName, String cookieValue, int expirySeconds) {
        Cookie cookie = new Cookie(cookieName, cookieValue);
        cookie.setPath(StringUtils.isBlank(request.getContextPath()) ? "/" : request.getContextPath());
        cookie.setSecure(false);
        cookie.setHttpOnly(true);
        cookie.setMaxAge(expirySeconds);
        response.addCookie(cookie);
    }

    public static Cookie getCookie(HttpServletRequest request, String cookieName) {
        Cookie[] var2 = request.getCookies();

        for (Cookie cookie : var2) {
            if (cookieName.equals(cookie.getName())) {
                return cookie;
            }
        }

        return null;
    }

    private static boolean isBlankOrUnknownIp(final String ip) {
        return StringUtils.isEmpty(ip) || "unknown".equalsIgnoreCase(ip) || StringUtils.split(ip, ".").length != 4;
    }

    static {
        try {
            userAgentParser = (new UserAgentService()).loadParser();
        } catch (ParseException | IOException var1) {
            throw new UnexpectedException(var1);
        }
    }
}
