/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.utils;

import net.sinodawn.framework.exception.InvalidDataException;
import net.sinodawn.framework.exception.ReflectionException;

import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.lang.reflect.Type;
import java.util.*;

@SuppressWarnings({"RawTypeCanBeGeneric", "rawtypes", "unused", "deprecation", "unchecked"})
public class ReflectionUtils {
    private static final Map<String, Annotation> CLASS_ANNOTATION_CONTAINER = new HashMap<>();
    private static final Map<String, Annotation> METHOD_ANNOTATION_CONTAINER = new HashMap<>();

    public ReflectionUtils() {
    }

    public static boolean isReadMethod(Method method) {
        String methodName = method.getName();
        return (StringUtils.startsWith(methodName, "CGLIB$get") || StringUtils.startsWith(methodName, "get") || StringUtils.startsWith(methodName, "CGLIB$is") || StringUtils.startsWith(methodName, "is")) && method.getParameterCount() == 0;
    }

    public static boolean isWriteMethod(Method method) {
        String methodName = method.getName();
        return (StringUtils.startsWith(methodName, "CGLIB$set") || StringUtils.startsWith(methodName, "set")) && method.getParameterCount() == 1;
    }

    public static Method findReadMethod(Class<?> clazz, String propertyName) {
        return findMethod(clazz, "get" + StringUtils.capitalize(propertyName));
    }

    public static Method findWriteMethod(Class<?> clazz, String propertyName) {
        return Arrays.stream(clazz.getMethods()).filter((m) -> m.getName().equals("set" + StringUtils.capitalize(propertyName)) && m.getParameterCount() == 1).findFirst().orElse(null);
    }

    public static Method findMethodByName(Class<?> clazz, String name) {
        for(Class searchClazz = clazz; searchClazz != null; searchClazz = searchClazz.getSuperclass()) {
            Method method = Arrays.stream(searchClazz.getDeclaredMethods()).filter((m) -> m.getName().equals(name)).findFirst().orElse(null);
            if (method != null) {
                return method;
            }
        }

        Class[] var8 = clazz != null ? clazz.getInterfaces() : new Class[0];
        int var4 = var8.length;

        for (Class<?> iClazz : var8) {
            Method method = Arrays.stream(iClazz.getMethods()).filter((m) -> m.getName().equals(name)).findFirst().orElse(null);
            if (method != null) {
                return method;
            }
        }

        return null;
    }

    public static List<Method> getMethodList(Class<?> clazz, int... modifiers) {
        List<Method> methodList = new ArrayList<>();

        for(Class searchClazz = clazz; searchClazz != null; searchClazz = searchClazz.getSuperclass()) {
            Method[] methods = searchClazz.getDeclaredMethods();
            int var6 = methods.length;

            for (Method method : methods) {
                if (methodList.stream().noneMatch((f) -> f.equals(method)) && (modifiers == null || Arrays.stream(modifiers).allMatch((m) -> (method.getModifiers() & m) != 0))) {
                    methodList.add(method);
                }
            }
        }

        return methodList;
    }

    public static List<Method> getAnnotatedMethodList(Class<?> clazz, Class<? extends Annotation> annotationClazz) {
        List<Method> methodList = new ArrayList<>();

        Class searchClazz;
        int var6;
        for(searchClazz = clazz; searchClazz != null; searchClazz = searchClazz.getSuperclass()) {
            Method[] methods = searchClazz.getDeclaredMethods();
            var6 = methods.length;

            for(int var7 = 0; var7 < var6; ++var7) {
                Method method = methods[var7];
                if (method.isAnnotationPresent(annotationClazz) && methodList.stream().noneMatch((m) -> sameNameAndParameterTypes(m, method))) {
                    methodList.add(method);
                }
            }
        }

        for(searchClazz = clazz; searchClazz != null; searchClazz = searchClazz.getSuperclass()) {
            Class[] var13 = searchClazz.getInterfaces();
            int var14 = var13.length;

            for(var6 = 0; var6 < var14; ++var6) {
                Class<?> iClazz = var13[var6];
                Method[] interfaceMethods = iClazz.getMethods();
                int var10 = interfaceMethods.length;

                for (Method interfaceMethod : interfaceMethods) {
                    if (interfaceMethod.isAnnotationPresent(annotationClazz) && methodList.stream().noneMatch((m) -> sameNameAndParameterTypes(m, interfaceMethod))) {
                        methodList.add(interfaceMethod);
                    }
                }
            }
        }

        return methodList;
    }

    public static <A extends Annotation> A getMethodAnnotation(Class<?> clazz, Method method, Class<A> annotationClazz) {
        for(Class searchClazz = clazz; searchClazz != null; searchClazz = searchClazz.getSuperclass()) {
            Method matchMethod = Arrays.stream(searchClazz.getDeclaredMethods()).filter((m) -> sameNameAndParameterTypes(m, method) && m.isAnnotationPresent(annotationClazz)).findFirst().orElse(null);
            if (matchMethod != null) {
                return matchMethod.getAnnotation(annotationClazz);
            }
        }

        Iterator var7 = getAllInterfaces(clazz).iterator();

        Method matchMethod;
        do {
            if (!var7.hasNext()) {
                return null;
            }

            Class<?> iClazz = (Class)var7.next();
            matchMethod = Arrays.stream(iClazz.getDeclaredMethods()).filter((m) -> sameNameAndParameterTypes(m, method) && m.isAnnotationPresent(annotationClazz)).findFirst().orElse(null);
        } while(matchMethod == null);

        return matchMethod.getAnnotation(annotationClazz);
    }

    public static List<Class<?>> getAllInterfaces(Class<?> clazz) {
        if (clazz != null && !Object.class.equals(clazz)) {
            List<Class<?>> interfaceList = new ArrayList<>();

            for(Class searchClazz = clazz; searchClazz != null; searchClazz = searchClazz.getSuperclass()) {
                Class[] var3 = searchClazz.getInterfaces();
                int var4 = var3.length;

                for (Class<?> iClass : var3) {
                    interfaceList.add(iClass);
                    interfaceList.addAll(getAllInterfaces(iClass));
                }
            }

            return interfaceList;
        } else {
            return CollectionUtils.emptyList();
        }
    }

    public static Method findMethod(Class<?> clazz, String name) {
        return findMethod(clazz, name, null);
    }

    /**
     * 根据Class方法名获取对应方法
     * @param clazz Class
     * @param name 方法名
     * @param parameterTypes Class...
     * @return Method
     */
    public static Method findMethod(Class<?> clazz, String name, Class<?>... parameterTypes) {
        for(Class searchClazz = clazz; searchClazz != null; searchClazz = searchClazz.getSuperclass()) {
            Method method = Arrays.stream(searchClazz.getDeclaredMethods())
                    .filter( (m) -> m.getName().equalsIgnoreCase(name)
                            && (parameterTypes == null || Arrays.equals(m.getParameterTypes(), parameterTypes))
                    ).findFirst().orElse(null);
            if (method != null) {
                return method;
            }
        }

        Class[] var9 = clazz != null ? clazz.getInterfaces() : new Class[0];
        int var5 = var9.length;

        for (Class<?> iClazz : var9) {
            Method method = Arrays.stream(iClazz.getMethods()).filter((m) -> m.getName().equalsIgnoreCase(name) && (parameterTypes == null || Arrays.equals(m.getParameterTypes(), parameterTypes))).findFirst().orElse(null);
            if (method != null) {
                return method;
            }
        }

        return null;
    }

    public static Method findMethodWithReturnType(Class<?> clazz, String name, Type returnType) {
        for(Class searchClazz = clazz; searchClazz != null; searchClazz = searchClazz.getSuperclass()) {
            Method method = Arrays.stream(searchClazz.getDeclaredMethods()).filter((m) -> m.getName().equals(name) && m.getReturnType().equals(returnType)).findFirst().orElse(null);
            if (method != null) {
                return method;
            }
        }

        Class[] var9 = clazz != null ? clazz.getInterfaces() : new Class[0];
        int var5 = var9.length;

        for (Class<?> iClazz : var9) {
            Method method = Arrays.stream(iClazz.getMethods()).filter((m) -> m.getName().equals(name) && m.getReturnType().equals(returnType)).findFirst().orElse(null);
            if (method != null) {
                return method;
            }
        }

        return null;
    }

    /**
     * 调用反射方法
     * @param method Method
     * @param target Object
     * @param args arguments
     * @return Object for result
     */
    public static Object invokeMethod(Method method, Object target, Object... args) {
        boolean accessible = method.isAccessible();

        Object var4;
        try {
            if (!accessible) {
                method.setAccessible(true);
            }

            var4 = method.invoke(target, args);
        } catch (IllegalArgumentException | InvocationTargetException | IllegalAccessException var8) {
            throw new ReflectionException(var8);
        } finally {
            if (!accessible) {
                method.setAccessible(false);
            }

        }

        return var4;
    }

    public static void invokeWriteMethod(Object target, String propertyName, Object arg) {
        Method writeMethod = findWriteMethod(target.getClass(), propertyName);
        if (writeMethod != null) {
            invokeMethod(writeMethod, target, arg);
        } else {
            throw new InvalidDataException("No [" + propertyName + "] write method of [" + target.getClass().getName() + "].");
        }
    }

    public static Object invokeReadMethod(Object target, String propertyName) {
        Method readMethod = findReadMethod(target.getClass(), propertyName);
        return invokeMethod(readMethod, target);
    }

    public static List<Field> getFiledList(Class<?> clazz, int... modifiers) {
        List<Field> fieldList = new ArrayList<>();

        for(Class searchType = clazz; Object.class != searchType && searchType != null; searchType = searchType.getSuperclass()) {
            Field[] fields = searchType.getDeclaredFields();
            int var6 = fields.length;

            for (Field field : fields) {
                if (fieldList.stream().noneMatch((f) -> f.getName().equals(field.getName())) && (modifiers == null || Arrays.stream(modifiers).allMatch((m) -> (field.getModifiers() & m) != 0))) {
                    fieldList.add(field);
                }
            }
        }

        return fieldList;
    }

    public static Field findField(Class<?> clazz, String name) {
        return findField(clazz, name, null);
    }

    public static Field findField(Class<?> clazz, String name, Class<?> type) {
        for(Class searchType = clazz; Object.class != searchType && searchType != null; searchType = searchType.getSuperclass()) {
            Field field = Arrays.stream(searchType.getDeclaredFields()).filter((f) -> f.getName().equalsIgnoreCase(name) && (type == null || f.getType().equals(type))).findFirst().orElse(null);
            if (field != null) {
                return field;
            }
        }

        return null;
    }

    public static Field findFieldOfReadMethod(Method readMethod) {
        if (!isReadMethod(readMethod)) {
            throw new InvalidDataException("Method [" + readMethod.getName() + "] is not a read method.");
        } else {
            Class<?> clazz = readMethod.getDeclaringClass();
            if (ClassUtils.isCglibProxyClass(clazz)) {
                clazz = clazz.getSuperclass();
            }

            String methodName = readMethod.getName();
            String propertyName;
            if (StringUtils.startsWith(methodName, "is")) {
                propertyName = methodName.substring(2);
            } else {
                propertyName = methodName.substring(3);
            }

            return findField(clazz, StringUtils.uncapitalize(propertyName), readMethod.getReturnType());
        }
    }

    public static Field findFieldOfWriteMethod(Method writeMethod) {
        if (!isWriteMethod(writeMethod)) {
            throw new InvalidDataException("Method [" + writeMethod.getName() + "] is not a write method.");
        } else {
            Class<?> clazz = writeMethod.getDeclaringClass();
            if (ClassUtils.isCglibProxyClass(clazz)) {
                clazz = clazz.getSuperclass();
            }

            String propertyName = writeMethod.getName().substring(3);
            return findField(clazz, StringUtils.uncapitalize(propertyName), writeMethod.getParameterTypes()[0]);
        }
    }

    public static Object getFieldValue(Object target, String propertyName) {
        Field property = findField(target.getClass(), propertyName);
        if (property == null) {
            return null;
        } else {
            Object value;

            try {
                if (property.isAccessible()) {
                    value = property.get(target);
                } else {
                    property.setAccessible(true);
                    value = property.get(target);
                    property.setAccessible(false);
                }

                return value;
            } catch (IllegalAccessException | IllegalArgumentException var5) {
                throw new ReflectionException(var5);
            }
        }
    }

    public static void setFieldValue(Object target, String propertyName, Object value) {
        Field property = findField(target.getClass(), propertyName);
        if (property != null) {
            try {
                if (property.isAccessible()) {
                    property.set(target, value);
                } else {
                    property.setAccessible(true);
                    property.set(target, value);
                    property.setAccessible(false);
                }

            } catch (IllegalAccessException | IllegalArgumentException var5) {
                throw new ReflectionException(var5);
            }
        }
    }

    public static <T extends Annotation> T getAnnotation(Class<?> targetClass, Class<T> annotationClass) {
        Class<?> clazz = ClassUtils.getRawType(targetClass);
        String key = clazz.getName() + "-" + annotationClass.getName();
        T cacheAnnotation = (T) CLASS_ANNOTATION_CONTAINER.get(key);
        if (cacheAnnotation == null) {
            for(Class searchClazz = clazz; searchClazz != null; searchClazz = searchClazz.getSuperclass()) {
                cacheAnnotation = (T) searchClazz.getAnnotation(annotationClass);
                if (cacheAnnotation != null) {
                    break;
                }

                Class[] var6 = searchClazz.getInterfaces();
                int var7 = var6.length;

                for (Class<?> iClazz : var6) {
                    T interfaceAnnotation = iClazz.getAnnotation(annotationClass);
                    if (interfaceAnnotation != null) {
                        return interfaceAnnotation;
                    }
                }
            }

            if (cacheAnnotation != null) {
                CLASS_ANNOTATION_CONTAINER.put(key, cacheAnnotation);
            }
        }

        return cacheAnnotation;
    }

    public static <T extends Annotation> T getAnnotation(Method method, Class<T> annotationClass) {
        Class<?> clazz = ClassUtils.getRawType(method.getDeclaringClass());
        String key = method.toGenericString() + "-" + annotationClass.getName();
        T cacheAnnotation = (T) METHOD_ANNOTATION_CONTAINER.get(key);
        if (cacheAnnotation == null) {
            for(Class searchClazz = clazz; searchClazz != null; searchClazz = searchClazz.getSuperclass()) {
                try {
                    Method searchMethod = searchClazz.getMethod(method.getName(), method.getParameterTypes());
                    cacheAnnotation = searchMethod.getAnnotation(annotationClass);
                    if (cacheAnnotation != null) {
                        break;
                    }

                    Class[] var7 = searchClazz.getInterfaces();
                    int var8 = var7.length;

                    for (Class<?> iClazz : var7) {
                        Method searchInterfaceMethod = iClazz.getMethod(method.getName(), method.getParameterTypes());
                        cacheAnnotation = searchInterfaceMethod.getAnnotation(annotationClass);
                        if (cacheAnnotation != null) {
                            break;
                        }
                    }
                } catch (SecurityException | NoSuchMethodException ignored) {
                }

                if (cacheAnnotation != null) {
                    break;
                }
            }

            if (cacheAnnotation != null) {
                METHOD_ANNOTATION_CONTAINER.put(key, cacheAnnotation);
            }
        }

        return cacheAnnotation;
    }

    public static boolean isValidBeanProperty(Class<?> type, Field property) {
        return !property.getName().equalsIgnoreCase("readonly") && findReadMethod(type, property.getName()) != null && findWriteMethod(type, property.getName()) != null;
    }

    private static boolean sameNameAndParameterTypes(Method m1, Method m2) {
        if (m1 == null) {
            return m2 == null;
        } else if (m2 == null) {
            return false;
        } else if (!m1.getName().equals(m2.getName())) {
            return false;
        } else {
            Class<?>[] types1 = m1.getParameterTypes();
            Class<?>[] types2 = m2.getParameterTypes();
            if (types1.length != types2.length) {
                return false;
            } else {
                int i = 0;

                for(int j = types1.length; i < j; ++i) {
                    if (!types1[i].equals(types2[i])) {
                        return false;
                    }
                }

                return true;
            }
        }
    }
}
