/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.utils;

import net.sinodawn.framework.exception.FileException;
import net.sinodawn.framework.io.text.FileToTextExtractor;
import net.sinodawn.framework.io.text.FileToTextExtractorFactory;
import org.apache.commons.fileupload.disk.DiskFileItem;
import org.springframework.web.multipart.MultipartFile;
import org.springframework.web.multipart.commons.CommonsMultipartFile;

import java.io.*;
import java.net.URL;
import java.nio.charset.Charset;
import java.nio.file.*;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;

public final class FileUtils {
   public static final void makeDirs(Path regularFilePath) {
      if (regularFilePath != null) {
         Path dirPath = regularFilePath.getParent();
         if (dirPath != null && !Files.exists(dirPath, new LinkOption[0])) {
            try {
               Files.createDirectories(dirPath);
            } catch (IOException var3) {
               throw new FileException(var3);
            }
         }

      }
   }

   public static final void write(Path regularFilePath, String content) {
      makeDirs(regularFilePath);

      try {
         BufferedWriter bufferedWriter = Files.newBufferedWriter(regularFilePath);
         Throwable var3 = null;

         try {
            bufferedWriter.write(content);
         } catch (Throwable var13) {
            var3 = var13;
            throw var13;
         } finally {
            if (bufferedWriter != null) {
               if (var3 != null) {
                  try {
                     bufferedWriter.close();
                  } catch (Throwable var12) {
                     var3.addSuppressed(var12);
                  }
               } else {
                  bufferedWriter.close();
               }
            }

         }

      } catch (IOException var15) {
         throw new FileException(var15);
      }
   }

   public static final void write(Path regularFilePath, InputStream is) {
      makeDirs(regularFilePath);

      try {
         OutputStream os = Files.newOutputStream(regularFilePath);
         Throwable var3 = null;

         try {
            byte[] buffer = new byte[4096];
            boolean var5 = false;

            int size;
            while((size = is.read(buffer)) != -1) {
               os.write(buffer, 0, size);
            }
         } catch (Throwable var14) {
            var3 = var14;
            throw var14;
         } finally {
            if (os != null) {
               if (var3 != null) {
                  try {
                     os.close();
                  } catch (Throwable var13) {
                     var3.addSuppressed(var13);
                  }
               } else {
                  os.close();
               }
            }

         }

      } catch (IOException var16) {
         throw new FileException(var16);
      }
   }

   public static final void write(File file, String content) {
      try {
         Writer writer = new BufferedWriter(new OutputStreamWriter(new FileOutputStream(file), "UTF-8"));
         Throwable var3 = null;

         try {
            writer.write(content);
            writer.flush();
         } catch (Throwable var13) {
            var3 = var13;
            throw var13;
         } finally {
            if (writer != null) {
               if (var3 != null) {
                  try {
                     writer.close();
                  } catch (Throwable var12) {
                     var3.addSuppressed(var12);
                  }
               } else {
                  writer.close();
               }
            }

         }

      } catch (IOException var15) {
         throw new FileException(var15);
      }
   }

   public static final void write(File file, InputStream is) {
      try {
         FileOutputStream fos = new FileOutputStream(file);
         Throwable var3 = null;

         try {
            byte[] buffer = new byte[4096048];
            boolean var5 = false;

            int size;
            while((size = is.read(buffer)) != -1) {
               fos.write(buffer, 0, size);
            }
         } catch (Throwable var14) {
            var3 = var14;
            throw var14;
         } finally {
            if (fos != null) {
               if (var3 != null) {
                  try {
                     fos.close();
                  } catch (Throwable var13) {
                     var3.addSuppressed(var13);
                  }
               } else {
                  fos.close();
               }
            }

         }

      } catch (IOException var16) {
         throw new FileException(var16);
      }
   }

   public static final List<String> readAllLines(File file) {
      try {
         FileInputStream fis = new FileInputStream(file);
         Throwable var2 = null;

         List var4;
         try {
            String content = StringUtils.read(fis);
            var4 = Arrays.asList(StringUtils.regexSplit(content, "(\n|\r)+"));
         } catch (Throwable var14) {
            var2 = var14;
            throw var14;
         } finally {
            if (fis != null) {
               if (var2 != null) {
                  try {
                     fis.close();
                  } catch (Throwable var13) {
                     var2.addSuppressed(var13);
                  }
               } else {
                  fis.close();
               }
            }

         }

         return var4;
      } catch (IOException var16) {
         throw new FileException(var16);
      }
   }

   public static final File createTempFile() {
      try {
         return File.createTempFile(StringUtils.randomUUID(), "");
      } catch (IOException var1) {
         throw new FileException(var1);
      }
   }

   public static final File createTempFile(String extension) {
      try {
         if (extension != null && !StringUtils.startsWith(extension, ".")) {
            extension = "." + extension;
         }

         return File.createTempFile(StringUtils.randomUUID(), extension);
      } catch (IOException var2) {
         throw new FileException(var2);
      }
   }

   public static final void createFile(Path path) {
      if (!Files.exists(path, new LinkOption[0])) {
         makeDirs(path);
         if (!Files.exists(path, new LinkOption[0])) {
            try {
               Files.createFile(path);
            } catch (IOException var2) {
               throw new FileException(var2);
            }
         }

      }
   }

   public static final void createFile(File file) {
      if (!file.exists()) {
         file.getParentFile().mkdirs();
         if (file.exists()) {
            try {
               file.createNewFile();
            } catch (IOException var2) {
               throw new FileException(var2);
            }
         }

      }
   }

   public static final String getFileExtension(String fileName) {
      return fileName.indexOf(".") == -1 ? "" : fileName.substring(fileName.lastIndexOf(".") + 1);
   }

   public static final String getFileExtension(File file) {
      return getFileExtension(file.toString());
   }

   public static final String getFileExtension(Path path) {
      return getFileExtension(path.getFileName().toString());
   }

   public static final String getFileBaseName(File file) {
      return getFileBaseName(file.toPath());
   }

   public static final String getFileBaseName(Path path) {
      String fileName = path.getFileName().toString();
      return getFileBaseName(fileName);
   }

   public static String getFileBaseName(String fileName) {
      return fileName.indexOf(".") == -1 ? fileName : fileName.substring(0, fileName.lastIndexOf("."));
   }

   public static String getFileNameWithNewExtension(String fileName, String newExtension) {
      return getFileBaseName(fileName) + "." + newExtension;
   }

   public static void deleteRecursively(Path path) {
      if (Files.isDirectory(path, new LinkOption[0])) {
         try {
            Files.list(path).forEach((p) -> {
               deleteRecursively(p);
            });
            Files.deleteIfExists(path);
         } catch (IOException var3) {
            throw new FileException(var3);
         }
      } else {
         try {
            Files.deleteIfExists(path);
         } catch (IOException var2) {
            throw new FileException(var2);
         }
      }

   }

   public static void deleteQuietly(File file) {
      if (file != null) {
         try {
            file.delete();
         } catch (Exception var2) {
         }
      }

   }

   public static final boolean isInJarURL(URL url) {
      return StringUtils.contains(url.getPath(), ".jar!");
   }

   public static List<File> getFileList(Path dir) {
      if (!Files.isDirectory(dir, new LinkOption[0])) {
         return CollectionUtils.emptyList();
      } else {
         final ArrayList fileList = new ArrayList();

         try {
            Files.walkFileTree(dir, new SimpleFileVisitor<Path>() {
               public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                  fileList.add(file.toFile());
                  return FileVisitResult.CONTINUE;
               }
            });
            return fileList;
         } catch (IOException var3) {
            throw new FileException(var3);
         }
      }
   }

   public static final String makeValidFilename(String name) {
      StringBuilder sb = new StringBuilder();

      for(int i = 0; i < name.length(); ++i) {
         char c = name.charAt(i);
         switch(c) {
         case ' ':
            sb.append("_");
         case '"':
         case '*':
         case '/':
         case ':':
         case '<':
         case '>':
         case '?':
         case '\\':
         case '|':
            break;
         default:
            sb.append(c);
         }
      }

      return sb.toString();
   }

   public static final void encode(Path srcPath, Path destPath, Charset targetCharset) {
      try {
         InputStream is = Files.newInputStream(srcPath);
         Throwable var4 = null;

         try {
            DataInputStream in = new DataInputStream(is);
            Throwable var6 = null;

            try {
               BufferedReader d = new BufferedReader(new InputStreamReader(in, EncodingUtils.detect(srcPath.toFile())));
               StringBuilder sb = new StringBuilder();

               for(String line = null; (line = d.readLine()) != null; sb.append(line)) {
                  if (sb.length() != 0) {
                     sb.append(System.lineSeparator());
                  }
               }

               makeDirs(destPath);
               Writer ow = Files.newBufferedWriter(destPath, targetCharset);
               Throwable var11 = null;

               try {
                  ow.write(sb.toString());
               } catch (Throwable var58) {
                  var11 = var58;
                  throw var58;
               } finally {
                  if (ow != null) {
                     if (var11 != null) {
                        try {
                           ow.close();
                        } catch (Throwable var57) {
                           var11.addSuppressed(var57);
                        }
                     } else {
                        ow.close();
                     }
                  }

               }
            } catch (Throwable var60) {
               var6 = var60;
               throw var60;
            } finally {
               if (in != null) {
                  if (var6 != null) {
                     try {
                        in.close();
                     } catch (Throwable var56) {
                        var6.addSuppressed(var56);
                     }
                  } else {
                     in.close();
                  }
               }

            }
         } catch (Throwable var62) {
            var4 = var62;
            throw var62;
         } finally {
            if (is != null) {
               if (var4 != null) {
                  try {
                     is.close();
                  } catch (Throwable var55) {
                     var4.addSuppressed(var55);
                  }
               } else {
                  is.close();
               }
            }

         }

      } catch (IOException var64) {
         throw new FileException(var64);
      }
   }

   public static String extractTextFromFile(File file) {
      FileToTextExtractor textExtractor = FileToTextExtractorFactory.getFileToTextExtractor(getFileExtension(file.getName()));
      return textExtractor == null ? "" : textExtractor.extract(file);
   }

   public static String extractText(InputStream is, String extension) {
      FileToTextExtractor textExtractor = FileToTextExtractorFactory.getFileToTextExtractor(extension);
      return textExtractor == null ? "" : textExtractor.extract(is);
   }

   public static String extractFullTextFromFile(File file) {
      FileToTextExtractor textExtractor = FileToTextExtractorFactory.getFileToFullTextExtractor(getFileExtension(file.getName()));
      return textExtractor == null ? "" : textExtractor.extract(file);
   }

   public static String extractFullText(InputStream is, String extension) {
      FileToTextExtractor textExtractor = FileToTextExtractorFactory.getFileToFullTextExtractor(extension);
      return textExtractor == null ? "" : textExtractor.extract(is);
   }

   public static boolean isImage(File file) {
      String extension = getFileExtension(file).toLowerCase();
      return "jpg".equals(extension) || "jpeg".equals(extension) || "png".equals(extension) || "bmp".equals(extension) || "gif".equals(extension);
   }

   public static boolean isExtension(File file, String extension) {
      return extension.equalsIgnoreCase(getFileExtension(file).toLowerCase());
   }

   public static final MultipartFile makeMultipartFile(File file) {
      DiskFileItem fileItem = new DiskFileItem("file", "multipart/form-data", true, file.getName(), 10485760, file.getParentFile());

      try {
         OutputStream os = fileItem.getOutputStream();
         Throwable var3 = null;

         try {
            InputStream is = new FileInputStream(file);
            Throwable var5 = null;

            try {
               int length;
               byte[] bytes = new byte[4096];

               while((length = is.read(bytes)) > 0) {
                  os.write(bytes, 0, length);
               }
            } catch (Throwable var31) {
               var5 = var31;
               throw var31;
            } finally {
               if (is != null) {
                  if (var5 != null) {
                     try {
                        is.close();
                     } catch (Throwable var30) {
                        var5.addSuppressed(var30);
                     }
                  } else {
                     is.close();
                  }
               }

            }
         } catch (Throwable var33) {
            var3 = var33;
            throw var33;
         } finally {
            if (os != null) {
               if (var3 != null) {
                  try {
                     os.close();
                  } catch (Throwable var29) {
                     var3.addSuppressed(var29);
                  }
               } else {
                  os.close();
               }
            }

         }
      } catch (IOException var35) {
         throw new FileException(var35);
      }

      return new CommonsMultipartFile(fileItem);
   }

   public static final MultipartFile makeMultipartFile(String name, InputStream is) {
      DiskFileItem fileItem = new DiskFileItem("file", "multipart/form-data", true, name, 10485760, getTempDir());

      try {
         OutputStream os = fileItem.getOutputStream();
         Throwable var4 = null;

         try {
            byte[] bytes = new byte[4096];

            int length;
            while((length = is.read(bytes)) > 0) {
               os.write(bytes, 0, length);
            }
         } catch (Throwable var15) {
            var4 = var15;
            throw var15;
         } finally {
            if (os != null) {
               if (var4 != null) {
                  try {
                     os.close();
                  } catch (Throwable var14) {
                     var4.addSuppressed(var14);
                  }
               } else {
                  os.close();
               }
            }

         }
      } catch (IOException var17) {
         throw new FileException(var17);
      }

      return new CommonsMultipartFile(fileItem);
   }

   public static final File getTempDir() {
      return new File(System.getProperty("java.io.tmpdir"));
   }
}
