/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.utils;

import net.sinodawn.framework.cache.memory.MemoryCacheManager;
import net.sinodawn.framework.data.TypeValuePair;
import net.sinodawn.framework.exception.ClassInstantiationException;
import net.sinodawn.framework.exception.UnexpectedException;
import org.springframework.core.io.Resource;
import org.springframework.core.io.support.PathMatchingResourcePatternResolver;
import org.springframework.core.io.support.ResourcePatternResolver;
import org.springframework.core.type.classreading.CachingMetadataReaderFactory;
import org.springframework.core.type.classreading.MetadataReader;

import java.io.IOException;
import java.lang.reflect.Constructor;
import java.lang.reflect.InvocationTargetException;
import java.time.temporal.Temporal;
import java.util.*;
import java.util.function.Predicate;

@SuppressWarnings({"MismatchedQueryAndUpdateOfCollection", "unused", "unchecked", "rawtypes", "deprecation"})
public class ClassUtils {
    private static final Map<Class<?>, Class<?>> primitiveWrapperTypeMap = new IdentityHashMap<>(8);
    private static final Map<Class<?>, Class<?>> primitiveTypeToWrapperMap = new IdentityHashMap<>(8);

    public ClassUtils() {
    }

    public static <T> T newInstance(final Class<T> clazz) {
        Constructor<?>[] constructors = clazz.getDeclaredConstructors();
        boolean seen = false;
        Constructor<?> best = null;
        Comparator<Constructor<?>> comparator = Comparator.comparingInt(Constructor::getParameterCount);
        for (Constructor<?> constructor1 : constructors) {
            if (!seen || comparator.compare(constructor1, best) < 0) {
                seen = true;
                best = constructor1;
            }
        }
        Constructor<?> constructor = (seen ? Optional.<Constructor<?>>of(best) : Optional.<Constructor<?>>empty()).get();
        Class<?>[] parameterTypes = constructor.getParameterTypes();
        Object[] parameterDefaultValues = new Object[parameterTypes.length];

        for(int i = 0; i < parameterTypes.length; ++i) {
            Class<?> parameterType = parameterTypes[i];
            parameterDefaultValues[i] = isPrimitiveType(parameterType) ? getPrimitiveDefaultValue(parameterType) : null;
        }

        try {
            return (T) constructor.newInstance(parameterDefaultValues);
        } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException | InstantiationException var7) {
            throw new ClassInstantiationException(clazz, var7);
        }
    }

    @SafeVarargs
    public static <T> T newInstance(final Class<T> clazz, final TypeValuePair<?>... typeValuePairs) {
        if (typeValuePairs != null && typeValuePairs.length != 0) {
            Class<?>[] parameterTypes = new Class[typeValuePairs.length];
            Object[] parameterValues = new Object[typeValuePairs.length];

            for(int i = 0; i < typeValuePairs.length; ++i) {
                parameterTypes[i] = typeValuePairs[i].getFirst();
                parameterValues[i] = typeValuePairs[i].getSecond();
            }

            try {
                Constructor<T> constructor = clazz.getConstructor(parameterTypes);
                return constructor.newInstance(parameterValues);
            } catch (SecurityException | InstantiationException | IllegalAccessException | IllegalArgumentException | InvocationTargetException | NoSuchMethodException var5) {
                throw new ClassInstantiationException(clazz, var5);
            }
        } else {
            try {
                return clazz.newInstance();
            } catch (IllegalAccessException | InstantiationException var6) {
                throw new ClassInstantiationException(clazz, var6);
            }
        }
    }

    /**
     * 返回指定类型默认值
     * @param primitiveType Class<?>
     * @return Object
     */
    public static Object getPrimitiveDefaultValue(Class<?> primitiveType) {
        if (Boolean.TYPE == primitiveType) {
            return false;
        } else if (Byte.TYPE == primitiveType) {
            return '0';
        } else if (Character.TYPE == primitiveType) {
            return '0';
        } else if (Short.TYPE == primitiveType) {
            return Short.parseShort("0");
        } else if (Integer.TYPE == primitiveType) {
            return 0;
        } else if (Float.TYPE == primitiveType) {
            return 0.0F;
        } else if (Double.TYPE == primitiveType) {
            return 0.0D;
        } else {
            return Long.TYPE == primitiveType ? 0L : null;
        }
    }

    public static Class<?> resolvePrimitiveIfNecessary(Class<?> clazz) {
        return clazz.isPrimitive() && clazz != Void.TYPE ? getWrapperType(clazz) : clazz;
    }

    public static Class<?> getWrapperType(Class<?> primitiveType) {
        return primitiveTypeToWrapperMap.get(primitiveType);
    }

    public static <T> Constructor<T> getConstructorIfAvailable(final Class<T> clazz, final Class<?>... parameterTypes) {
        try {
            return clazz.getConstructor(parameterTypes);
        } catch (NoSuchMethodException var3) {
            return null;
        }
    }

    public static Class<?> getRawType(Class<?> clazz) {
        if (clazz.getName().startsWith("com.sun.proxy.$Proxy")) {
            return clazz.getInterfaces()[0];
        } else {
            while(clazz.getName().contains("CGLIB$")) {
                clazz = clazz.getSuperclass();
            }
            return clazz;
        }
    }

    public static boolean isCglibProxyClass(Class<?> clazz) {
        return clazz != null && clazz.getName().contains("CGLIB$");
    }

    public static boolean isPrimitiveType(Class<?> clazz) {
        return primitiveTypeToWrapperMap.containsKey(clazz);
    }

    public static boolean isWrapperType(Class<?> clazz) {
        return primitiveWrapperTypeMap.containsKey(clazz);
    }

    public static <T> Class<T> getClass(String className) {
        ClassLoader contextCL = Thread.currentThread().getContextClassLoader();
        ClassLoader loader = contextCL == null ? ClassLoader.getSystemClassLoader() : contextCL;

        try {
            String abbreviation = getClassAbbreviation(className);
            return className.equals(abbreviation) ? (Class<T>) Class.forName(getFullyQualifiedName(className), false, loader) : (Class<T>) Class.forName("[" + abbreviation, false, loader).getComponentType();
        } catch (ClassNotFoundException var4) {
            throw new UnexpectedException(var4);
        }
    }

    public static boolean isSimpleType(Class<?> type) {
        return isPrimitiveType(type) || isWrapperType(type) || CharSequence.class.isAssignableFrom(type) || Number.class.isAssignableFrom(type) || Date.class.isAssignableFrom(type) || Map.class.isAssignableFrom(type) || Collection.class.isAssignableFrom(type) || Temporal.class.isAssignableFrom(type) || Object.class == type;
    }

    public static <T> Class<T> getSinoClass(String simpleName) {
        return (Class<T>) getAllClasses("net.sinodawn").stream()
                .filter((c) -> c.getSimpleName().equalsIgnoreCase(simpleName)).findFirst().orElse(null);
    }

    public static List<Class<?>> getPredicatedClasses(String packageName, Predicate<Class<?>> classTester) {
        String packageDirName = StringUtils.removeStart(StringUtils.removeEnd(StringUtils.replace(packageName, ".", "/"), "/"), "/");
        ResourcePatternResolver resourcePatternResolver = new PathMatchingResourcePatternResolver();
        CachingMetadataReaderFactory metadataReaderFactory = new CachingMetadataReaderFactory(resourcePatternResolver);

        try {
            Resource[] resources = resourcePatternResolver.getResources("classpath*:" + packageDirName + "/**/*.class");
            List<Class<?>> list = new ArrayList();
            int var8 = resources.length;

            for (Resource r : resources) {
                MetadataReader metadataReader = metadataReaderFactory.getMetadataReader(r);
                Class<?> clazz = getClass(metadataReader.getClassMetadata().getClassName());
                if (classTester.test(clazz)) {
                    list.add(clazz);
                }
            }

            return list;
        } catch (IOException var13) {
            throw new UnexpectedException(var13);
        }
    }

    public static List<Class<?>> getAllClasses(String packageName) {
        List<Class<?>> classList = MemoryCacheManager.get("classesUnderPackage:" + packageName);
        if (classList == null) {
            classList = new ArrayList<>();
            String packageDirName = StringUtils.removeStart(StringUtils.removeEnd(StringUtils.replace(packageName, ".", "/"), "/"), "/");
            ResourcePatternResolver resourcePatternResolver = new PathMatchingResourcePatternResolver();
            CachingMetadataReaderFactory metadataReaderFactory = new CachingMetadataReaderFactory(resourcePatternResolver);

            try {
                Resource[] resources = resourcePatternResolver.getResources("classpath*:" + packageDirName + "/**/*.class");
                int var7 = resources.length;

                for (Resource r : resources) {
                    MetadataReader metadataReader = metadataReaderFactory.getMetadataReader(r);
                    if (!"net.sinodawn.framework.utils.EncodingUtils$BytesEncodingDetect".equals(metadataReader.getClassMetadata().getClassName())) {
                        Class<?> clazz = getClass(metadataReader.getClassMetadata().getClassName());
                        classList.add(clazz);
                    }
                }
            } catch (IOException var12) {
                throw new UnexpectedException(var12);
            }

            MemoryCacheManager.put("classesUnderPackage:" + packageName, Collections.unmodifiableList(classList));
        }

        return classList;
    }

    private static String getFullyQualifiedName(final String className) {
        if (!className.endsWith("[]")) {
            return className;
        } else {
            StringBuilder sb = new StringBuilder();
            String name = className;

            while(name.endsWith("[]")) {
                name = name.substring(0, name.length() - 2);
                sb.append("[");
            }

            String abbreviation = getClassAbbreviation(name);
            if (StringUtils.isEmpty(abbreviation)) {
                sb.append("L").append(name).append(";");
            } else {
                sb.append(abbreviation);
            }

            return sb.toString();
        }
    }

    private static String getClassAbbreviation(final String className) {
        if ("int".equals(className)) {
            return "I";
        } else if ("long".equals(className)) {
            return "J";
        } else if ("double".equals(className)) {
            return "D";
        } else if ("float".equals(className)) {
            return "F";
        } else if ("char".equals(className)) {
            return "C";
        } else if ("short".equals(className)) {
            return "S";
        } else if ("boolean".equals(className)) {
            return "Z";
        } else if ("byte".equals(className)) {
            return "B";
        } else {
            return "void".equals(className) ? "V" : className;
        }
    }

    static {
        primitiveWrapperTypeMap.put(Boolean.class, Boolean.TYPE);
        primitiveWrapperTypeMap.put(Byte.class, Byte.TYPE);
        primitiveWrapperTypeMap.put(Character.class, Character.TYPE);
        primitiveWrapperTypeMap.put(Double.class, Double.TYPE);
        primitiveWrapperTypeMap.put(Float.class, Float.TYPE);
        primitiveWrapperTypeMap.put(Integer.class, Integer.TYPE);
        primitiveWrapperTypeMap.put(Long.class, Long.TYPE);
        primitiveWrapperTypeMap.put(Short.class, Short.TYPE);
        primitiveWrapperTypeMap.forEach((w, p) -> {
            Class var10000 = primitiveTypeToWrapperMap.put(p, w);
        });
    }
}
