/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.utils;


import com.alibaba.fastjson.JSON;
import net.sinodawn.framework.exception.EmptyArrayException;
import net.sinodawn.framework.exception.OutOfRangeException;
import net.sinodawn.framework.exception.TypeMismatchException;

import java.lang.reflect.Array;
import java.util.Arrays;
import java.util.List;
import java.util.stream.Collectors;

@SuppressWarnings({"unused", "unchecked"})
public abstract class ArrayUtils {
    public ArrayUtils() {
    }

    public static Class<?> getArrayType(Class<?> arrayClass) {
        if (!arrayClass.isArray()) {
            throw new TypeMismatchException("array", arrayClass);
        } else {
            Class componentType;
            componentType = arrayClass.getComponentType();
            while (componentType.isArray()) {
                componentType = componentType.getComponentType();
            }

            return componentType;
        }
    }

    public static boolean hasElement(Object array) {
        return array != null && array.getClass().isArray() && Array.getLength(array) > 0;
    }

    public static <T> boolean contains(T[] array, T value) {
        return array != null && array.length > 0 && Arrays.stream(array).anyMatch((t) -> {
            return value == null && t == null || value != null && value.equals(t);
        });
    }

    public static boolean containsIgnoreCase(String[] array, String value) {
        return array != null && array.length > 0 && Arrays.stream(array).anyMatch((t) -> {
            return value == null && t == null || value != null && value.equalsIgnoreCase(t);
        });
    }

    public static <T> T[] concat(T[] first, T[] second) {
        if (first == null) {
            return second;
        } else if (second == null) {
            return first;
        } else {
            T[] array = (T[]) Array.newInstance(first.getClass().getComponentType(), first.length + second.length);
            System.arraycopy(first, 0, array, 0, first.length);
            System.arraycopy(second, 0, array, first.length, second.length);
            return array;
        }
    }

    public static <T> T[] append(T[] array, T target) {
        if (target == null) {
            return array;
        } else {
            int length = array == null ? 0 : array.length;
            T[] newArray = (T[]) Array.newInstance(target.getClass(), length + 1);
            if (length > 0) {
                System.arraycopy(array, 0, newArray, 0, length);
            }

            newArray[length] = target;
            return newArray;
        }
    }

    public static <T> String arrayToDelimitedString(T[] array, String delimit) {
        if (!hasElement(array)) {
            return "";
        } else if (array.length == 1) {
            return ConvertUtils.convert(array[0], String.class, "");
        } else {
            StringBuilder sb = new StringBuilder();
            Arrays.stream(array).forEach((v) -> sb.append(ConvertUtils.convert(v, String.class, "")).append(delimit));
            return sb.substring(0, sb.length() - delimit.length());
        }
    }

    public static <T> T[] requireNotEmpty(T[] array) {
        if (!hasElement(array)) {
            throw new EmptyArrayException();
        } else {
            return array;
        }
    }

    public static <T> T[] emptyArray(Class<T> componentType) {
        return newInstance(componentType, 0);
    }

    public static <T> T[] newInstance(Class<T> componentType, int size) {
        return (T[]) Array.newInstance(componentType, size);
    }

    public static <T> T[] instanceToArray(T instance) {
        T[] array = (T[]) Array.newInstance(instance.getClass(), 1);
        array[0] = instance;
        return array;
    }

    public static <A, T> T getUniqueTypeValue(A[] array, Class<T> type) {
        if (array != null && array.length != 0) {
            List<T> list = Arrays.stream(array).filter((a) -> a != null && a.getClass().equals(type))
                    .map(m -> (T)m).collect(Collectors.toList());
            if (list.size() > 1) {
                throw new OutOfRangeException(JSON.toJSONString(array) + " contains more than one instance of type " + type + ".");
            } else {
                return list.isEmpty() ? null : list.get(0);
            }
        } else {
            return null;
        }
    }

    public static <T> Class<? extends T> getUniqueAssignableType(Class<?>[] array, Class<T> type) {
        if (array != null && array.length != 0) {
            List<Class<?>> list = Arrays.stream(array).filter((a) -> a != null && type.isAssignableFrom(a)).collect(Collectors.toList());
            if (list.size() > 1) {
                throw new OutOfRangeException(JSON.toJSONString(array) + " contains more than one class assignable type " + type + ".");
            } else {
                return list.isEmpty() ? null : (Class)list.get(0);
            }
        } else {
            return null;
        }
    }

    public static <T> boolean isEquals(T[] left, T[] right) {
        if (left == null && right == null) {
            return true;
        } else if (left == null && right != null || left != null && right == null) {
            return false;
        } else if (left.length != right.length) {
            return false;
        } else {
            int i = 0;

            for(int j = left.length; i < j; ++i) {
                T leftValue = left[i];
                T rightValue = right[i];
                if (leftValue == null) {
                    if (rightValue != null) {
                        return false;
                    }
                } else if (!leftValue.equals(rightValue)) {
                    return false;
                }
            }

            return true;
        }
    }

    public static <T> List<T> asList(T... items) {
        return items == null ? CollectionUtils.emptyList() : Arrays.asList(items);
    }

    public static <T> boolean isEmpty(T... items) {
        return items == null || items.length == 0;
    }

    public static <T> T[] subarray(T[] array, int start, int length) {
        if (array != null && array.length != 0) {
            Class<T> type = (Class<T>) array[0].getClass();
            if (array.length < start + length) {
                return (T[]) emptyArray(type);
            } else {
                T[] subArray = newInstance(type, length);
                System.arraycopy(array, start, subArray, 0, length);
                return subArray;
            }
        } else {
            return array;
        }
    }
}
