/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.utils;

import net.sinodawn.framework.exception.FileException;
import net.sinodawn.framework.exception.ZipException;
import java.io.File;
import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.nio.file.CopyOption;
import java.nio.file.Files;
import java.nio.file.Path;
import java.time.LocalDateTime;
import java.util.Arrays;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

public abstract class ArchiveUtils {
   private static final int MAX_BUFFER_SIZE = 1024;

   public static final void extractResourceIfNotPresent(String resourceName, Path destPath) {
      File f = destPath.toFile();
      if (f.isDirectory()) {
         throw new FileException("core.zip.dest-not-be-directory");
      } else {
         if (!f.exists()) {
            try {
               InputStream exeIs = ArchiveUtils.class.getResourceAsStream(resourceName);
               Throwable var4 = null;

               try {
                  if (exeIs != null) {
                     Files.copy(exeIs, destPath, new CopyOption[0]);
                  }
               } catch (Throwable var14) {
                  var4 = var14;
                  throw var14;
               } finally {
                  if (exeIs != null) {
                     if (var4 != null) {
                        try {
                           exeIs.close();
                        } catch (Throwable var13) {
                           var4.addSuppressed(var13);
                        }
                     } else {
                        exeIs.close();
                     }
                  }

               }
            } catch (IOException var16) {
               throw new FileException(var16);
            }
         }

      }
   }

   public static void zip(Path dest, Path... sources) {
      if (sources == null) {
         zip(dest.toFile());
      } else {
         zip(dest.toFile(), (File[])Arrays.stream(sources).map((s) -> {
            return s.toFile();
         }).toArray((x$0) -> {
            return new File[x$0];
         }));
      }

   }

   public static void zip(File dest, File... sources) {
      if (dest.isDirectory()) {
         throw new FileException("SINO.FILE.ZIP.DEST-MUST-BE-A-FILE");
      } else {
         if (dest.exists()) {
            dest.delete();

            try {
               dest.createNewFile();
            } catch (IOException var35) {
               throw new FileException(var35);
            }
         } else {
            FileUtils.createFile(dest);
         }

         if (sources != null && sources.length != 0) {
            try {
               FileOutputStream fos = new FileOutputStream(dest);
               Throwable var3 = null;

               try {
                  ZipOutputStream out = new ZipOutputStream(fos);
                  Throwable var5 = null;

                  try {
                     File[] var6 = sources;
                     int var7 = sources.length;

                     for(int var8 = 0; var8 < var7; ++var8) {
                        File source = var6[var8];
                        putFile(out, source, null);
                     }
                  } catch (Throwable var36) {
                     var5 = var36;
                     throw var36;
                  } finally {
                     if (out != null) {
                        if (var5 != null) {
                           try {
                              out.close();
                           } catch (Throwable var34) {
                              var5.addSuppressed(var34);
                           }
                        } else {
                           out.close();
                        }
                     }

                  }
               } catch (Throwable var38) {
                  var3 = var38;
                  throw var38;
               } finally {
                  if (fos != null) {
                     if (var3 != null) {
                        try {
                           fos.close();
                        } catch (Throwable var33) {
                           var3.addSuppressed(var33);
                        }
                     } else {
                        fos.close();
                     }
                  }

               }

            } catch (IOException var40) {
               throw new ZipException(var40);
            }
         }
      }
   }

   private static final void putFile(ZipOutputStream out, File file, String dir) {
      int len;
      if (file.isDirectory()) {
         String dirPath = StringUtils.isEmpty(dir) ? file.getName() : dir + File.separator + file.getName();
         File[] subFiles = file.listFiles();
         if (subFiles.length == 0) {
            try {
               out.putNextEntry(new ZipEntry(dirPath));
               out.closeEntry();
            } catch (IOException var19) {
               throw new ZipException(var19);
            }
         } else {
            File[] var26 = subFiles;
            len = subFiles.length;

            for(int var28 = 0; var28 < len; ++var28) {
               File subFile = var26[var28];
               putFile(out, subFile, dirPath);
            }
         }
      } else if (file.isFile()) {
         try {
            InputStream is = new FileInputStream(file);
            Throwable var4 = null;

            try {
               String zipEntryName = StringUtils.isEmpty(dir) ? file.getName() : dir + File.separator + file.getName();

               try {
                  out.putNextEntry(new ZipEntry(zipEntryName));
               } catch (java.util.zip.ZipException var20) {
                  if (!StringUtils.contains(var20.getMessage(), "duplicate entry")) {
                     throw var20;
                  }

                  zipEntryName = FileUtils.getFileBaseName(zipEntryName) + DateTimeUtils.getDateTimeSerialNumber(LocalDateTime.now()) + "." + FileUtils.getFileExtension(file.getName());
                  out.putNextEntry(new ZipEntry(zipEntryName));
               }

               byte[] buffer = new byte[1024];

               while((len = is.read(buffer)) > 0) {
                  out.write(buffer, 0, len);
               }

               out.closeEntry();
            } catch (Throwable var21) {
               var4 = var21;
               throw var21;
            } finally {
               if (is != null) {
                  if (var4 != null) {
                     try {
                        is.close();
                     } catch (Throwable var18) {
                        var4.addSuppressed(var18);
                     }
                  } else {
                     is.close();
                  }
               }

            }
         } catch (IOException var23) {
            throw new ZipException(var23);
         }
      }

   }
}
