/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.support.base.service;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.alibaba.fastjson.parser.Feature;
import net.sinodawn.framework.at.annotation.AuditTrailEntry;
import net.sinodawn.framework.at.annotation.AuditTrailType;
import net.sinodawn.framework.audit.aunnotation.Audit;
import net.sinodawn.framework.beans.BeanPropertyDescriptor;
import net.sinodawn.framework.beans.BeanPropertyHelper;
import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.context.LocalContextHelper;
import net.sinodawn.framework.data.CaseInsensitiveLinkedMap;
import net.sinodawn.framework.data.Pair;
import net.sinodawn.framework.data.page.Page;
import net.sinodawn.framework.data.page.PageRequest;
import net.sinodawn.framework.data.page.Pagination;
import net.sinodawn.framework.database.sql.Order;
import net.sinodawn.framework.exception.UnsupportedException;
import net.sinodawn.framework.mybatis.MybatisHelper;
import net.sinodawn.framework.mybatis.mapper.MapperParameter;
import net.sinodawn.framework.mybatis.mapper.SearchFilter;
import net.sinodawn.framework.mybatis.page.MybatisPageHelper;
import net.sinodawn.framework.mybatis.page.PageRowBounds;
import net.sinodawn.framework.restful.data.RestJsonWrapperBean;
import net.sinodawn.framework.restful.data.RestValidationResultBean;
import net.sinodawn.framework.support.PersistableHelper;
import net.sinodawn.framework.support.PersistableMetadataHelper;
import net.sinodawn.framework.support.auditable.service.GenericAuditableService;
import net.sinodawn.framework.support.base.bean.CoreServiceStatusDTO;
import net.sinodawn.framework.support.base.dao.GenericDao;
import net.sinodawn.framework.support.base.mapper.GenericMapper;
import net.sinodawn.framework.support.domain.Activatable;
import net.sinodawn.framework.support.domain.Auditable;
import net.sinodawn.framework.support.domain.Persistable;
import net.sinodawn.framework.support.domain.Suspendable;
import net.sinodawn.framework.utils.*;
import net.sinodawn.module.sys.bpmn.bean.CoreBpmnInstanceTaskBean;
import net.sinodawn.module.sys.bpmn.diagram.ProcessStatus;
import net.sinodawn.module.sys.bpmn.engine.CoreBpmnRuntimeService;
import net.sinodawn.module.sys.bpmn.exception.BpmnException;
import net.sinodawn.module.sys.metadata.service.CoreTableService;
import org.springframework.transaction.annotation.Transactional;

import java.io.Serializable;
import java.lang.reflect.Field;
import java.util.Arrays;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Supplier;
import java.util.stream.Collectors;

@SuppressWarnings({"unused", "unchecked", "rawtypes"})
public interface GenericService<T extends Persistable<ID>, ID extends Serializable> {
    <D extends GenericDao<T, ID>> D getDao();
    default List<T> selectAll() {
        return this.getDao().selectAll();
    }

    default List<T> selectListByIds(List<ID> idList) {
        return this.getDao().selectListByIds(idList, new Order[0]);
    }

    default List<T> selectList(T item, Order... orders) {
        return this.getDao().selectList(item, orders);
    }

    default T selectDetail(ID id) {
        Map<String, Object> map = this.getDao().selectDetail(id);
        return map != null && !map.isEmpty() ? PersistableHelper.mapToPersistable(map, this.getDao().getType()) : null;
    }

    @Transactional
    @AuditTrailEntry(AuditTrailType.DELETE)
    @Audit("AUDIT.DELETE")
    default void delete(RestJsonWrapperBean wrapper) {
        List<T> deleteList = wrapper.parse(this.getDao().getType());
        if (!deleteList.isEmpty()) {
            Class<T> itemType = this.getDao().getType();
            CoreTableService tableService = (CoreTableService)ApplicationContextHelper.getBean(CoreTableService.class);
            deleteList.stream().forEach((t) -> {
                tableService.deleteCascade(PersistableMetadataHelper.getTableName(itemType), (Serializable)t.getId());
            });
        }
    }

    @Transactional
    @AuditTrailEntry(AuditTrailType.DELETE)
    @Audit("AUDIT.DELETE")
    default void delete(ID id) {
        this.getDao().delete(id);
    }

    @Transactional
    @AuditTrailEntry(AuditTrailType.DELETE)
    @Audit("AUDIT.DELETE")
    default void deleteCascadeById(ID id) {
        CoreTableService tableService = (CoreTableService)ApplicationContextHelper.getBean(CoreTableService.class);
        tableService.deleteCascade(PersistableMetadataHelper.getTableName(this.getDao().getType()), id);
    }

    default T selectById(ID id) {
        return this.getDao().selectById(id);
    }

    default T selectByIdIfPresent(ID id) {
        return this.getDao().selectByIdIfPresent(id);
    }

    default String selectColumnById(ID id, String column) {
        return (String)this.getDao().selectColumnById(id, column, String.class);
    }

    default <V> V selectColumnById(ID id, String column, Class<V> columnType) {
        return this.getDao().selectColumnById(id, column, columnType);
    }

    @Transactional
    @AuditTrailEntry(AuditTrailType.INSERT)
    @Audit("AUDIT.INSERT")
    default ID insert(RestJsonWrapperBean wrapper) {
        List<T> beans = wrapper.parse(this.getDao().getType());
        beans.forEach(c -> {
            if (ObjectUtils.isEmpty(c.getId())) {
                c.setId((ID) ApplicationContextHelper.getNextIdentity());
            }
        });
        getDao().insert(beans);
        return beans.isEmpty() ? (ID) "0" : beans.get(0).getId();
    }

    @Transactional
    @AuditTrailEntry(AuditTrailType.UPDATE)
    @Audit("AUDIT.INSTANT_SAVE")
    default void instantSave(RestJsonWrapperBean wrapper) {
        List<GenericService<?, ?>> serviceList = wrapper.getBodyServiceList();
        int size = serviceList.size();

        for(int i = 0; i < size; ++i) {
            GenericService service = serviceList.get(i);
            if (service != null) {
                service.updateIfChanged(wrapper.parse(service.getDao().getType(), i));
            }
        }

    }

    @Transactional
    @AuditTrailEntry(AuditTrailType.UPDATE)
    @Audit("AUDIT.SAVE")
    default void save(ID id, RestJsonWrapperBean wrapper) {
        List<GenericService<?, ?>> serviceList = wrapper.getBodyServiceList();
        int size = serviceList.size();

        for(int i = 0; i < size; ++i) {
            GenericService service = serviceList.get(i);
            if (service == null) {
                this.updateIfChanged(wrapper.parse(this.getDao().getType(), i));
            } else {
                service.updateIfChanged(wrapper.parse(service.getDao().getType(), i));
            }
        }

    }

    @Transactional
    @AuditTrailEntry(AuditTrailType.UPDATE)
    @Audit("AUDIT.SAVE")
    default void updateIfChanged(T item) {
        this.getDao().updateIfChanged(item);
    }

    @AuditTrailEntry(AuditTrailType.UPDATE)
    @Audit("AUDIT.SAVE")
    default void update(T proxy) {
        this.getDao().update(proxy, new String[0]);
    }

    @AuditTrailEntry(AuditTrailType.UPDATE)
    @Audit("AUDIT.SAVE")
    default void updateIfChanged(List<T> itemList) {
        this.getDao().updateIfChanged(itemList);
    }

    @AuditTrailEntry(AuditTrailType.UPDATE)
    @Audit("AUDIT.SAVE")
    default void updateCreatedBy(T item) {
        this.getDao().updateCreatedBy(item);
    }

    @AuditTrailEntry(AuditTrailType.UPDATE)
    @Audit("AUDIT.SAVE")
    default void updateCreatedBy(List<T> itemList) {
        this.getDao().updateCreatedBy(itemList);
    }

    // TODO 日期转字符串格式异常
    default Page<T> selectPagination(MapperParameter parameter, PageRowBounds rowBounds) {
        return this.selectPagination(() -> this.getDao().selectByCondition(parameter), rowBounds);
    }

    default <V> Page<T> selectPagination(Supplier<List<V>> mapOrItemSupplier, PageRowBounds rowBounds) {
        Page<V> page = MybatisPageHelper.get(rowBounds, mapOrItemSupplier);
        if (page.getRows().isEmpty()) {
            return new Pagination(page, CollectionUtils.emptyList());
        } else {
            Class<V> clazz = (Class<V>) ClassUtils.getRawType(page.getRows().get(0).getClass());
            if (Map.class.isAssignableFrom(clazz)) {
                List<T> itemList = (List)page.getRows().parallelStream().map((m) -> PersistableHelper.mapToPersistable((Map)m, this.getDao().getType())).collect(Collectors.toList());
                return new Pagination(page, itemList);
            } else if (this.getDao().getType().isAssignableFrom(clazz)) {
                return (Page<T>) page;
            } else {
                throw new UnsupportedException();
            }
        }
    }

    default Page<T> selectPagination(MapperParameter parameter) {
        return this.selectPagination(parameter, null);
    }

    default Page<T> selectPagination(RestJsonWrapperBean wrapper) {
        MapperParameter parameter = wrapper.extractMapFilter();
        PageRowBounds rowBounds = wrapper.extractPageRowBounds();
        return this.selectPagination(parameter, rowBounds);
    }

    default Page<T> selectPaginationByFilter(SearchFilter filter, RestJsonWrapperBean wrapper) {
        MapperParameter parameter = wrapper.extractMapFilter();
        parameter.setFilter(filter);
        PageRowBounds rowBounds = wrapper.extractPageRowBounds();
        return this.selectPagination(parameter, rowBounds);
    }

    default Page<T> selectRawPagination(RestJsonWrapperBean wrapper) {
        MapperParameter parameter = wrapper.extractMapFilter();
        PageRowBounds rowBounds = wrapper.extractPageRowBounds();
        Class<T> type = this.getDao().getType();
        if (Auditable.class.isAssignableFrom(type)) {
            parameter.setProcessStatus("PROCESSSTATUS", ProcessStatus.DRAFT);
        }

        parameter.setRawQueries();
        BeanPropertyDescriptor createdByOrgIdPropertyDescriptor = PersistableMetadataHelper.getCreatedByOrgIdPropertyDescriptor(type);
        if (createdByOrgIdPropertyDescriptor != null) {
            parameter.setOrgAuthority();
        }

        if (!wrapper.getAuthorityList().isEmpty()) {
            wrapper.getAuthorityList().forEach((a) -> {
                parameter.setAuthorityParameter(a);
            });
        }

        return this.selectPagination(parameter, rowBounds);
    }

    default Page<T> selectByRole(Long roleId, RestJsonWrapperBean wrapper) {
        MapperParameter parameter = wrapper.extractMapFilter();
        parameter.setSelectByRoleIdParam(roleId);
        PageRowBounds rowBounds = wrapper.extractPageRowBounds();
        return this.selectPagination(parameter, rowBounds);
    }

    default Page<T> selectSearchablePagination(RestJsonWrapperBean wrapper) {
        MapperParameter parameter = wrapper.extractMapFilter();
        PageRowBounds rowBounds = wrapper.extractPageRowBounds();
        Class<T> type = this.getDao().getType();
        if (Auditable.class.isAssignableFrom(type)) {
            parameter.setProcessStatus("PROCESSSTATUS", ProcessStatus.APPROVE_OR_DONE);
        }

        parameter.setSearchableQueries();
        BeanPropertyDescriptor createdByOrgIdPropertyDescriptor = PersistableMetadataHelper.getCreatedByOrgIdPropertyDescriptor(type);
        if (createdByOrgIdPropertyDescriptor != null) {
            parameter.setOrgAuthority();
        }

        if (!wrapper.getAuthorityList().isEmpty()) {
            wrapper.getAuthorityList().forEach((a) -> {
                parameter.setAuthorityParameter(a);
            });
        }

        return this.selectPagination(parameter, rowBounds);
    }

    default RestValidationResultBean validate(ID id, RestJsonWrapperBean jsonWrapper) {
        return new RestValidationResultBean(true);
    }

    default RestValidationResultBean validateUnique(RestJsonWrapperBean wrapper) {
        String validateUniqueFieldJson = wrapper.getParamValue("vu");
        if (null == validateUniqueFieldJson) {
            return new RestValidationResultBean(true);
        } else {
            Map<String, Object> validateUniqueFieldMap = (Map) JSONObject.parseObject(validateUniqueFieldJson, new TypeReference<Map<String, Object>>() {
            }, new Feature[0]);
            Map<String, Object> columnMap = new CaseInsensitiveLinkedMap(validateUniqueFieldMap);
            ID id = (ID) ConvertUtils.convert(columnMap.get("id"), this.getDao().getEntityContext().getIdContext().getType());
            columnMap.remove("id");
            return this.validateUnique(id, columnMap);
        }
    }

    default RestValidationResultBean validateUnique(ID id, String columnName, String columnValue) {
        Map<String, Object> columnMap = new CaseInsensitiveLinkedMap();
        columnMap.put(columnName, columnValue);
        return this.validateUnique(id, columnMap);
    }

    default RestValidationResultBean validateUnique(ID id, Map<String, Object> columnMap) {
        Class<T> type = this.getDao().getType();
        T searchTerms;
        if (columnMap == null || columnMap.isEmpty() || columnMap.size() == 1 && columnMap.containsKey("ID") || columnMap.size() == 1 && columnMap.containsKey(null)) {
            searchTerms = this.getDao().selectByIdIfPresent(id);
            return searchTerms == null ? new RestValidationResultBean(true) : new RestValidationResultBean(false, "core.validator.column.not-unique");
        } else {
            searchTerms = ClassUtils.newInstance(type);
            columnMap.forEach((k, v) -> {
                BeanPropertyDescriptor propertyDescriptor = BeanPropertyHelper.getBeanPropertyDescriptor(type, k);
                if (propertyDescriptor == null) {
                    searchTerms.setExt$Item(k, v == null ? null : v.toString());
                } else {
                    ReflectionUtils.invokeMethod(propertyDescriptor.getWriteMethod(), searchTerms, new Object[]{ConvertUtils.convert(v, propertyDescriptor.getPropertyType())});
                }

            });
            List<T> idList = this.getDao().selectColumnList(searchTerms, "ID", (Class<T>)this.getDao().getEntityContext().getIdContext().getType());
            if (idList.isEmpty()) {
                return new RestValidationResultBean(true);
            } else if (idList.size() > 1) {
                return new RestValidationResultBean(false, "core.validator.column.not-unique");
            } else {
                return !idList.get(0).equals(id) ? new RestValidationResultBean(false, "core.validator.column.not-unique") : new RestValidationResultBean(true);
            }
        }
    }

    default boolean isAvailableByCurrentUser(ID id) {
        Map<String, Object> map = new HashMap();
        map.put("id_SEQ", id);
        RestJsonWrapperBean wrapper = new RestJsonWrapperBean(JSONObject.toJSONString(map));
        wrapper.removePageParams();
        Page<T> rawPage = this.selectRawPagination(wrapper);
        if (rawPage.getNumberOfElements() > 0) {
            return true;
        } else {
            Page<T> searchPage = this.selectSearchablePagination(wrapper);
            return searchPage.getNumberOfElements() > 0;
        }
    }

    default boolean isUpdatedByCurrentUser(ID id) {
        Map<String, Object> map = new HashMap();
        map.put("id_SEQ", id);
        RestJsonWrapperBean wrapper = new RestJsonWrapperBean(JSONObject.toJSONString(map));
        wrapper.removePageParams();
        Page<T> page = this.selectRawPagination(wrapper);
        return page.getNumberOfElements() > 0;
    }

    default CoreServiceStatusDTO selectStatus(ID id, String currentStatusCode) {
        CoreServiceStatusDTO status = new CoreServiceStatusDTO();
        T item = id == null ? null : this.getDao().selectByIdIfPresent(id);
        if (Activatable.class.isAssignableFrom(this.getDao().getType())) {
            CoreServiceStatusDTO.ActiveStatus activeStatus = new CoreServiceStatusDTO.ActiveStatus();
            if (item != null) {
                activeStatus.setActivatedFlag(((Activatable)item).getActivatedFlag());
            } else {
                activeStatus.setActivatedFlag("0");
            }

            status.setActiveStatus(activeStatus);
        }

        if (Suspendable.class.isAssignableFrom(this.getDao().getType())) {
            CoreServiceStatusDTO.SuspendStatus suspendStatus = new CoreServiceStatusDTO.SuspendStatus();
            if (item != null) {
                suspendStatus.setLastSuspendedFlag(((Suspendable)item).getLastSuspendedFlag());
            } else {
                suspendStatus.setLastSuspendedFlag("0");
            }

            status.setSuspendStatus(suspendStatus);
        }

        if (Auditable.class.isAssignableFrom(this.getDao().getType())) {
            CoreServiceStatusDTO.AuditStatus auditStatus = new CoreServiceStatusDTO.AuditStatus();
            if (item != null) {
                auditStatus.setProcessStatus(((Auditable)item).getProcessStatus());
                if (!StringUtils.isEmpty(currentStatusCode)) {
                    auditStatus.setStatusCode(currentStatusCode);
                } else {
                    CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
                    List<CoreBpmnInstanceTaskBean> instanceTaskList = runtimeService.selectAuditableInstanceTaskList(this.getDao().getTable(), id, LocalContextHelper.getLoginUserId());
                    if (instanceTaskList.size() > 1) {
                        throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.MULTIPLE_TASK_ASSIGNED");
                    }

                    if (instanceTaskList.size() > 0) {
                        auditStatus.setStatusCode(((CoreBpmnInstanceTaskBean)instanceTaskList.get(0)).getStatusCode());
                    } else {
                        auditStatus.setStatusCode(ProcessStatus.DRAFT.name().toLowerCase());
                    }
                }
            } else {
                auditStatus.setProcessStatus(ProcessStatus.DRAFT.name().toLowerCase());
                auditStatus.setStatusCode(ProcessStatus.DRAFT.name().toLowerCase());
            }

            status.setAuditStatus(auditStatus);
        }

//        CorePageViewConfigService viewConfigService;
//        if (status.getAuditStatus() != null) {
//            viewConfigService = (CorePageViewConfigService)ApplicationContextHelper.getBean(CorePageViewConfigService.class);
//            String statusCode = status.getAuditStatus().getStatusCode();
//            CorePageViewConfigBean viewConfig = viewConfigService.selectById(this.getDao().getTable() + "$BPMN$" + statusCode);
//            if (viewConfig == null) {
//                viewConfig = viewConfigService.selectById(this.getDao().getTable() + "$BPMN_DEFAULT$" + statusCode);
//            }
//
//            if (viewConfig != null) {
//                status.setViewConfig(viewConfig.getConfig());
//            }
//        } else {
//            CorePageViewConfigBean viewConfig;
//            if (status.getActiveStatus() != null) {
//                viewConfigService = (CorePageViewConfigService)ApplicationContextHelper.getBean(CorePageViewConfigService.class);
//                viewConfig = viewConfigService.selectById(this.getDao().getTable() + "$ACTIVE$" + status.getActiveStatus().getActivatedFlag());
//                if (viewConfig != null) {
//                    status.setViewConfig(viewConfig.getConfig());
//                }
//            } else if (status.getSuspendStatus() != null) {
//                viewConfigService = (CorePageViewConfigService)ApplicationContextHelper.getBean(CorePageViewConfigService.class);
//                viewConfig = viewConfigService.selectById(this.getDao().getTable() + "$SUSPEND$" + status.getSuspendStatus().getLastSuspendedFlag());
//                if (viewConfig != null) {
//                    status.setViewConfig(viewConfig.getConfig());
//                }
//            }
//        }

        return status;
    }

    default void postUpdate(List<String> updatedColumnNameList, List<ID> idList) {
    }

    @Transactional
    @AuditTrailEntry(AuditTrailType.SWAP)
    @Audit("AUDIT.SWAP")
    default void swap(String column, ID firstId, ID secondId) {
        T first = this.getDao().selectById(firstId);
        T second = this.getDao().selectById(secondId);
        Field field = ReflectionUtils.findField(this.getDao().getType(), column);
        if (field == null) {
            String key = column.toLowerCase();
            String firstValue = (String)first.getExt$().get(key);
            first.setExt$Item(key, (String)second.getExt$().get(key));
            second.setExt$Item(key, firstValue);
        } else {
            Object firstValue = ReflectionUtils.invokeReadMethod(first, field.getName());
            Object secondValue = ReflectionUtils.invokeReadMethod(second, field.getName());
            ReflectionUtils.invokeWriteMethod(first, field.getName(), secondValue);
            ReflectionUtils.invokeWriteMethod(second, field.getName(), firstValue);
        }

        this.getDao().update(Arrays.asList(first, second), new String[]{column});
    }

    default List<T> selectListByFilter(SearchFilter filter, Order... orders) {
        MapperParameter parameter = new MapperParameter();
        parameter.setFilter(filter);
        Order[] var4 = orders;
        int var5 = orders.length;

        for(int var6 = 0; var6 < var5; ++var6) {
            Order order = var4[var6];
            parameter.setOrderParam(order.getColumn(), order.getDirection());
        }

        return this.selectList(parameter);
    }

    default List<T> selectListByFilter(SearchFilter filter, RestJsonWrapperBean wrapper) {
        MapperParameter parameter = wrapper == null ? new MapperParameter() : wrapper.extractMapFilter();
        parameter.setFilter(filter);
        return this.selectList(parameter);
    }

    default List<T> selectList(MapperParameter parameter) {
        List<Map<String, Object>> mapList = this.getDao().selectByCondition(parameter);
        return mapList.isEmpty() ? CollectionUtils.emptyList() : (List)mapList.parallelStream().map((m) -> {
            return (Persistable)PersistableHelper.mapToPersistable(m, this.getDao().getType());
        }).collect(Collectors.toList());
    }

    default T selectFirst(MapperParameter parameter) {
        PageRequest page = new PageRequest();
        page.setPageSize(1);
        PageRowBounds rowBounds = new PageRowBounds(page);
        Page<Map<String, Object>> mapList = MybatisPageHelper.get(rowBounds, () -> {
            return this.getDao().selectByCondition(parameter);
        });
        return mapList.getRows().isEmpty() ? null : (T) PersistableHelper.mapToPersistable((Map)mapList.getRows().get(0), this.getDao().getType());
    }

    default T selectFirstByFilter(SearchFilter filter, RestJsonWrapperBean wrapper) {
        MapperParameter parameter = wrapper == null ? new MapperParameter() : wrapper.extractMapFilter();
        parameter.setFilter(filter);
        return this.selectFirst(parameter);
    }

    default T selectFirstByFilter(SearchFilter filter, Order... orders) {
        MapperParameter parameter = new MapperParameter();
        parameter.setFilter(filter);
        Order[] var4 = orders;
        int var5 = orders.length;

        for(int var6 = 0; var6 < var5; ++var6) {
            Order order = var4[var6];
            parameter.setOrderParam(order.getColumn(), order.getDirection());
        }

        return this.selectFirst(parameter);
    }

    default String getSinoAttrsMybatisMapper() {
        GenericMapper<ID> mapper = this.getDao().getMapper();
        Class<? extends GenericMapper<ID>> mapperClazz = (Class<? extends GenericMapper<ID>>) ClassUtils.getRawType(mapper.getClass());
        return MybatisHelper.getMapperContent(mapperClazz);
    }

//    default Page<CoreControllerRequestEntryDTO> getSinoAttrsReqEntryPagination() {
//        Class clazz = ClassUtils.getRawType(this.getClass());
//
//        while(!Object.class.equals(clazz)) {
//            String className = clazz.getSimpleName();
//            String packageName = clazz.getPackage().getName();
//            String controllerClassName = StringUtils.removeEnd(StringUtils.removeEnd(packageName, ".impl"), ".service") + ".controller." + StringUtils.removeEnd(StringUtils.removeEnd(className, "Impl"), "Service") + "Controller";
//
//            try {
//                Class<?> controllerClazz = ClassUtils.getClass(controllerClassName);
//                Object proxy = ApplicationContextHelper.getBean(controllerClazz);
//                Class<?> actualClazz = ClassUtils.getRawType(proxy.getClass());
//                Object instance = ClassUtils.newInstance(actualClazz);
//                RequestMapping requestMapping = (RequestMapping)ReflectionUtils.getAnnotation(actualClazz, RequestMapping.class);
//                List<Method> methodList = ReflectionUtils.getAnnotatedMethodList(actualClazz, RequestMapping.class);
//                List<CoreControllerRequestEntryDTO> requestEntryList = new ArrayList();
//                Iterator var12 = methodList.iterator();
//
//                while(var12.hasNext()) {
//                    Method method = (Method)var12.next();
//                    CoreControllerRequestEntryDTO entry = new CoreControllerRequestEntryDTO();
//
//                    try {
//                        entry.setPageUrl((String)ReflectionUtils.invokeMethod(method, instance));
//                    } catch (ReflectionException var16) {
//                        continue;
//                    }
//
//                    Log log = (Log)ReflectionUtils.getMethodAnnotation(actualClazz, method, Log.class);
//                    if (log != null) {
//                        entry.setRequestName(log.value());
//                    } else {
//                        entry.setRequestName(method.getName());
//                    }
//
//                    entry.setRequestUrl(requestMapping.value()[0] + ((RequestMapping)method.getAnnotation(RequestMapping.class)).value()[0]);
//                    requestEntryList.add(entry);
//                }
//
//                if (StringUtils.isEmpty(ServletUtils.getCurrentRequest().getParameter("filter_page_config"))) {
//                    List<String> requestUrlList = (List)requestEntryList.stream().map((e) -> {
//                        return e.getRequestUrl();
//                    }).collect(Collectors.toList());
//                    CoreAdminPageConfigService pageConfigService = (CoreAdminPageConfigService)ApplicationContextHelper.getBean(CoreAdminPageConfigService.class);
//                    List<CoreAdminPageConfigBean> pageConfigList = pageConfigService.getDao().selectListByOneColumnValues(requestUrlList, "REQUESTURI", new Order[0]);
//                    requestEntryList.removeIf((e) -> {
//                        return pageConfigList.stream().anyMatch((c) -> {
//                            return e.getRequestUrl().equals(c.getRequestUri());
//                        });
//                    });
//                }
//
//                return new Pagination(requestEntryList);
//            } catch (Exception var17) {
//                var17.printStackTrace();
//                clazz = clazz.getSuperclass();
//            }
//        }
//
//        return new Pagination();
//    }

    default JSONObject print(ID id) {
        throw new UnsupportedException("You should implements this method if required.");
    }

    default Long selectProcId(ID id) {
        if (GenericAuditableService.class.isAssignableFrom(this.getClass())) {
            GenericAuditableService auditableService = (GenericAuditableService)this;
            T item = this.getDao().selectById(id);
            List<Pair<ID, Long>> pairList = auditableService.getBpmnProcIdList(Arrays.asList(item));
            return pairList.isEmpty() ? null : (Long)((Pair)pairList.get(0)).getSecond();
        } else {
            return null;
        }
    }

    default <V> void updateColumn(ID id, String columnName, V value) {
        T item = (T) ClassUtils.newInstance(this.getDao().getType());
        item.setId(id);
        BeanPropertyDescriptor propertyDescriptor = BeanPropertyHelper.getBeanPropertyDescriptor(this.getDao().getType(), columnName);
        if (propertyDescriptor == null) {
            item.setExt$Item(columnName, (String)ConvertUtils.convert(value, String.class));
        } else {
            propertyDescriptor.setPropertyValue(item, value);
        }

        this.getDao().update(item, new String[]{columnName});
    }
}
