/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.support.base.resource;

import com.alibaba.fastjson.JSONObject;
import net.sinodawn.framework.data.page.Page;
import net.sinodawn.framework.log.annotation.Log;
import net.sinodawn.framework.log.annotation.LogType;
import net.sinodawn.framework.restful.data.RestJsonWrapperBean;
import net.sinodawn.framework.restful.data.RestValidationResultBean;
import net.sinodawn.framework.support.base.bean.CoreServiceStatusDTO;
import net.sinodawn.framework.support.base.service.GenericService;
import net.sinodawn.framework.support.domain.Persistable;
import net.sinodawn.framework.validator.permission.annotation.ValidatePermissionWith;
import net.sinodawn.framework.validator.permission.support.DefaultAvailablePermissionValidator;
import net.sinodawn.framework.validator.permission.support.DefaultUpdatablePermissionValidator;
import org.springframework.http.HttpStatus;
import org.springframework.web.bind.annotation.*;

import java.io.Serializable;
import java.util.Objects;

/**
 * 通用resource接口，默认实现了常规CRUD
 * @param <S> 对应service类
 * @param <T> 对应持久层Bean
 * @param <ID> 持久层Bean　ID
 */
@SuppressWarnings("unused")
public interface GenericResource<S extends GenericService<T, ID>, T extends Persistable<ID>, ID extends Serializable> {
    S getService();

    @Log(
            value = "通用查询",
            type = LogType.SELECT
    )
    @RequestMapping(
            value = {"/queries"},
            method = {RequestMethod.POST}
    )
    default Page<T> selectPagination(RestJsonWrapperBean wrapper) {
        return this.getService().selectPagination(wrapper);
    }

    @Log(
            value = "编辑页查询",
            type = LogType.SELECT
    )
    @RequestMapping(
            value = {"/queries/raw"},
            method = {RequestMethod.POST}
    )
    default Page<T> selectRawPagination(RestJsonWrapperBean wrapper) {
        return this.getService().selectRawPagination(wrapper);
    }

    @Log(
            value = "详细信息查询",
            type = LogType.SELECT
    )
    @RequestMapping(
            value = {"/{id}"},
            method = {RequestMethod.GET}
    )
    @ValidatePermissionWith(
            value = DefaultAvailablePermissionValidator.class,
            table = "#this.getService().getDao().getTable()"
    )
    default T selectDetail(@PathVariable ID id) {
        return this.getService().selectDetail(id);
    }

    @Log(
            value = "单个字段查询",
            type = LogType.SELECT
    )
    @RequestMapping(
            value = {"/{id}/columns/{column}"},
            method = {RequestMethod.GET}
    )
    @ValidatePermissionWith(
            value = DefaultAvailablePermissionValidator.class,
            table = "#this.getService().getDao().getTable()"
    )
    default String selectColumnById(@PathVariable ID id, @PathVariable String column) {
        return this.getService().selectColumnById(id, column);
    }

    @Log(
            value = "获取状态信息",
            type = LogType.SELECT
    )
    @RequestMapping(
            value = {"/{id}/info/status", "/info/status"},
            method = {RequestMethod.GET}
    )
    default CoreServiceStatusDTO selectStatus(@PathVariable(required = false) ID id, @RequestParam(required = false) String currentStatusCode) {
        return this.getService().selectStatus(id, currentStatusCode);
    }

    @Log(
            value = "删除",
            type = LogType.DELETE
    )
    @RequestMapping(
            method = {RequestMethod.DELETE}
    )
    @ValidatePermissionWith(
            value = DefaultUpdatablePermissionValidator.class,
            table = "#this.getService().getDao().getTable()"
    )
    default void delete(RestJsonWrapperBean wrapper) {
        this.getService().delete(wrapper);
    }

    @Log(
            value = "新增",
            type = LogType.INSERT
    )
    @RequestMapping(
            method = {RequestMethod.POST}
    )
    default ID insert(RestJsonWrapperBean wrapper) {
        return this.getService().insert(wrapper);
    }

    @Log(
            value = "实时修改",
            type = LogType.UPDATE
    )
    @RequestMapping(
            value = {"/instant"},
            method = {RequestMethod.PUT}
    )
    @ResponseStatus(HttpStatus.OK)
    default void instantSave(RestJsonWrapperBean wrapper) {
        this.getService().instantSave(wrapper);
    }

    @Log(
            value = "修改",
            type = LogType.UPDATE
    )
    @RequestMapping(
            value = {"/{id}"},
            method = {RequestMethod.PUT}
    )
    @ResponseStatus(HttpStatus.OK)
    @ValidatePermissionWith(
            value = DefaultUpdatablePermissionValidator.class,
            table = "#this.getService().getDao().getTable()",
            category = "normal"
    )
    default void save(@PathVariable ID id, RestJsonWrapperBean wrapper) {
        this.getService().save(id, wrapper);
    }

    @Log(
            value = "查询页查询",
            type = LogType.SELECT
    )
    @RequestMapping(
            value = {"/queries/searchable"},
            method = {RequestMethod.POST}
    )
    default Page<T> selectSearchablePagination(RestJsonWrapperBean wrapper) {
        return this.getService().selectSearchablePagination(wrapper);
    }

    @Log(
            value = "查询角色下的数据",
            type = LogType.SELECT
    )
    @RequestMapping(
            value = {"/queries/roles/{roleId}"},
            method = {RequestMethod.POST}
    )
    default Page<T> selectByRole(@PathVariable Long roleId, RestJsonWrapperBean wrapper) {
        return this.getService().selectByRole(roleId, wrapper);
    }

    @Log(
            value = "验证数据准确性",
            type = LogType.VALIDATE
    )
    @RequestMapping(
            value = {"/{id}/action/validate"},
            method = {RequestMethod.POST}
    )
    default RestValidationResultBean validate(@PathVariable ID id, RestJsonWrapperBean wrapper) {
        return this.getService().validate(id, wrapper);
    }

    @Log(
            value = "验证数据唯一性",
            type = LogType.VALIDATE
    )
    @RequestMapping(
            value = {"/{id}/action/validate-unique", "/{id}/action/validate-unique/{columnName}/{columnValue}", "/action/validate-unique/{columnName}/{columnValue}"},
            method = {RequestMethod.POST}
    )
    default RestValidationResultBean validateUnique(@PathVariable(required = false) ID id, @PathVariable(required = false) String columnName, @PathVariable(required = false) String columnValue) {
        return this.getService().validateUnique(id, columnName, columnValue);
    }

    @Log(
            value = "验证数据唯一性",
            type = LogType.VALIDATE
    )
    @RequestMapping(
            value = {"/action/validate-unique"},
            method = {RequestMethod.POST}
    )
    default RestValidationResultBean validateUnique(RestJsonWrapperBean wrapper) {
        return this.getService().validateUnique(wrapper);
    }

    @Log(
            value = "互换对应的值",
            type = LogType.UPDATE
    )
    @RequestMapping(
            value = {"/action/swap/{column}/{firstId}/{secondId}", "/action/swap/{firstId}/{secondId}"},
            method = {RequestMethod.PUT}
    )
    default void swap(@PathVariable(required = false) String column, @PathVariable ID firstId, @PathVariable ID secondId) {
        this.getService().swap(Objects.toString(column, "ORDERNO"), firstId, secondId);
    }

    @Log(
            value = "查询当前业务对应的Mapper的内容",
            type = LogType.SELECT
    )
    @RequestMapping(
            value = {"/sino-attrs/mybatis-mapper"},
            method = {RequestMethod.GET}
    )
    default String getSinoAttrsMybatisMapper() {
        return this.getService().getSinoAttrsMybatisMapper();
    }

//    @Log(
//            value = "查询当前业务下功能",
//            type = LogType.SELECT
//    )
//    @RequestMapping(
//            value = {"/sino-attrs/reqentries/queries"},
//            method = {RequestMethod.POST}
//    )
//    default Page<CoreControllerRequestEntryDTO> getSinoAttrsReqEntryPagination() {
//        return this.getService().getSinoAttrsReqEntryPagination();
//    }

    @Log(
            value = "打印详情",
            type = LogType.PRINT
    )
    @RequestMapping(
            value = {"/{id}/action/print"},
            method = {RequestMethod.GET}
    )
    @ValidatePermissionWith(
            value = DefaultAvailablePermissionValidator.class,
            table = "#this.getService().getDao().getTable()",
            category = "normal"
    )
    default JSONObject print(@PathVariable ID id) {
        return this.getService().print(id);
    }
}
