/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.support.base.dao;

import net.sinodawn.framework.aspect.annotation.IgnoreAop;
import net.sinodawn.framework.context.SinoAopContext;
import net.sinodawn.framework.data.page.Page;
import net.sinodawn.framework.data.page.Pageable;
import net.sinodawn.framework.database.context.EntityColumnContext;
import net.sinodawn.framework.database.context.EntityContext;
import net.sinodawn.framework.database.sql.Order;
import net.sinodawn.framework.exception.UnsupportedException;
import net.sinodawn.framework.exception.database.NotUniqueResultJdbcException;
import net.sinodawn.framework.mybatis.mapper.MapperParameter;
import net.sinodawn.framework.mybatis.mapper.MatchPattern;
import net.sinodawn.framework.mybatis.mapper.SearchFilter;
import net.sinodawn.framework.support.PersistableMetadataHelper;
import net.sinodawn.framework.support.base.mapper.GenericMapper;
import net.sinodawn.framework.support.domain.Persistable;
import net.sinodawn.framework.utils.*;

import java.io.Serializable;
import java.util.*;
import java.util.stream.Collectors;

@SuppressWarnings({"unused", "unchecked"})

public interface GenericDao<T extends Persistable<ID>, ID extends Serializable> {
    int STATEMENT_MAX_BATCH_BIND_VALUES = 400;

    @IgnoreAop
    default String getTable() {
        return PersistableMetadataHelper.getTableName(this.getType());
    }

    Class<T> getType();

    EntityContext getEntityContext();

    default GenericMapper<ID> getMapper() {
        throw new UnsupportedException();
    }

    void insert(T item);

    void insert(List<T> itemList);

    void delete(ID id);

    void deleteByIdList(List<ID> idList);

    void deleteBy(List<T> itemList, String... searchColNames);

    default void deleteBy(T item, String... searchColNames) {
        this.deleteBy(Arrays.asList(item), searchColNames);
    }

    void update(T item, List<String> updateColNames, String... searchColNames);

    default void update(T item, String... updateColNames) {
        this.update(item, ArrayUtils.asList(updateColNames));
    }

    void updateByIds(T item, List<ID> idList, String... updateColNames);

    default void update(List<T> itemList, List<String> updateColNames, String... searchColNames) {
        if (ArrayUtils.isEmpty(searchColNames)) {
            this.update(itemList, (String[])updateColNames.toArray(new String[0]));
        } else {
            Iterator var4 = itemList.iterator();

            while(var4.hasNext()) {
                T item = (T) var4.next();
                this.update(item, updateColNames, searchColNames);
            }
        }

    }

    void update(List<T> itemList, String... updateColNames);

    default void update(T updateItem, T searchItem) {
        List<ID> idList = this.selectIdList(searchItem);
        if (!idList.isEmpty()) {
            this.updateByIds(updateItem, idList);
        }

    }

    List<T> updateIfChanged(List<T> rawOrProxyItemList);

    default T updateIfChanged(T rawOrProxyItem) {
        List<T> itemList = this.updateIfChanged(Arrays.asList(rawOrProxyItem));
        return itemList.isEmpty() ? null : itemList.get(0);
    }

    default void updateCreatedBy(T item) {
        this.updateCreatedBy(Arrays.asList(item));
    }

    void updateCreatedBy(List<T> itemList);

    T selectByIdIfPresent(ID id);

    default T selectById(ID id) {
        GenericDao<T, ID> dao = (GenericDao)SinoAopContext.currentProxy();
        return Objects.requireNonNull(dao.selectByIdIfPresent(id), "SINO.EXCEPTION.NO_VALUE_EXISTS");
    }

    T selectOneIfPresent(T item, String... selectColNames);

    default T selectOne(T item, String... selectColNames) {
        return Objects.requireNonNull(this.selectOneIfPresent(item, selectColNames), "SINO.EXCEPTION.NO_VALUE_EXISTS");
    }

    <V> V selectColumnById(ID id, String colName, Class<V> colType);

    <V> List<V> selectColumnList(T item, List<String> searchColNames, String colName, Class<V> colType, Order... orders);

    default <V> List<V> selectColumnList(T item, String colName, Class<V> colType, Order... orders) {
        return this.selectColumnList(item, CollectionUtils.emptyList(), colName, colType, orders);
    }

    default <V> V selectColumnIfPresent(T item, String colName, Class<V> colType) {
        List<V> columnList = this.selectColumnList(item, colName, colType);
        if (columnList.size() > 1) {
            throw new NotUniqueResultJdbcException();
        } else {
            return columnList.isEmpty() ? null : columnList.get(0);
        }
    }

    default <V> V selectColumn(T item, String colName, Class<V> colType) {
        return Objects.requireNonNull(this.selectColumnIfPresent(item, colName, colType), "SINO.EXCEPTION.NO_VALUE_EXISTS");
    }

    default List<ID> selectIdList(T item, Order... orders) {
        EntityColumnContext idContext = this.getEntityContext().getIdContext();
        return (List<ID>)this.selectColumnList(item, idContext.getColumnName(), idContext.getType(), orders);
    }

    default List<ID> selectIdList(T item, List<String> searchColNames, Order... orders) {
        EntityColumnContext idContext = this.getEntityContext().getIdContext();
        return (List<ID>)this.selectColumnList(item, searchColNames, idContext.getColumnName(), idContext.getType(), orders);
    }

    default ID selectIdIfPresent(T item) {
        EntityColumnContext idContext = this.getEntityContext().getIdContext();
        return (ID) this.selectColumnIfPresent(item, idContext.getColumnName(), idContext.getType());
    }

    default ID selectId(T item) {
        return Objects.requireNonNull(this.selectIdIfPresent(item), "SINO.EXCEPTION.NO_VALUE_EXISTS");
    }

    List<T> selectAll(List<Order> orderList, String... selectColNames);

    default List<T> selectAll(Order... orders) {
        return this.selectAll(Arrays.asList(orders));
    }

    default List<T> selectAll(String... selectColNames) {
        return this.selectAll(CollectionUtils.emptyList(), selectColNames);
    }

    default List<T> selectAll() {
        return this.selectAll(CollectionUtils.emptyList());
    }

    List<T> selectListByIds(List<ID> idList, List<String> selectColNames, Order... orders);

    default List<T> selectListByIds(List<ID> idList, Order... orders) {
        return this.selectListByIds(idList, CollectionUtils.emptyList(), orders);
    }

    <V> List<V> selectColumnsByIds(List<ID> idList, String colName, Class<V> colType, Order... orders);

    List<T> selectList(T item, List<String> searchColNames, List<String> selectColNames, Order... orders);

    default List<T> selectList(T item, List<String> selectColNames, Order... orders) {
        return this.selectList(item, CollectionUtils.emptyList(), selectColNames, orders);
    }

    default List<T> selectList(T item, Order... orders) {
        return this.selectList(item, CollectionUtils.emptyList(), orders);
    }

    List<T> selectList(List<T> itemList, List<String> searchColNames, List<String> selectColNames, Order... orders);

    <V> List<V> selectList(List<T> itemList, List<String> searchColNames, String colName, Class<V> colType, Order... orders);

    default <V> List<V> selectList(List<T> itemList, String colName, Class<V> colType, Order... orders) {
        return this.selectList(itemList, CollectionUtils.emptyList(), colName, colType, orders);
    }

    default <V> List<T> selectListByOneColumnValues(List<V> valueList, String colName, List<String> selectColNames, Order... orders) {
        List<T> itemList = valueList.stream().map((v) -> {
            T item = ClassUtils.newInstance(this.getType());
            this.getEntityContext().getColumnContextList().stream().filter((c) -> {
                return c.getColumnName().equalsIgnoreCase(colName);
            }).forEach((p) -> {
                ReflectionUtils.invokeMethod(p.getPropertyDescriptor().getWriteMethod(), item, new Object[]{ConvertUtils.convert(v, p.getType())});
            });
            return item;
        }).collect(Collectors.toList());
        return this.selectList(itemList, Arrays.asList(colName), selectColNames, orders);
    }

    default <V> List<T> selectListByOneColumnValues(List<V> valueList, String colName, Order... orders) {
        return this.selectListByOneColumnValues(valueList, colName, CollectionUtils.emptyList(), orders);
    }

    default <V> List<T> selectListByOneColumnValue(V value, String colName, List<String> selectColNames, Order... orders) {
        return this.selectListByOneColumnValues(Arrays.asList(value), colName, selectColNames, orders);
    }

    default <V> List<T> selectListByOneColumnValue(V value, String colName, Order... orders) {
        return this.selectListByOneColumnValues(Arrays.asList(value), colName, CollectionUtils.emptyList(), orders);
    }

    default List<ID> selectIdList(List<T> itemList, List<String> searchColNames, Order... orders) {
        EntityColumnContext idContext = this.getEntityContext().getIdContext();
        return (List<ID>)this.selectList(itemList, searchColNames, idContext.getColumnName(), idContext.getType(), orders);
    }

    default List<ID> selectIdList(List<T> item, Order... orders) {
        EntityColumnContext idContext = this.getEntityContext().getIdContext();
        return (List<ID>)this.selectList(item, CollectionUtils.emptyList(), idContext.getColumnName(), idContext.getType(), orders);
    }

    T selectFirstIfPresent(T item, List<String> searchColNames, List<String> selectColNames, Order... orders);

    default T selectFirst(T item, List<String> searchColNames, List<String> selectColNames, Order... orders) {
        return Objects.requireNonNull(this.selectFirstIfPresent(item, searchColNames, selectColNames, orders), "SINO.EXCEPTION.NO_VALUE_EXISTS");
    }

    default T selectFirstIfPresent(T item, List<String> selectColNames, Order... orders) {
        return this.selectFirstIfPresent(item, CollectionUtils.emptyList(), selectColNames, orders);
    }

    default T selectFirst(T item, List<String> selectColNames, Order... orders) {
        return Objects.requireNonNull(this.selectFirstIfPresent(item, selectColNames, orders), "SINO.EXCEPTION.NO_VALUE_EXISTS");
    }

    default T selectFirstIfPresent(T item, Order... orders) {
        return this.selectFirstIfPresent(item, CollectionUtils.emptyList(), CollectionUtils.emptyList(), orders);
    }

    default T selectFirst(T item, Order... orders) {
        return Objects.requireNonNull(this.selectFirstIfPresent(item, orders), "SINO.EXCEPTION.NO_VALUE_EXISTS");
    }

    List<T> selectUnionList(T item, List<String> searchColNames, List<String> selectColNames, Order... orders);

    default List<T> selectUnionList(T item, List<String> selectColNames, Order... orders) {
        return this.selectUnionList(item, CollectionUtils.emptyList(), selectColNames, orders);
    }

    default List<T> selectUnionList(T item, Order... orders) {
        return this.selectUnionList(item, CollectionUtils.emptyList(), CollectionUtils.emptyList(), orders);
    }

    default T selectOneUnionIfPresent(T item, List<String> searchColNames, List<String> selectColNames, Order... orders) {
        List<T> resultList = this.selectUnionList(item, searchColNames, selectColNames, orders);
        if (resultList.isEmpty()) {
            return null;
        } else if (resultList.size() > 1) {
            throw new NotUniqueResultJdbcException();
        } else {
            return resultList.get(0);
        }
    }

    default T selectOneUnion(T item, List<String> searchColNames, List<String> selectColNames, Order... orders) {
        return Objects.requireNonNull(this.selectOneUnionIfPresent(item, searchColNames, selectColNames, orders), "SINO.EXCEPTION.NO_VALUE_EXISTS");
    }

    default T selectOneUnionIfPresent(T item, List<String> selectColNames, Order... orders) {
        return this.selectOneUnionIfPresent(item, CollectionUtils.emptyList(), selectColNames, orders);
    }

    default T selectOneUnion(T item, List<String> selectColNames, Order... orders) {
        return Objects.requireNonNull(this.selectOneUnionIfPresent(item, selectColNames, orders));
    }

    default T selectOneUnionIfPresent(T item, Order... orders) {
        return this.selectOneUnionIfPresent(item, CollectionUtils.emptyList(), CollectionUtils.emptyList(), orders);
    }

    default T selectOneUnion(T item, Order... orders) {
        return Objects.requireNonNull(this.selectOneUnionIfPresent(item, orders));
    }

    int countBy(T item, String... searchColNames);

    default int count() {
        return this.countBy((T) ClassUtils.newInstance(this.getType()));
    }

    int countBy(List<T> itemList, String... searchColNames);

    Page<T> selectPaginationByMybatis(String mybatisStatementId, Pageable pageable);

    default Map<String, Object> selectDetail(ID id) {
        MapperParameter parameter = new MapperParameter();
        parameter.setFilter(SearchFilter.instance().match("ID", id).filter(MatchPattern.EQ));
        List<Map<String, Object>> mapList = this.getMapper().selectByCondition(parameter);
        return mapList.isEmpty() ? CollectionUtils.emptyMap() : (Map)mapList.get(0);
    }

    default List<Map<String, Object>> selectByCondition(MapperParameter parameter) {
        parameter.setTableName(PersistableMetadataHelper.getTableName(this.getType()));
        return this.getMapper().selectByCondition(parameter);
    }

    default void cacheEvict(T oldItem, T newItem) {
    }
}
