/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.support.auditable.service;

import com.alibaba.fastjson.JSON;
import net.sinodawn.framework.at.annotation.AuditTrailEntry;
import net.sinodawn.framework.at.annotation.AuditTrailType;
import net.sinodawn.framework.audit.aunnotation.Audit;
import net.sinodawn.framework.beans.BeanPropertyDescriptor;
import net.sinodawn.framework.beans.BeanPropertyHelper;
import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.context.LocalContextHelper;
import net.sinodawn.framework.data.Pair;
import net.sinodawn.framework.data.page.Page;
import net.sinodawn.framework.database.context.EntityColumnContext;
import net.sinodawn.framework.database.core.DatabaseManager;
import net.sinodawn.framework.exception.core.ApplicationWarningException;
import net.sinodawn.framework.mybatis.mapper.MapperParameter;
import net.sinodawn.framework.mybatis.page.PageRowBounds;
import net.sinodawn.framework.restful.data.RestJsonWrapperBean;
import net.sinodawn.framework.support.PersistableHelper;
import net.sinodawn.framework.support.auditable.bean.CoreBpmnParameterDTO;
import net.sinodawn.framework.support.auditable.bean.CoreBpmnTaskStatusDTO;
import net.sinodawn.framework.support.base.service.GenericService;
import net.sinodawn.framework.support.domain.Auditable;
import net.sinodawn.framework.support.domain.Suspendable;
import net.sinodawn.framework.utils.*;
import net.sinodawn.module.mdm.user.bean.CoreUserBean;
import net.sinodawn.module.mdm.user.service.CoreUserService;
import net.sinodawn.module.sys.bpmn.bean.*;
import net.sinodawn.module.sys.bpmn.diagram.BpmnDiagramHelper;
import net.sinodawn.module.sys.bpmn.diagram.ProcessStatus;
import net.sinodawn.module.sys.bpmn.diagram.attribute.CandidatorFilterStrategy;
import net.sinodawn.module.sys.bpmn.diagram.attribute.NextCandidatorOptStrategy;
import net.sinodawn.module.sys.bpmn.engine.CoreBpmnRuntimeService;
import net.sinodawn.module.sys.bpmn.engine.CoreBpmnRuntimeSource;
import net.sinodawn.module.sys.bpmn.engine.cache.BpmnRuntimeCacheProvider;
import net.sinodawn.module.sys.bpmn.exception.BpmnException;
import net.sinodawn.module.sys.role.bean.CoreRoleUserBean;
import net.sinodawn.module.sys.role.service.CoreRoleService;
import net.sinodawn.module.sys.role.service.CoreRoleUserService;
import org.dom4j.Element;
import org.springframework.transaction.annotation.Transactional;

import java.io.Serializable;
import java.time.temporal.Temporal;
import java.util.*;
import java.util.function.Supplier;
import java.util.stream.Collectors;

public interface GenericAuditableService<T extends Auditable<ID>, ID extends Serializable> extends GenericService<T, ID> {
   default Page<T> selectAuditablePagination(RestJsonWrapperBean wrapper) {
      MapperParameter parameter = wrapper.extractMapFilter();
      PageRowBounds rowBounds = wrapper.extractPageRowBounds();
      if (Auditable.class.isAssignableFrom(this.getDao().getType())) {
         parameter.setProcessStatus("PROCESSSTATUS", ProcessStatus.APPROVE);
      }

      parameter.setAuditAuthority();
      parameter.setOrgAuthority();
      parameter.setAuditableQueries();
      if (!wrapper.getAuthorityList().isEmpty()) {
         wrapper.getAuthorityList().forEach((a) -> {
            parameter.setAuthorityParameter(a);
         });
      }

      return this.selectPagination(parameter, rowBounds);
   }

   default Page<T> selectWithdrawablePagination(RestJsonWrapperBean wrapper) {
      MapperParameter parameter = wrapper.extractMapFilter();
      PageRowBounds rowBounds = wrapper.extractPageRowBounds();
      if (Auditable.class.isAssignableFrom(this.getDao().getType())) {
         parameter.setProcessStatus("PROCESSSTATUS", ProcessStatus.APPROVE);
      }

      parameter.setOrgAuthority();
      parameter.setWithdrawableQueries();
      if (!wrapper.getAuthorityList().isEmpty()) {
         wrapper.getAuthorityList().forEach((a) -> {
            parameter.setAuthorityParameter(a);
         });
      }

      return this.selectPagination(parameter, rowBounds);
   }

   default Page<T> selectUndoablePagination(RestJsonWrapperBean wrapper) {
      MapperParameter parameter = wrapper.extractMapFilter();
      PageRowBounds rowBounds = wrapper.extractPageRowBounds();
      Class<T> type = this.getDao().getType();
      if (Auditable.class.isAssignableFrom(this.getDao().getType())) {
         parameter.setProcessStatus("PROCESSSTATUS", ProcessStatus.DONE);
      }

      if (Suspendable.class.isAssignableFrom(type)) {
         parameter.setSuspendedFlag("0");
      }

      parameter.setOrgAuthority();
      parameter.setUndoableQueries();
      if (!wrapper.getAuthorityList().isEmpty()) {
         wrapper.getAuthorityList().forEach((a) -> {
            parameter.setAuthorityParameter(a);
         });
      }

      return this.selectPagination(parameter, rowBounds);
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_START)
   @Audit("AUDIT.PROCESS_START")
   default List<CoreBpmnInstanceStatusDTO<ID>> startProcess(RestJsonWrapperBean wrapper) {
      List<T> itemList = wrapper.parse(this.getDao().getType());
      if (itemList.isEmpty()) {
         return CollectionUtils.emptyList();
      } else {
         this.checkAndUpdateVersion(itemList);
         CoreBpmnRuntimeService runtimeService = ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
         List<ID> itemIdList = (List)itemList.stream().map((i) -> (Serializable)i.getId()).collect(Collectors.toList());
         List<T> selectItemList = this.getBpmnVarsItemList(itemIdList);
         List<CoreBpmnRuntimeSource<T, ID>> runtimeSourceList = this.parseSubmittableRuntimeSource(itemIdList, CoreBpmnParameterDTO.of(wrapper), this.getBpmnVars(wrapper, selectItemList));
         return runtimeService.startProcess(runtimeSourceList);
      }
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_START)
   @Audit("AUDIT.PROCESS_START")
   default List<CoreBpmnInstanceStatusDTO<ID>> startProcess(List<ID> itemIdList, CoreBpmnParameterDTO parameter) {
      if (itemIdList.isEmpty()) {
         return CollectionUtils.emptyList();
      } else {
         CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
         List<T> itemList = this.getBpmnVarsItemList(itemIdList);
         List<CoreBpmnRuntimeSource<T, ID>> runtimeSourceList = this.parseSubmittableRuntimeSource(itemIdList, parameter, this.getBpmnVars((RestJsonWrapperBean)null, itemList));
         return runtimeService.startProcess(runtimeSourceList);
      }
   }

   /** @deprecated */
   @Deprecated
   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_START)
   @Audit("AUDIT.PROCESS_START")
   default List<CoreBpmnInstanceStatusDTO<ID>> startProcess(List<ID> itemIdList, String comment) {
      CoreBpmnParameterDTO parameter = new CoreBpmnParameterDTO();
      parameter.setComment(comment);
      return this.startProcess(itemIdList, parameter);
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_WITHDRAW)
   @Audit("AUDIT.PROCESS_WITHDRAW")
   default List<CoreBpmnInstanceStatusDTO<ID>> withdrawProcess(RestJsonWrapperBean wrapper) {
      List<T> itemList = wrapper.parse(this.getDao().getType());
      if (itemList.isEmpty()) {
         return CollectionUtils.emptyList();
      } else {
         this.checkAndUpdateVersion(itemList);
         CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
         List<ID> itemIdList = (List)itemList.stream().map((i) -> {
            return (Serializable)i.getId();
         }).collect(Collectors.toList());
         List<T> selectItemList = this.getBpmnVarsItemList(itemIdList);
         List<CoreBpmnRuntimeSource<T, ID>> runtimeSourceList = this.parseWithdrawableRuntimeSource(itemIdList, CoreBpmnParameterDTO.of(wrapper), this.getBpmnVars(wrapper, selectItemList));
         return runtimeService.withdrawProcess(runtimeSourceList);
      }
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_WITHDRAW)
   @Audit("AUDIT.PROCESS_WITHDRAW")
   default List<CoreBpmnInstanceStatusDTO<ID>> withdrawProcess(List<ID> itemIdList, CoreBpmnParameterDTO parameter) {
      if (itemIdList.isEmpty()) {
         return CollectionUtils.emptyList();
      } else {
         CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
         List<T> itemList = this.getBpmnVarsItemList(itemIdList);
         List<CoreBpmnRuntimeSource<T, ID>> runtimeSourceList = this.parseWithdrawableRuntimeSource(itemIdList, parameter, this.getBpmnVars((RestJsonWrapperBean)null, itemList));
         return runtimeService.withdrawProcess(runtimeSourceList);
      }
   }

   /** @deprecated */
   @Deprecated
   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_WITHDRAW)
   @Audit("AUDIT.PROCESS_WITHDRAW")
   default List<CoreBpmnInstanceStatusDTO<ID>> withdrawProcess(List<ID> itemIdList, String comment) {
      CoreBpmnParameterDTO parameter = new CoreBpmnParameterDTO();
      parameter.setComment(comment);
      return this.withdrawProcess(itemIdList, parameter);
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_COMPLETE)
   @Audit("AUDIT.PROCESS_COMPLETE")
   default List<CoreBpmnInstanceStatusDTO<ID>> completeTask(RestJsonWrapperBean wrapper) {
      List<T> itemList = wrapper.parse(this.getDao().getType());
      if (itemList.isEmpty()) {
         return CollectionUtils.emptyList();
      } else {
         this.checkAndUpdateVersion(itemList);
         CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
         List<ID> itemIdList = (List)itemList.stream().map((i) -> {
            return (Serializable)i.getId();
         }).collect(Collectors.toList());
         List<T> selectItemList = this.getBpmnVarsItemList(itemIdList);
         List<CoreBpmnRuntimeSource<T, ID>> runtimeSourceList = this.parseAuditableRuntimeSource(itemIdList, CoreBpmnParameterDTO.of(wrapper), this.getBpmnVars(wrapper, selectItemList));
         return runtimeService.completeTask(runtimeSourceList);
      }
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_COMPLETE)
   @Audit("AUDIT.PROCESS_COMPLETE")
   default List<CoreBpmnInstanceStatusDTO<ID>> completeTask(List<ID> itemIdList, CoreBpmnParameterDTO parameter) {
      if (itemIdList.isEmpty()) {
         return CollectionUtils.emptyList();
      } else {
         CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
         List<T> itemList = this.getBpmnVarsItemList(itemIdList);
         List<CoreBpmnRuntimeSource<T, ID>> runtimeSourceList = this.parseAuditableRuntimeSource(itemIdList, parameter, this.getBpmnVars((RestJsonWrapperBean)null, itemList));
         return runtimeService.completeTask(runtimeSourceList);
      }
   }

   /** @deprecated */
   @Deprecated
   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_COMPLETE)
   @Audit("AUDIT.PROCESS_COMPLETE")
   default List<CoreBpmnInstanceStatusDTO<ID>> completeTask(List<ID> itemIdList, String comment) {
      CoreBpmnParameterDTO parameter = new CoreBpmnParameterDTO();
      parameter.setComment(comment);
      return this.completeTask(itemIdList, parameter);
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_TRANSFER)
   @Audit("AUDIT.PROCESS_TRANSFER")
   default void transferTask(RestJsonWrapperBean wrapper) {
      List<T> itemList = wrapper.parse(this.getDao().getType());
      if (!itemList.isEmpty()) {
         this.checkAndUpdateVersion(itemList);
         CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
         List<ID> itemIdList = (List)itemList.stream().map((i) -> {
            return (Serializable)i.getId();
         }).collect(Collectors.toList());
         List<CoreBpmnRuntimeSource<T, ID>> runtimeSourceList = this.parseAuditableRuntimeSource(itemIdList, CoreBpmnParameterDTO.of(wrapper), CollectionUtils.emptyMap());
         runtimeService.transferTask(runtimeSourceList);
      }
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_END)
   @Audit("AUDIT.PROCESS_END")
   default List<CoreBpmnInstanceStatusDTO<ID>> endTask(RestJsonWrapperBean wrapper) {
      List<T> itemList = wrapper.parse(this.getDao().getType());
      if (itemList.isEmpty()) {
         return CollectionUtils.emptyList();
      } else {
         this.checkAndUpdateVersion(itemList);
         CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
         List<ID> itemIdList = (List)itemList.stream().map((i) -> {
            return (Serializable)i.getId();
         }).collect(Collectors.toList());
         List<T> selectItemList = this.getBpmnVarsItemList(itemIdList);
         List<CoreBpmnRuntimeSource<T, ID>> runtimeSourceList = this.parseAuditableRuntimeSource(itemIdList, CoreBpmnParameterDTO.of(wrapper), this.getBpmnVars(wrapper, selectItemList));
         return runtimeService.endTask(runtimeSourceList);
      }
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_END)
   @Audit("AUDIT.PROCESS_END")
   default List<CoreBpmnInstanceStatusDTO<ID>> endTask(List<ID> itemIdList, CoreBpmnParameterDTO parameter) {
      if (itemIdList.isEmpty()) {
         return CollectionUtils.emptyList();
      } else {
         CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
         List<T> itemList = this.getBpmnVarsItemList(itemIdList);
         List<CoreBpmnRuntimeSource<T, ID>> runtimeSourceList = this.parseAuditableRuntimeSource(itemIdList, parameter, this.getBpmnVars((RestJsonWrapperBean)null, itemList));
         return runtimeService.endTask(runtimeSourceList);
      }
   }

   /** @deprecated */
   @Deprecated
   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_COMPLETE)
   @Audit("AUDIT.PROCESS_COMPLETE")
   default List<CoreBpmnInstanceStatusDTO<ID>> endTask(List<ID> itemIdList, String comment) {
      CoreBpmnParameterDTO parameter = new CoreBpmnParameterDTO();
      parameter.setComment(comment);
      return this.endTask(itemIdList, parameter);
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_REJECT)
   @Audit("AUDIT.PROCESS_REJECT")
   default List<CoreBpmnInstanceStatusDTO<ID>> rejectTask(RestJsonWrapperBean wrapper) {
      List<T> itemList = wrapper.parse(this.getDao().getType());
      if (itemList.isEmpty()) {
         return CollectionUtils.emptyList();
      } else {
         this.checkAndUpdateVersion(itemList);
         CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
         List<ID> itemIdList = (List)itemList.stream().map((i) -> {
            return (Serializable)i.getId();
         }).collect(Collectors.toList());
         List<T> selectItemList = this.getBpmnVarsItemList(itemIdList);
         List<CoreBpmnRuntimeSource<T, ID>> runtimeSourceList = this.parseAuditableRuntimeSource(itemIdList, CoreBpmnParameterDTO.of(wrapper), this.getBpmnVars(wrapper, selectItemList));
         return runtimeService.rejectTask(runtimeSourceList);
      }
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_REJECT)
   @Audit("AUDIT.PROCESS_REJECT")
   default List<CoreBpmnInstanceStatusDTO<ID>> rejectTask(List<ID> itemIdList, CoreBpmnParameterDTO parameter) {
      if (itemIdList.isEmpty()) {
         return CollectionUtils.emptyList();
      } else {
         List<T> itemList = this.getBpmnVarsItemList(itemIdList);
         List<CoreBpmnRuntimeSource<T, ID>> runtimeSourceList = this.parseAuditableRuntimeSource(itemIdList, parameter, this.getBpmnVars((RestJsonWrapperBean)null, itemList));
         CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
         return runtimeService.rejectTask(runtimeSourceList);
      }
   }

   /** @deprecated */
   @Deprecated
   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_COMPLETE)
   @Audit("AUDIT.PROCESS_COMPLETE")
   default List<CoreBpmnInstanceStatusDTO<ID>> rejectTask(List<ID> itemIdList, String comment) {
      CoreBpmnParameterDTO parameter = new CoreBpmnParameterDTO();
      parameter.setComment(comment);
      return this.rejectTask(itemIdList, parameter);
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_REJECT)
   @Audit("AUDIT.PROCESS_ODD_REJECT")
   default List<CoreBpmnInstanceStatusDTO<ID>> oddRejectTask(RestJsonWrapperBean wrapper) {
      List<T> itemList = wrapper.parse(this.getDao().getType());
      if (itemList.isEmpty()) {
         return CollectionUtils.emptyList();
      } else {
         this.checkAndUpdateVersion(itemList);
         CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
         List<ID> itemIdList = (List)itemList.stream().map((i) -> {
            return (Serializable)i.getId();
         }).collect(Collectors.toList());
         List<T> selectItemList = this.getBpmnVarsItemList(itemIdList);
         List<CoreBpmnRuntimeSource<T, ID>> runtimeSourceList = this.parseAuditableRuntimeSource(itemIdList, CoreBpmnParameterDTO.of(wrapper), this.getBpmnVars(wrapper, selectItemList));
         return runtimeService.oddRejectTask(runtimeSourceList);
      }
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_REJECT)
   @Audit("AUDIT.PROCESS_ODD_REJECT")
   default List<CoreBpmnInstanceStatusDTO<ID>> oddRejectTask(List<ID> itemIdList, CoreBpmnParameterDTO parameter) {
      if (itemIdList.isEmpty()) {
         return CollectionUtils.emptyList();
      } else {
         List<T> itemList = this.getBpmnVarsItemList(itemIdList);
         List<CoreBpmnRuntimeSource<T, ID>> runtimeSourceList = this.parseAuditableRuntimeSource(itemIdList, parameter, this.getBpmnVars((RestJsonWrapperBean)null, itemList));
         CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
         return runtimeService.oddRejectTask(runtimeSourceList);
      }
   }

   /** @deprecated */
   @Deprecated
   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_COMPLETE)
   @Audit("AUDIT.PROCESS_COMPLETE")
   default List<CoreBpmnInstanceStatusDTO<ID>> oddRejectTask(List<ID> itemIdList, String comment) {
      CoreBpmnParameterDTO parameter = new CoreBpmnParameterDTO();
      parameter.setComment(comment);
      return this.oddRejectTask(itemIdList, parameter);
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_UNDO)
   @Audit("AUDIT.PROCESS_UNDO")
   default List<CoreBpmnInstanceStatusDTO<ID>> undo(RestJsonWrapperBean wrapper) {
      List<T> itemList = wrapper.parse(this.getDao().getType());
      if (itemList.isEmpty()) {
         return CollectionUtils.emptyList();
      } else {
         this.checkAndUpdateVersion(itemList);
         CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
         List<ID> itemIdList = (List)itemList.stream().map((i) -> {
            return (Serializable)i.getId();
         }).collect(Collectors.toList());
         List<T> selectItemList = this.getBpmnVarsItemList(itemIdList);
         List<CoreBpmnRuntimeSource<T, ID>> runtimeSourceList = this.parseUndoableRuntimeSource(itemIdList, CoreBpmnParameterDTO.of(wrapper), this.getBpmnVars(wrapper, selectItemList));
         return runtimeService.undo(runtimeSourceList);
      }
   }

   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_UNDO)
   @Audit("AUDIT.PROCESS_UNDO")
   default List<CoreBpmnInstanceStatusDTO<ID>> undo(List<ID> itemIdList, CoreBpmnParameterDTO parameter) {
      if (itemIdList.isEmpty()) {
         return CollectionUtils.emptyList();
      } else {
         List<T> itemList = this.getBpmnVarsItemList(itemIdList);
         List<CoreBpmnRuntimeSource<T, ID>> runtimeSourceList = this.parseUndoableRuntimeSource(itemIdList, parameter, this.getBpmnVars((RestJsonWrapperBean)null, itemList));
         CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
         return runtimeService.undo(runtimeSourceList);
      }
   }

   /** @deprecated */
   @Deprecated
   @Transactional
   @AuditTrailEntry(AuditTrailType.PROCESS_COMPLETE)
   @Audit("AUDIT.PROCESS_COMPLETE")
   default List<CoreBpmnInstanceStatusDTO<ID>> undo(List<ID> itemIdList, String comment) {
      CoreBpmnParameterDTO parameter = new CoreBpmnParameterDTO();
      parameter.setComment(comment);
      return this.undo(itemIdList, parameter);
   }

   default CoreBpmnTaskStatusDTO selectBpmnTaskStatus(RestJsonWrapperBean wrapper) {
      List<ID> idList = wrapper.parseId((Class<ID>)this.getDao().getEntityContext().getIdContext().getType());
      if (idList.isEmpty()) {
         return new CoreBpmnTaskStatusDTO();
      } else {
         String pass = wrapper.getParamValue("bpmn_pass");
         if ("-1".equals(pass)) {
            return new CoreBpmnTaskStatusDTO();
         } else {
            List<CoreBpmnRuntimeSource<T, ID>> sourceList = this.parseRuntimeSource(idList, wrapper);
            if (sourceList.stream().anyMatch((s) -> {
               return ProcessStatus.DONE.name().equalsIgnoreCase(s.getOldItem().getProcessStatus());
            })) {
               return new CoreBpmnTaskStatusDTO();
            } else {
               CoreBpmnTaskStatusDTO taskStatus = null;

               try {
                  BpmnRuntimeCacheProvider.init(sourceList);
                  Iterator var6 = sourceList.iterator();

                  while(var6.hasNext()) {
                     CoreBpmnRuntimeSource<T, ID> source = (CoreBpmnRuntimeSource)var6.next();
                     CoreBpmnTaskStatusDTO sourceStatus = new CoreBpmnTaskStatusDTO();
                     Element currentElement = BpmnRuntimeCacheProvider.getBpmnRuntimeTaskElement(source);
                     sourceStatus.setAttachmentStrategy(BpmnDiagramHelper.getAttachmentStrategy(currentElement));
                     sourceStatus.setCommentRequired(BpmnDiagramHelper.isCommentRequiredTask(currentElement));
                     sourceStatus.setAuthRequired(BpmnDiagramHelper.isAuthRequiredUserTask(currentElement));
                     sourceStatus.setTransfer(BpmnDiagramHelper.isTransferTask(currentElement));
                     if ("1".equals(pass) && BpmnRuntimeCacheProvider.isLastTaskCandidator(source)) {
                        sourceStatus.setNextCandidatorOptStrategy(BpmnDiagramHelper.getNextCandidatorOptStrategy(currentElement));
                        if (NextCandidatorOptStrategy.assigned.equals(sourceStatus.getNextCandidatorOptStrategy()) || NextCandidatorOptStrategy.assignedRole.equals(sourceStatus.getNextCandidatorOptStrategy())) {
                           CoreBpmnInstanceNextTaskElementDTO<ID> nextTaskElement = BpmnRuntimeCacheProvider.getNextTask(source);
                           List nextTaskElementList;
                           ArrayList candidatorIdList;
                           if (NextCandidatorOptStrategy.assigned.equals(sourceStatus.getNextCandidatorOptStrategy())) {
                              nextTaskElementList = nextTaskElement.getNextTaskElementList();
                              candidatorIdList = new ArrayList();
                              List<Long> candidateRoleIdList = new ArrayList();
                              CoreBpmnInstanceBean instance = BpmnRuntimeCacheProvider.getBpmnRuntimeInstance(source);
                              String initiator = instance == null ? LocalContextHelper.getLoginUserId() : instance.getInitiator();
                              Iterator var16 = nextTaskElementList.iterator();

                              while(var16.hasNext()) {
                                 Element taskElement = (Element)var16.next();
                                 CoreBpmnTaskCandidatorsDTO candidators = BpmnDiagramHelper.getUserTaskCandidators(source, taskElement, initiator);
                                 if (candidators.getCandidatorIdList() != null) {
                                    candidatorIdList.addAll(candidators.getCandidatorIdList());
                                 }

                                 if (candidators.getCandidateRoleIdList() != null) {
                                    candidateRoleIdList.addAll(candidators.getCandidateRoleIdList());
                                 }
                              }

                              List<CoreRoleUserBean> candidatorList;
                              if (!candidateRoleIdList.isEmpty()) {
                                 CoreRoleUserService roleUserService = ApplicationContextHelper.getBean(CoreRoleUserService.class);
                                 candidatorList = roleUserService.getDao().selectListByOneColumnValues(candidateRoleIdList.stream().distinct().collect(Collectors.toList()), "ROLEID");
                                 candidatorList.forEach((r) -> {
                                    candidatorIdList.add(r.getUserId());
                                 });
                              }

                              if (!candidatorIdList.isEmpty()) {
                                 CoreUserService userService = ApplicationContextHelper.getBean(CoreUserService.class);
                                 List<CoreUserBean> candidatorList1 = userService.selectListByIds((List)candidatorIdList.stream().distinct().collect(Collectors.toList()));
                                 if (CandidatorFilterStrategy.sameDept.equals(BpmnDiagramHelper.getCandidatorFilterStrategy(currentElement))) {
                                    String createdByOrgId = (String)PersistableHelper.getPropertyValue(source.getOldItem(), "createdByOrgId");
                                    if (!StringUtils.isBlank(createdByOrgId)) {
                                       candidatorList1.removeIf((c) -> {
                                          return !c.getOrgId().equals(createdByOrgId);
                                       });
                                    }
                                 }

                                 sourceStatus.setCandidatorList(candidatorList1);
                              }
                           } else if (NextCandidatorOptStrategy.assignedRole.equals(sourceStatus.getNextCandidatorOptStrategy())) {
                              nextTaskElementList = nextTaskElement.getNextTaskElementList();
                              candidatorIdList = new ArrayList();
                              CoreBpmnInstanceBean instance = BpmnRuntimeCacheProvider.getBpmnRuntimeInstance(source);
                              String initiator = instance == null ? LocalContextHelper.getLoginUserId() : instance.getInitiator();
                              Iterator var24 = nextTaskElementList.iterator();

                              while(var24.hasNext()) {
                                 Element taskElement = (Element)var24.next();
                                 CoreBpmnTaskCandidatorsDTO candidators = BpmnDiagramHelper.getUserTaskCandidators(source, taskElement, initiator);
                                 if (candidators.getCandidateRoleIdList() != null) {
                                    candidatorIdList.addAll(candidators.getCandidateRoleIdList());
                                 }
                              }

                              if (!candidatorIdList.isEmpty()) {
                                 CoreRoleService roleService = (CoreRoleService)ApplicationContextHelper.getBean(CoreRoleService.class);
                                 sourceStatus.setCandidateRoleList(roleService.selectListByIds((List)candidatorIdList.stream().distinct().collect(Collectors.toList())));
                              }
                           }
                        }
                     }

                     if (taskStatus == null) {
                        taskStatus = sourceStatus;
                     } else {
                        this.checkAndMerge(taskStatus, sourceStatus);
                     }
                  }
               } finally {
                  BpmnRuntimeCacheProvider.remove();
               }

               if (NextCandidatorOptStrategy.assigned.equals(taskStatus.getNextCandidatorOptStrategy()) && CollectionUtils.isEmpty((Collection)taskStatus.getCandidatorList())) {
                  throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.TIP.ASSIGNED_STRATEGY_REQUIRES_CANDIDATORS");
               } else if (NextCandidatorOptStrategy.assignedRole.equals(taskStatus.getNextCandidatorOptStrategy()) && CollectionUtils.isEmpty((Collection)taskStatus.getCandidateRoleList())) {
                  throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.TIP.ASSIGNED_ROLE_STRATEGY_REQUIRES_CANDIDATEROLES");
               } else {
                  return taskStatus;
               }
            }
         }
      }
   }

   default List<T> getBpmnVarsItemList(List<ID> idList) {
      return this.getDao().selectListByIds(idList);
   }

   default Map<ID, Map<String, Object>> getBpmnVars(RestJsonWrapperBean wrapper, List<T> itemList) {
      Map<ID, Map<String, Object>> map = new HashMap();
      List<BeanPropertyDescriptor> propertyDescriptorList = BeanPropertyHelper.getBeanPropertyDescriptorList(this.getDao().getType());

      for (T t : itemList) {
         T item = (T) t;
         Map<String, Object> vars = new HashMap();
         propertyDescriptorList.forEach((pd) -> {
            if (String.class.equals(pd.getPropertyType()) || Date.class.isAssignableFrom(pd.getPropertyType()) || Temporal.class.isAssignableFrom(pd.getPropertyType()) || Number.class.isAssignableFrom(pd.getPropertyType())) {
               if (Temporal.class.isAssignableFrom(pd.getPropertyType())) {
                  vars.put(pd.getName().toLowerCase(), ConvertUtils.convert(pd.getPropertyValue(item), Date.class));
               } else {
                  vars.put(pd.getName().toLowerCase(), pd.getPropertyValue(item));
               }
            }

         });
         item.getExt$().forEach((k, v) -> {
            vars.put(k.toLowerCase(), v);
         });
         map.put(item.getId(), vars);
      }

      return map;
   }

   default Supplier<String> getDescSupplier(CoreBpmnRuntimeSource<T, ID> source) {
      return () -> {
         return ObjectUtils.toString(source.getId());
      };
   }

   default List<String> getChosenAuditorList(RestJsonWrapperBean wrapper) {
      return JSON.parseArray(wrapper.getParamValue("auditor"), String.class);
   }

   default List<CoreBpmnRuntimeSource<T, ID>> parseRuntimeSource(List<ID> itemIdList, RestJsonWrapperBean wrapper) {
      List<T> selectedItemList = this.getBpmnVarsItemList(itemIdList);
      Map<ID, Map<String, Object>> map = this.getBpmnVars(wrapper, selectedItemList);
      CoreBpmnParameterDTO parameter = CoreBpmnParameterDTO.of(wrapper);
      List<CoreBpmnRuntimeSource<T, ID>> sourceList = new ArrayList();

      for (T t : selectedItemList) {
         T item = (T) t;
         CoreBpmnRuntimeSource<T, ID> source = CoreBpmnRuntimeSource.of(this.getDao().getTable(), item, parameter.getComment(), (Map) map.get(item.getId()));
         source.setDescSupplier(this.getDescSupplier(source));
         sourceList.add(source);
      }

      boolean draft = selectedItemList.stream().anyMatch((i) -> {
         return StringUtils.startsWithIgnoreCase(i.getProcessStatus(), ProcessStatus.DRAFT.name());
      });
      if (draft) {
         List<Pair<ID, Long>> procIdList = this.getBpmnProcIdList(selectedItemList);
         sourceList.forEach((s) -> {
            s.setProcId((Long)((Pair)procIdList.stream().filter((p) -> {
               return s.getId().equals(p.getFirst());
            }).findAny().get()).getSecond());
         });
      }

      return sourceList;
   }

   default List<CoreBpmnRuntimeSource<T, ID>> parseSubmittableRuntimeSource(List<ID> itemIdList, CoreBpmnParameterDTO parameter, Map<ID, Map<String, Object>> replacementMap) {
      List<T> selectedItemList = this.getBpmnVarsItemList(itemIdList);
      List<Pair<ID, Long>> procIdList = this.getBpmnProcIdList(selectedItemList);
      List<CoreBpmnRuntimeSource<T, ID>> sourceList = new ArrayList();

      for (T t : selectedItemList) {
         T item = (T) t;
         CoreBpmnRuntimeSource<T, ID> source = CoreBpmnRuntimeSource.of(this.getDao().getTable(), item, parameter.getComment(), (Map) replacementMap.get(item.getId()));
         source.setProcId((Long) ((Pair) procIdList.stream().filter((p) -> {
            return ((Serializable) item.getId()).equals(p.getFirst());
         }).findAny().get()).getSecond());
         source.setDescSupplier(this.getDescSupplier(source));
         String attachment = parameter.getAttachment();
         if (!StringUtils.isBlank(attachment)) {
            source.setAttachmentId(NumberUtils.parseLong(attachment));
         }

         String[] var11 = parameter.getNextCandidators();
         int var12 = var11.length;

         int var13;
         String nextCandidateRole;
         for (var13 = 0; var13 < var12; ++var13) {
            nextCandidateRole = var11[var13];
            source.addNextCandidator(nextCandidateRole);
         }

         var11 = parameter.getNextCandidateRoles();
         var12 = var11.length;

         for (var13 = 0; var13 < var12; ++var13) {
            nextCandidateRole = var11[var13];
            source.addNextCandidatorRole(NumberUtils.parseLong(nextCandidateRole));
         }

         sourceList.add(source);
      }

      return sourceList;
   }

   default List<CoreBpmnRuntimeSource<T, ID>> parseAuditableRuntimeSource(List<ID> itemIdList, CoreBpmnParameterDTO parameter, Map<ID, Map<String, Object>> replacementMap) {
      List<T> selectedItemList = this.getBpmnVarsItemList(itemIdList);
      List<CoreBpmnRuntimeSource<T, ID>> sourceList = new ArrayList();
      Iterator var6 = selectedItemList.iterator();

      while(var6.hasNext()) {
         T item = (T) var6.next();
         CoreBpmnRuntimeSource<T, ID> source = CoreBpmnRuntimeSource.of(this.getDao().getTable(), item, parameter.getComment(), (Map)replacementMap.get(item.getId()));
         source.setCurrentStatusCode(parameter.getCurrentStatusCode());
         source.setTargetStatusCode(parameter.getTargetStatusCode());
         String attachment = parameter.getAttachment();
         if (!StringUtils.isBlank(attachment)) {
            source.setAttachmentId(NumberUtils.parseLong(attachment));
         }

         String[] var10 = parameter.getNextCandidators();
         int var11 = var10.length;

         int var12;
         String nextCandidateRole;
         for(var12 = 0; var12 < var11; ++var12) {
            nextCandidateRole = var10[var12];
            source.addNextCandidator(nextCandidateRole);
         }

         var10 = parameter.getNextCandidateRoles();
         var11 = var10.length;

         for(var12 = 0; var12 < var11; ++var12) {
            nextCandidateRole = var10[var12];
            source.addNextCandidatorRole(NumberUtils.parseLong(nextCandidateRole));
         }

         if (!StringUtils.isBlank(parameter.getTransferCandidator())) {
            source.setTransferCandidator(parameter.getTransferCandidator());
         }

         source.setDescSupplier(this.getDescSupplier(source));
         sourceList.add(source);
      }

      return sourceList;
   }

   default List<CoreBpmnRuntimeSource<T, ID>> parseUndoableRuntimeSource(List<ID> itemIdList, CoreBpmnParameterDTO parameter, Map<ID, Map<String, Object>> replacementMap) {
      List<T> selectedItemList = this.getBpmnVarsItemList(itemIdList);
      List<CoreBpmnRuntimeSource<T, ID>> sourceList = new ArrayList();
      Iterator var6 = selectedItemList.iterator();

      while(var6.hasNext()) {
         T item = (T) var6.next();
         CoreBpmnRuntimeSource<T, ID> source = CoreBpmnRuntimeSource.of(this.getDao().getTable(), item, parameter.getComment(), (Map)replacementMap.get(item.getId()));
         source.setDescSupplier(this.getDescSupplier(source));
         sourceList.add(source);
      }

      return sourceList;
   }

   default List<CoreBpmnRuntimeSource<T, ID>> parseWithdrawableRuntimeSource(List<ID> itemIdList, CoreBpmnParameterDTO parameter, Map<ID, Map<String, Object>> replacementMap) {
      List<T> selectedItemList = this.getBpmnVarsItemList(itemIdList);
      List<CoreBpmnRuntimeSource<T, ID>> sourceList = new ArrayList();
      Iterator var6 = selectedItemList.iterator();

      while(var6.hasNext()) {
         T item = (T) var6.next();
         CoreBpmnRuntimeSource<T, ID> source = CoreBpmnRuntimeSource.of(this.getDao().getTable(), item, parameter.getComment(), (Map)replacementMap.get(item.getId()));
         source.setDescSupplier(this.getDescSupplier(source));
         sourceList.add(source);
      }

      return sourceList;
   }

   default List<Pair<ID, Long>> getBpmnProcIdList(List<T> itemList) {
      CoreBpmnRuntimeService runtimeService = (CoreBpmnRuntimeService)ApplicationContextHelper.getBean(CoreBpmnRuntimeService.class);
      Map<ID, CoreBpmnProcBean> map = runtimeService.selectRuntimeBpmnProcList(itemList);
      return (List)map.entrySet().stream().map((e) -> {
         return Pair.of(e.getKey(), ((CoreBpmnProcBean)e.getValue()).getId());
      }).collect(Collectors.toList());
   }

   default void checkAndMerge(CoreBpmnTaskStatusDTO taskStatus, CoreBpmnTaskStatusDTO newTaskStatus) {
      if (taskStatus != null) {
         if (taskStatus.getAuthRequired() != newTaskStatus.getAuthRequired()) {
            throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.TIP.AUTH_REQUIRED_DIFFER");
         } else if (!ObjectUtils.equals(taskStatus.getAttachmentStrategy(), newTaskStatus.getAttachmentStrategy())) {
            throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.TIP.ATTACHMENT_STRATEGY_DIFFER");
         } else if (!ObjectUtils.equals(taskStatus.getNextCandidatorOptStrategy(), newTaskStatus.getNextCandidatorOptStrategy())) {
            throw new BpmnException("CORE.MODULE.SYS.T_CORE_BPMN_PROC.ENGINE.TIP.NEXT_CANDIDATOR_OPT_STRATEGY_DIFFER");
         } else {
            if (taskStatus.isTransfer() != newTaskStatus.isTransfer()) {
               taskStatus.setTransfer(false);
            }

            if (NextCandidatorOptStrategy.assigned.equals(taskStatus.getNextCandidatorOptStrategy()) && taskStatus.getCandidatorList() != null) {
               if (newTaskStatus.getCandidatorList() == null) {
                  taskStatus.setCandidatorList(null);
               } else {
                  taskStatus.getCandidatorList().removeIf((c) -> {
                     return newTaskStatus.getCandidatorList().stream().noneMatch((n) -> {
                        return n.getId().equals(c.getId());
                     });
                  });
               }
            }

            if (NextCandidatorOptStrategy.assignedRole.equals(taskStatus.getNextCandidatorOptStrategy()) && taskStatus.getCandidateRoleList() != null) {
               if (newTaskStatus.getCandidateRoleList() == null) {
                  taskStatus.setCandidateRoleList(null);
               } else {
                  taskStatus.getCandidateRoleList().removeIf((c) -> {
                     return newTaskStatus.getCandidateRoleList().stream().noneMatch((n) -> {
                        return n.getId().equals(c.getId());
                     });
                  });
               }
            }

         }
      }
   }

   default void checkAndUpdateVersion(List<T> itemList) {
      EntityColumnContext versionContext = DatabaseManager.getEntityContext(this.getDao().getType()).getVersionContext();
      if (versionContext != null) {
         List<ID> idList = (List)itemList.stream().filter((i) -> {
            return i.getVersion() != null;
         }).map((i) -> {
            return (Serializable)i.getId();
         }).collect(Collectors.toList());
         if (!idList.isEmpty()) {
            List<T> selectedVersionItemList = this.getDao().selectListByIds(idList, Arrays.asList("ID", "VERSION"));
            Iterator var5 = selectedVersionItemList.iterator();

            while(var5.hasNext()) {
               T selectedVersionItem = (T) var5.next();
               if (itemList.stream().noneMatch((i) -> {
                  return ((Serializable)i.getId()).equals(selectedVersionItem.getId()) && i.getVersion().equals(selectedVersionItem.getVersion());
               })) {
                  throw new ApplicationWarningException("SINO.EXCEPTION.CONRRENT_MODIFICATION", new String[0]);
               }
            }
         }

         itemList.forEach((i) -> {
            i.setVersion((Long)Optional.ofNullable(i.getVersion()).orElse(1L) + 1L);
         });
         this.getDao().update(itemList, "VERSION");
      }

   }
}
