/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.support;

import net.sinodawn.framework.beans.BeanPropertyDescriptor;
import net.sinodawn.framework.beans.BeanPropertyHelper;
import net.sinodawn.framework.data.annotation.*;
import net.sinodawn.framework.exception.database.NoIdPropertyJdbcException;
import net.sinodawn.framework.support.domain.Orderable;
import net.sinodawn.framework.support.domain.Persistable;
import net.sinodawn.framework.utils.ClassUtils;
import net.sinodawn.framework.utils.StringUtils;
import org.springframework.data.annotation.Id;
import org.springframework.data.annotation.Transient;

@SuppressWarnings({"unused", "BooleanMethodIsAlwaysInverted"})
public class PersistableMetadataHelper {
    public PersistableMetadataHelper() {
    }

    /**
     * 解析Table注解
     * @param persistableClass Bean
     * @return TableName
     */
    public static String getTableName(Class<?> persistableClass) {
        persistableClass = ClassUtils.getRawType(persistableClass);
        Table table = persistableClass.getAnnotation(Table.class);
        return table != null && !StringUtils.isEmpty(table.value()) ? table.value().toUpperCase() : StringUtils.removeEnd(persistableClass.getSimpleName().toUpperCase(), "BEAN");
    }

    public static boolean isModifiedTable(String table) {
        return StringUtils.contains(table, "_M_");
    }

    public static BeanPropertyDescriptor getIdPropertyDescriptor(Class<? extends Persistable<?>> persistableClass) {
        return BeanPropertyHelper.getBeanPropertyDescriptorList(persistableClass).stream()
                .filter(PersistableMetadataHelper::isIdPropertyDescriptor)
                .findFirst().orElseThrow(() -> new NoIdPropertyJdbcException(persistableClass));
    }

    public static BeanPropertyDescriptor getCreatedByOrgIdPropertyDescriptor(Class<? extends Persistable<?>> persistableClass) {
        return BeanPropertyHelper.getBeanPropertyDescriptorList(persistableClass).stream().filter(PersistableMetadataHelper::isCreatedByOrgIdPropertyDescriptor).findFirst().orElse(null);
    }

    public static BeanPropertyDescriptor getExt$PropertyDescriptor(Class<? extends Persistable<?>> persistableClass) {
        return BeanPropertyHelper.getBeanPropertyDescriptorList(persistableClass).stream().filter(PersistableMetadataHelper::isExt$PropertyDescriptor).findFirst().orElse(null);
    }

    public static BeanPropertyDescriptor getOrderNoPropertyDescriptor(Class<? extends Persistable<?>> persistableClass) {
        return Orderable.class.isAssignableFrom(persistableClass) ? BeanPropertyHelper.getBeanPropertyDescriptorList(persistableClass).stream().filter(PersistableMetadataHelper::isOrderNoPropertyDescriptor).findFirst().orElse(null) : null;
    }

    public static boolean isIdPropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return propertyDescriptor.isAnnotationPresent(Id.class);
    }

    public static boolean isReadOnlyPropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return propertyDescriptor.isAnnotationPresent(ReadOnly.class);
    }

    public static boolean isTransientPropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return propertyDescriptor.isAnnotationPresent(Transient.class);
    }

    public static boolean isNotNullPropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return propertyDescriptor.isAnnotationPresent(NotNull.class);
    }

    public static boolean isClobPropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return propertyDescriptor.isAnnotationPresent(Clob.class);
    }

    public static boolean isBlobPropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return propertyDescriptor.isAnnotationPresent(Blob.class);
    }

    public static boolean isExt$PropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return propertyDescriptor.isExt$BeanProperty();
    }

    public static boolean isInsertablePropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return !isTransientPropertyDescriptor(propertyDescriptor);
    }

    public static boolean isOrderNoPropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return "orderNo".equalsIgnoreCase(propertyDescriptor.getName());
    }

    public static boolean isCreatedByIdPropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return "createdById".equalsIgnoreCase(propertyDescriptor.getName());
    }

    public static boolean isCreatedByNamePropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return "createdByName".equalsIgnoreCase(propertyDescriptor.getName());
    }

    public static boolean isCreatedTimePropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return "createdTime".equalsIgnoreCase(propertyDescriptor.getName());
    }

    public static boolean isCreatedByOrgIdPropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return "createdByOrgId".equalsIgnoreCase(propertyDescriptor.getName());
    }

    public static boolean isCreatedByOrgNamePropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return "createdByOrgName".equalsIgnoreCase(propertyDescriptor.getName());
    }

    public static boolean isUpdatablePropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return (isExt$PropertyDescriptor(propertyDescriptor) || !isTransientPropertyDescriptor(propertyDescriptor)) && !isReadOnlyPropertyDescriptor(propertyDescriptor) && !isCreatedByIdPropertyDescriptor(propertyDescriptor) && !isCreatedByNamePropertyDescriptor(propertyDescriptor) && !isCreatedTimePropertyDescriptor(propertyDescriptor) && !isCreatedByOrgIdPropertyDescriptor(propertyDescriptor) && !isCreatedByOrgNamePropertyDescriptor(propertyDescriptor) && !isIdPropertyDescriptor(propertyDescriptor);
    }

    public static boolean isSearchTermsPropertyDescriptor(BeanPropertyDescriptor propertyDescriptor) {
        return (isExt$PropertyDescriptor(propertyDescriptor) || !isTransientPropertyDescriptor(propertyDescriptor)) && !isReadOnlyPropertyDescriptor(propertyDescriptor);
    }
}
