/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.security.support;

import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.exception.database.JdbcException;
import net.sinodawn.framework.security.AccountCategory;
import net.sinodawn.framework.security.authentication.AuthenticationHelper;
import net.sinodawn.framework.security.bean.LoginUser;
import net.sinodawn.framework.security.captcha.SecurityCaptchaService;
import net.sinodawn.framework.security.service.DefaultUserDetailsChecker;
import net.sinodawn.framework.security.sso.RemoteAuthenticator;
import net.sinodawn.framework.security.sso.SsoAuthenticator;
import net.sinodawn.framework.security.sso.SsoAuthenticatorRegistry;
import net.sinodawn.framework.utils.CollectionUtils;
import net.sinodawn.framework.utils.ServletUtils;
import net.sinodawn.framework.utils.StringUtils;
import net.sinodawn.module.mdm.user.bean.CoreUserBean;
import net.sinodawn.module.mdm.user.service.CoreUserService;
import net.sinodawn.module.sys.password.bean.CorePasswordPolicyBean;
import net.sinodawn.module.sys.password.service.CorePasswordPolicyService;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Value;
import org.springframework.context.annotation.Lazy;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.LockedException;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.authentication.dao.DaoAuthenticationProvider;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.CredentialsContainer;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.crypto.password.PasswordEncoder;

import java.time.LocalDateTime;
import java.util.Iterator;
import java.util.List;
import java.util.stream.Collectors;

/**
 * AuthenticationProvider默认实现
 */
public class DefaultDaoAuthenticationProvider extends DaoAuthenticationProvider {
   private static final Logger logger = LogManager.getLogger(DaoAuthenticationProvider.class);
   @Autowired
   @Lazy
   protected CoreUserService userService;
   @Autowired
   private CorePasswordPolicyService passwordPolicyService;
   @Autowired
   @Lazy
   private PasswordEncoder passwordEncoder;
   @Value("${sino.security.remote-authenticator-name:}")
   private String remoteAuthenticatorName;
   @Autowired
   @Lazy
   private SecurityCaptchaService captchaService;

   public DefaultDaoAuthenticationProvider() {
      this.setHideUserNotFoundExceptions(false);
   }

   @Override
   public Authentication authenticate(Authentication authentication) throws AuthenticationException {
      this.captchaService.checkCaptcha();
      String loginType = ServletUtils.getCurrentRequest().getParameter("loginType");
      LoginUser loginUser;
      if (!StringUtils.isEmpty(loginType)) {
         SsoAuthenticator authenticator = SsoAuthenticatorRegistry.INSTANCE.getAuthenticator(loginType);
         if (authenticator == null) {
            throw new SecurityException("SINO.SECURITY.SSO_LOGIN.INVALID_LOGINTYPE");
         }

         loginUser = authenticator.authenticate();
         authentication = new UsernamePasswordAuthenticationToken(loginUser.getUsername(), loginUser.getAdditionalCheck() ? ((Authentication)authentication).getCredentials() : loginUser.getPassword(), CollectionUtils.emptyList());
         if (loginUser.getAdditionalCheck()) {
            authentication.setAuthenticated(false);
         }
      }

      if (StringUtils.isEmpty(loginType) && !StringUtils.isBlank(this.remoteAuthenticatorName)) {
         RemoteAuthenticator remoteAuthenticator = ApplicationContextHelper.getBeanIfPresent(this.remoteAuthenticatorName);
         if (remoteAuthenticator != null) {
            loginUser = remoteAuthenticator.authenticate((String) authentication.getPrincipal(), AuthenticationHelper.getRawPassword((String)((Authentication)authentication).getCredentials()));
            CoreUserBean user = this.userService.selectById(loginUser.getUsername());
            this.logoutPreviousLoginIfNecessary(user);
            return new UsernamePasswordAuthenticationToken(loginUser, user.getPassword());
         }
      }

      return super.authenticate(authentication);
   }

   @Override
   protected void additionalAuthenticationChecks(UserDetails userDetails, UsernamePasswordAuthenticationToken authentication) throws AuthenticationException {
      try {
         CoreUserBean user = this.userService.selectById(userDetails.getUsername());

         try {
            if (!authentication.isAuthenticated() && ((LoginUser)userDetails).getAdditionalCheck()) {
               String rawPassword = AuthenticationHelper.getRawPassword((String)authentication.getCredentials());
               DefaultUserDetailsChecker.checkPassword((String)authentication.getPrincipal(), rawPassword);
            }

            List<CorePasswordPolicyBean> passwordPolicyList = this.passwordPolicyService.selectEffectedList(user.getId());
            List<CorePasswordPolicyBean> expiryIntervalList = passwordPolicyList.stream().filter((p) -> p.getExpiryInterval() != null && p.getExpiryInterval() > 0L).collect(Collectors.toList());
            if (!expiryIntervalList.isEmpty()) {
               LocalDateTime now = LocalDateTime.now();
               boolean expiredPassword = false;
               Iterator<CorePasswordPolicyBean> var8 = expiryIntervalList.iterator();

               label148:
               while(true) {
                  CorePasswordPolicyBean passwordPolicy;
                  do {
                     if (!var8.hasNext()) {
                        if (expiredPassword) {
                           CoreUserBean updateUser = new CoreUserBean();
                           updateUser.setId(user.getId());
                           updateUser.setExpiredPasswordUses(user.getExpiredPasswordUses() + 1);
                           this.userService.update(updateUser);
                        }
                        break label148;
                     }

                     passwordPolicy = var8.next();
                  } while(!user.getPasswordUpdatedTime().plusDays(passwordPolicy.getExpiryInterval()).isBefore(now));

                  if (passwordPolicy.getExpiredMaxUse() == null || passwordPolicy.getExpiredMaxUse() <= (long)user.getExpiredPasswordUses()) {
                     CoreUserBean updateUser = new CoreUserBean();
                     updateUser.setId(user.getId());
                     updateUser.setStatus("locked");
                     this.userService.update(updateUser);
                     throw new LockedException("SINO.SECURITY.LOGIN.LOCKED");
                  }

                  expiredPassword = true;
               }
            }

            if (user.getFailedLoginAttempts() > 0) {
               CoreUserBean updateUser = new CoreUserBean();
               updateUser.setId(user.getId());
               updateUser.setFailedLoginAttempts(0);
               this.userService.update(updateUser);
            }

            this.logoutPreviousLoginIfNecessary(user);
         } catch (BadCredentialsException var15) {
            DefaultUserDetailsChecker.postAuthenticationFailure(userDetails.getUsername());
         } finally {
            if (userDetails instanceof CredentialsContainer) {
               ((CredentialsContainer)userDetails).eraseCredentials();
            }

         }

      } catch (Exception var17) {
         if (AuthenticationException.class.isAssignableFrom(var17.getClass())) {
            throw var17;
         } else {
            logger.error(var17.getMessage(), var17);
            throw new JdbcException("SINO.EXCEPTION.UNEXPECTED");
         }
      }
   }

   private void logoutPreviousLoginIfNecessary(CoreUserBean user) {
      if ("0".equals(user.getMultiLogin())) {
         List<String> tokenList = AuthenticationHelper.getLoginTokenList();

         for (String token : tokenList) {
            Authentication tokenAuthentication = AuthenticationHelper.getCacheAuthentication(token);
            if (tokenAuthentication != null && user.getId().equals(tokenAuthentication.getName())) {
               AuthenticationHelper.markOffline(token, AccountCategory.OFFLINE_PRELOGIN.name());
               AuthenticationHelper.logout(token, "不允许重复登录，强制下线");
            }
         }
      }

   }
}
