/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.security.service;

import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.i18n.I18nHelper;
import net.sinodawn.framework.security.bean.LoginUser;
import net.sinodawn.framework.security.sso.SsoAuthenticator;
import net.sinodawn.framework.security.sso.SsoAuthenticatorRegistry;
import net.sinodawn.framework.utils.ServletUtils;
import net.sinodawn.framework.utils.StringUtils;
import net.sinodawn.module.mdm.user.bean.CoreUserBean;
import net.sinodawn.module.mdm.user.service.CoreUserService;
import net.sinodawn.module.sys.log.service.CoreLoginLogService;
import net.sinodawn.module.sys.password.bean.CorePasswordPolicyBean;
import net.sinodawn.module.sys.password.service.CorePasswordPolicyService;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.DisabledException;
import org.springframework.security.authentication.LockedException;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UsernameNotFoundException;

import javax.servlet.http.HttpServletRequest;
import java.time.LocalDateTime;
import java.util.List;
import java.util.stream.Collectors;

/**
 * UserDetail检查类默认实现
 */
public class DefaultUserDetailsChecker {
   /**
    * 检查userId、password
    */
   public static void checkPassword(String loginId, String encodedPassword) {
      CoreUserBean user = ApplicationContextHelper.getBean(CoreUserService.class).selectByLoginId(loginId);
      if (user == null) {
         getLoginLogService().insertLoginFailureLog(loginId, I18nHelper.getMessage("SINO.SECURITY.LOGIN.INCORRECT_USER_OR_PASSWORD"));
         throw new BadCredentialsException("SINO.SECURITY.LOGIN.INCORRECT_USER_OR_PASSWORD");
      } else if (!user.getPassword().equals(encodedPassword)) {
         getLoginLogService().insertLoginFailureLog(loginId, I18nHelper.getMessage("SINO.SECURITY.LOGIN.INCORRECT_USER_OR_PASSWORD"));
         throw new BadCredentialsException("SINO.SECURITY.LOGIN.INCORRECT_USER_OR_PASSWORD");
      }
   }

   public static void check(CoreUserBean user) {
      if (user == null) {
         throw new UsernameNotFoundException("SINO.SECURITY.LOGIN.INCORRECT_USER_OR_PASSWORD");
      } else if ("1".equals(user.getLastSuspendedFlag())) {
         getLoginLogService().insertLoginFailureLog(user.getId(), I18nHelper.getMessage("SINO.SECURITY.LOGIN.DEACTIVATED"));
         throw new DisabledException("SINO.SECURITY.LOGIN.DEACTIVATED");
      } else if ("locked".equals(user.getStatus())) {
         getLoginLogService().insertLoginFailureLog(user.getId(), I18nHelper.getMessage("SINO.SECURITY.LOGIN.LOCKED"));
         throw new LockedException("SINO.SECURITY.LOGIN.LOCKED");
      } else if ("deprecated".equals(user.getStatus())) {
         getLoginLogService().insertLoginFailureLog(user.getId(), I18nHelper.getMessage("SINO.SECURITY.LOGIN.DEPRECATED"));
         throw new LockedException("SINO.SECURITY.LOGIN.DEPRECATED");
      } else {
         LocalDateTime now = LocalDateTime.now();
         if (user.getExpiryDate() != null && user.getExpiryDate().isBefore(now)) {
            getLoginLogService().insertLoginFailureLog(user.getId(), I18nHelper.getMessage("SINO.SECURITY.LOGIN.EXPIRED"));
            throw new LockedException("SINO.SECURITY.LOGIN.EXPIRED");
         }
      }
   }

   public static void check(UserDetails userDetails) {
      CoreUserBean user = (CoreUserBean)((CoreUserService)ApplicationContextHelper.getBean(CoreUserService.class)).selectById(userDetails.getUsername());
      check(user);
      HttpServletRequest request = ServletUtils.getCurrentRequest();
      if (request != null) {
         String loginType = request.getParameter("loginType");
         if (!StringUtils.isBlank(loginType)) {
            SsoAuthenticator authenticator = SsoAuthenticatorRegistry.INSTANCE.getAuthenticator(loginType);
            if (authenticator != null) {
               authenticator.checkLoginUser((LoginUser)userDetails);
            }
         }
      }

   }

   public static void postAuthenticationFailure(String loginId) {
      CoreUserService userService = ApplicationContextHelper.getBean(CoreUserService.class);
      CoreUserBean user = userService.selectByLoginId(loginId);
      if (user != null) {
         List<CorePasswordPolicyBean> passwordPolicyList = ApplicationContextHelper.getBean(CorePasswordPolicyService.class).selectEffectedList(user.getId()).stream().filter((p) -> p.getMaxFailedAttempts() != null && p.getMaxFailedAttempts() > 0L).collect(Collectors.toList());
         if (!passwordPolicyList.isEmpty()) {
            CoreUserBean updateUser = new CoreUserBean();
            updateUser.setId(user.getId());
            updateUser.setFailedLoginAttempts(user.getFailedLoginAttempts() + 1);
            if ((long)updateUser.getFailedLoginAttempts() > passwordPolicyList.stream().mapToLong(CorePasswordPolicyBean::getMaxFailedAttempts).min().getAsLong()) {
               updateUser.setStatus("locked");
            }

            userService.update(updateUser);
            if ("locked".equals(updateUser.getStatus())) {
               getLoginLogService().insertLoginFailureLog(loginId, I18nHelper.getMessage("SINO.SECURITY.LOGIN.LOCKED"));
               throw new LockedException("SINO.SECURITY.LOGIN.LOCKED");
            }
         }
      } else {
         getLoginLogService().insertLoginFailureLog(loginId, I18nHelper.getMessage("SINO.SECURITY.LOGIN.INCORRECT_USER_OR_PASSWORD"));
      }

      throw new BadCredentialsException("SINO.SECURITY.LOGIN.INCORRECT_USER_OR_PASSWORD");
   }

   private static CoreLoginLogService getLoginLogService() {
      return ApplicationContextHelper.getBean(CoreLoginLogService.class);
   }
}
