/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.security.authentication;

import net.sinodawn.framework.cache.redis.RedisHelper;
import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.context.LocalContextHelper;
import net.sinodawn.framework.security.constant.SecurityConstant;
import net.sinodawn.framework.security.provider.service.SecurityProviderService;
import net.sinodawn.framework.utils.EncryptUtils;
import net.sinodawn.framework.utils.ServletUtils;
import net.sinodawn.framework.utils.StringUtils;
import net.sinodawn.module.sys.log.service.CoreLoginLogService;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.cache.CacheManager;
import org.springframework.data.redis.core.RedisTemplate;
import org.springframework.security.core.Authentication;
import org.springframework.stereotype.Component;

import javax.servlet.http.Cookie;
import javax.servlet.http.HttpServletRequest;
import java.util.ArrayList;
import java.util.List;
import java.util.Objects;
import java.util.Optional;

@SuppressWarnings("rawtypes")
@Component
public class AuthenticationHelper {

    private static CacheManager cacheManager;

    private static RedisTemplate redisTemplate;

    @Autowired
    public void setCacheManager(CacheManager cacheManager) {
        AuthenticationHelper.cacheManager = cacheManager;
    }

    @Autowired
    public void setRedisTemplate(RedisTemplate redisTemplate) {
        AuthenticationHelper.redisTemplate = redisTemplate;
    }

    // TODO 获取登陆token
    public static List<String> getLoginTokenList() {
        List<String> tokens = new ArrayList<>();
        try {
            Optional.ofNullable(redisTemplate.keys(SecurityConstant.REMEMBER_ME_CACHE_NAME + "*")).ifPresent(tokens::addAll);
        } catch (Exception ignored) {

        }

        return tokens;
    }

    public static void login(HttpServletRequest request, Authentication successfulAuthentication, String rememberMeToken) {
        Objects.requireNonNull(cacheManager.getCache(SecurityConstant.REMEMBER_ME_CACHE_NAME)).put(rememberMeToken, successfulAuthentication);
        ApplicationContextHelper.getBean(CoreLoginLogService.class).insertLoginSuccessLog(
            rememberMeToken,
            successfulAuthentication.getName(),
            LocalContextHelper.getLoginOrgId()
        );
    }

    public static void logout(String rememberMeToken, String remark) {
        Objects.requireNonNull(cacheManager.getCache(SecurityConstant.REMEMBER_ME_CACHE_NAME)).evict(rememberMeToken);
        ApplicationContextHelper.getBean(CoreLoginLogService.class).assignLogout(rememberMeToken, remark);
    }

    public static Authentication getCacheAuthentication(String rememberMeToken) {
        return Objects.requireNonNull(cacheManager.getCache(SecurityConstant.REMEMBER_ME_CACHE_NAME)).get(rememberMeToken, Authentication.class);
    }

    public static void updateCacheAuthentication(String rememberMeToken, Authentication authentication) {
        Objects.requireNonNull(cacheManager.getCache(SecurityConstant.REMEMBER_ME_CACHE_NAME)).put(rememberMeToken, authentication);
    }

    public static void markOffline(String rememberMeToken, String category) {
        RedisHelper.put("MARK_OFFLINE", rememberMeToken, category, 300);
    }

    public static String getOfflineCategory(String rememberMeToken) {
        return RedisHelper.get("MARK_OFFLINE", rememberMeToken);
    }

    public static String getLoginUserToken() {
        HttpServletRequest request = ServletUtils.getCurrentRequest();
        if (request == null) {
            return null;
        } else {
            Cookie[] cookies = request.getCookies();
            if (cookies != null && cookies.length > 0) {

                for (Cookie cookie : cookies) {
                    if (SecurityConstant.COOKIE_NAME.equals(cookie.getName())) {
                        return cookie.getValue();
                    }
                }
            }

            return request.getHeader(SecurityConstant.REMEMBER_ME_KEY);
        }
    }

    public static String getRawPassword(String password) {
        if (ServletUtils.getCurrentRequest() != null) {
            String publicKey = ServletUtils.getCurrentRequest().getHeader("SECURITY-RSA-KEY");
            if (!StringUtils.isBlank(publicKey)) {
                SecurityProviderService providerService = ApplicationContextHelper.getBean(SecurityProviderService.class);
                return EncryptUtils.RSADecrypt(password, providerService.getPrivateKey(publicKey));
            }
        }

        return password;
    }
}
