/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.restful.resolver.feign;

import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONObject;
import net.sinodawn.framework.beans.BeanPropertyDescriptor;
import net.sinodawn.framework.beans.BeanPropertyHelper;
import net.sinodawn.framework.exception.ResolverArgumentException;
import net.sinodawn.framework.restful.resolver.AbstractJsonResolveArgumentHandler;
import net.sinodawn.framework.restful.resolver.feign.annotation.FeignRequestParam;
import net.sinodawn.framework.support.domain.BaseData;
import net.sinodawn.framework.utils.ClassUtils;
import net.sinodawn.framework.utils.ConvertUtils;
import net.sinodawn.framework.utils.JsonUtils;
import net.sinodawn.framework.utils.StringUtils;
import org.springframework.core.MethodParameter;
import org.springframework.web.context.request.NativeWebRequest;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.ParameterizedType;
import java.util.Iterator;
import java.util.List;

public class ToRestJsonFeignResolverArgumentHandler extends AbstractJsonResolveArgumentHandler<Object> {
   private static final String FEIGN_PARAM_ATTRIBUTE = "SINO_FEIGN_PARAM";

   public ToRestJsonFeignResolverArgumentHandler(MethodParameter parameter, NativeWebRequest webRequest) {
      super(parameter, webRequest);
   }

   public boolean supports() {
      HttpServletRequest request = (HttpServletRequest)this.getWebRequest().getNativeRequest(HttpServletRequest.class);
      return !"GET".equals(request.getMethod()) && StringUtils.contains(request.getContentType(), "application/json");
   }

   public Object resolveArgument() {
      String requestJson = this.getRequestBody();
      if (StringUtils.isEmpty(requestJson)) {
         return null;
      } else {
         JSONObject jsonObject = JSON.parseObject(requestJson);
         MethodParameter parameter = this.getParameter();
         FeignRequestParam parameterAnnotation = (FeignRequestParam)parameter.getParameterAnnotation(FeignRequestParam.class);
         String key = parameterAnnotation.value();
         Object value;
         if (!StringUtils.isEmpty(key)) {
            value = jsonObject.get(key);
            if (value == null && parameterAnnotation.required()) {
               throw new ResolverArgumentException(String.format("required param %s is not present", key));
            }
         } else {
            key = parameter.getParameterName();
            value = jsonObject.get(key);
         }

         Class<?> parameterType = parameter.getParameterType();
         if (value != null) {
            if (ClassUtils.isPrimitiveType(parameterType)) {
               throw new ResolverArgumentException(String.format("param %s can not be primitiveType", key));
            } else if (ClassUtils.isWrapperType(parameterType)) {
               return ConvertUtils.convert(value, parameterType);
            } else {
               return String.class.equals(parameterType) ? value.toString() : this.parseObject(value, parameterType);
            }
         } else if (ClassUtils.isWrapperType(parameterType)) {
            if (parameterAnnotation.required()) {
               throw new ResolverArgumentException(String.format("required param %s is not present", key));
            } else {
               return null;
            }
         } else if (!parameterAnnotation.parseAllFields()) {
            if (parameterAnnotation.required()) {
               throw new ResolverArgumentException(String.format("required param %s is not present", key));
            } else {
               return null;
            }
         } else {
            Object result = this.parseObject(jsonObject, parameterType);
            if (!parameterAnnotation.required()) {
               return result;
            } else {
               boolean haveValue = false;
               List<BeanPropertyDescriptor> bpdList = BeanPropertyHelper.getBeanPropertyDescriptorList(parameterType);
               Iterator var11 = bpdList.iterator();

               while(var11.hasNext()) {
                  BeanPropertyDescriptor bpd = (BeanPropertyDescriptor)var11.next();
                  if (!bpd.isExt$BeanProperty() && bpd.getPropertyValue(result) != null) {
                     haveValue = true;
                  }
               }

               if (!haveValue) {
                  throw new ResolverArgumentException(String.format("required param %s is not present", key));
               } else {
                  return result;
               }
            }
         }
      }
   }

   private String getRequestBody() {
      NativeWebRequest webRequest = this.getWebRequest();
      String jsonBody = (String)webRequest.getAttribute("SINO_FEIGN_PARAM", 0);
      if (jsonBody == null) {
         jsonBody = this.getRequestJson();
         webRequest.setAttribute("SINO_FEIGN_PARAM", jsonBody, 0);
      }

      return jsonBody;
   }

   private Object parseObject(Object value, Class<?> parameterType) {
      String json = value.toString();
      if (List.class.isAssignableFrom(parameterType)) {
         Class<?> actualTypeArgument = (Class)((ParameterizedType)this.getParameter().getGenericParameterType()).getActualTypeArguments()[0];
         return BaseData.class.isAssignableFrom(actualTypeArgument) ? JsonUtils.parseList(json, actualTypeArgument) : JSON.parseArray(json, actualTypeArgument);
      } else {
         return BaseData.class.isAssignableFrom(parameterType) ? JsonUtils.parse(json, parameterType) : JSON.parseObject(json, parameterType);
      }
   }
}
