/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.restful.data.extractor;

import jdk.nashorn.api.scripting.ScriptObjectMirror;
import net.sinodawn.framework.data.Pair;
import net.sinodawn.framework.io.excel.support.PropertyContext;
import net.sinodawn.framework.io.excel.support.SheetContext;
import net.sinodawn.framework.io.excel.support.impl.DefaultPropertyContextImpl;
import net.sinodawn.framework.mybatis.mapper.MatchPattern;
import net.sinodawn.framework.restful.data.RestJsonWrapperBean;
import net.sinodawn.framework.utils.CollectionUtils;
import net.sinodawn.framework.utils.ConvertUtils;
import net.sinodawn.framework.utils.DateTimeUtils;
import net.sinodawn.framework.utils.StringUtils;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.apache.poi.ss.usermodel.*;

import javax.script.Invocable;
import javax.script.ScriptEngine;
import javax.script.ScriptEngineManager;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.Map.Entry;
import java.util.function.BiConsumer;
import java.util.function.Function;
import java.util.stream.Collectors;

public class DownloadParamExtractor implements ParamExtractor<SheetContext> {
   private static final Logger logger = LogManager.getLogger(DownloadParamExtractor.class);
   private RestJsonWrapperBean jsonWrapper;

   public DownloadParamExtractor(RestJsonWrapperBean jsonWrapper) {
      this.jsonWrapper = jsonWrapper;
   }

   public RestJsonWrapperBean getJsonWrapper() {
      return this.jsonWrapper;
   }

   public SheetContext extract() {
      String topic = this.jsonWrapper.getParamValue("dt");
      List<TitleDef> titleDefList = this.jsonWrapper.parse(TitleDef.class);
      List<PropertyContext> propertyContextList = new ArrayList();
      titleDefList.forEach((t) -> {
         String title = t.getName();
         String name = StringUtils.removeStart(t.getId(), "ext$.");
         MatchPattern matchPattern = MatchPattern.EQ;
         int pos = name.lastIndexOf("_");
         MatchPattern pattern;
         if (pos > 0) {
            pattern = MatchPattern.getPatternIfPossible(name.substring(pos + 1));
            if (pattern != null) {
               matchPattern = pattern;
               name = name.substring(0, pos);
            }
         }

         pattern = null;
         Function function;
         if (!CollectionUtils.isEmpty(t.getOptions())) {
            if (MatchPattern.SB.equals(matchPattern)) {
               function = (v) -> {
                  return (String)ConvertUtils.convert(t.getOptions().entrySet().stream().filter((e) -> {
                     return !StringUtils.isBlank((String)e.getKey());
                  }).filter((e) -> {
                     return StringUtils.startsWith((String)ConvertUtils.convert(v, String.class), (String)e.getKey());
                  }).map((e) -> {
                     return (String)e.getValue();
                  }).findFirst().orElse(null), String.class);
               };
            } else if (MatchPattern.SC.equals(matchPattern)) {
               function = (v) -> {
                  return (String)ConvertUtils.convert(t.getOptions().entrySet().stream().filter((e) -> {
                     return !StringUtils.isBlank((String)e.getKey());
                  }).filter((e) -> {
                     return StringUtils.contains((String)e.getKey(), (String)ConvertUtils.convert(v, String.class));
                  }).map((e) -> {
                     return (String)e.getValue();
                  }).findFirst().orElse(null), String.class);
               };
            } else if (MatchPattern.SE.equals(matchPattern)) {
               function = (v) -> {
                  return (String)ConvertUtils.convert(t.getOptions().entrySet().stream().filter((e) -> {
                     return !StringUtils.isBlank((String)e.getKey());
                  }).filter((e) -> {
                     return StringUtils.endsWith((String)e.getKey(), (String)ConvertUtils.convert(v, String.class));
                  }).map((e) -> {
                     return (String)e.getValue();
                  }).findFirst().orElse(null), String.class);
               };
            } else {
               function = (v) -> {
                  String value = (String)ConvertUtils.convert(v, String.class);
                  return !StringUtils.isBlank(value) && value.contains(",") ? StringUtils.join((Collection)Arrays.asList(value.split(",")).stream().map((e) -> {
                     return this.getOptionValue(t, e);
                  }).collect(Collectors.toList()), " ") : this.getOptionValue(t, v);
               };
            }
         } else if ("number".equals(t.getType())) {
            function = (v) -> {
               return (Double)ConvertUtils.convert(v, Double.class);
            };
         } else if ("date".equals(t.getType())) {
            function = (v) -> {
               return DateTimeUtils.formatLocalDate((LocalDate)ConvertUtils.convert(v, LocalDate.class));
            };
         } else if ("dateTime".equals(t.getType())) {
            function = (v) -> {
               return DateTimeUtils.formatLocalDateTime((LocalDateTime)ConvertUtils.convert(v, LocalDateTime.class));
            };
         } else {
            function = (v) -> {
               return (String)ConvertUtils.convert(v, String.class);
            };
         }

         HorizontalAlignment alignment = this.getAlignment(t.getAlignment());
         PropertyContext propertyContext = DefaultPropertyContextImpl.of(title, name, function, alignment);
         if (CollectionUtils.isEmpty(t.getOptions())) {
            propertyContext.setNumeric("number".equals(t.getType()));
         }

         if (!StringUtils.isBlank(t.getStyleFormatter())) {
            String finalName = name;
            BiConsumer<Cell, Object> styleConsumer = (cell, item) -> {
               String styleFormatter = StringUtils.replace(t.getStyleFormatter().replaceAll("(\\\\r|\\\\n)+", "").trim(), "function", "function get");
               ScriptEngineManager manager = new ScriptEngineManager();
               ScriptEngine engine = manager.getEngineByName("JavaScript");

               try {
                  engine.eval("var field = '" + finalName + "';");
                  engine.eval(styleFormatter);
                  Invocable invocable = (Invocable)engine;
                  Object result = invocable.invokeFunction("get", item);
                  if (result instanceof ScriptObjectMirror) {
                     Set<Entry<String, Object>> set = ((ScriptObjectMirror)result).entrySet();

                     for (Entry<String, Object> stringObjectEntry : set) {
                        Entry<String, Object> entry = stringObjectEntry;
                        String key = entry.getKey();
                        Object value = entry.getValue();
                        CellStyle cs = cell.getCellStyle();
                        IndexedColors indexedColors;
                        if ("backgroundColor".equals(key)) {
                           indexedColors = this.getIndexedColorsIfPresent(value.toString());
                           if (indexedColors != null) {
                              cs.setFillForegroundColor(indexedColors.getIndex());
                              cs.setFillPattern(FillPatternType.SOLID_FOREGROUND);
                           }
                        } else if ("color".equals(key)) {
                           indexedColors = this.getIndexedColorsIfPresent(value.toString());
                           if (indexedColors != null) {
                              int index = cs.getFontIndexAsInt();
                              Font font = cell.getSheet().getWorkbook().getFontAt(index);
                              font.setColor(indexedColors.getIndex());
                           }
                        }
                     }
                  }
               } catch (Exception var19) {
                  logger.warn(var19.getMessage());
               }

            };
            propertyContext.setStyleConsumer(styleConsumer);
         }

         propertyContextList.add(propertyContext);
      });
      List<Pair<String, Pair<Integer, Integer>>> subTopicList = new ArrayList();

      for(int i = 0; i < titleDefList.size(); ++i) {
         TitleDef def = (TitleDef)titleDefList.get(i);
         if (!StringUtils.isEmpty(def.getParentTitle()) && !subTopicList.stream().anyMatch((t) -> {
            return ((String)t.getFirst()).equals(def.getParentTitle());
         })) {
            for(int j = i; j < titleDefList.size(); ++j) {
               TitleDef subDef = (TitleDef)titleDefList.get(j);
               if (j == titleDefList.size() - 1) {
                  if (def.getParentTitle().equals(subDef.getParentTitle())) {
                     subTopicList.add(Pair.of(def.getParentTitle(), Pair.of(i + 1, j + 1)));
                     break;
                  }
               } else if (!def.getParentTitle().equals(subDef.getParentTitle())) {
                  subTopicList.add(Pair.of(def.getParentTitle(), Pair.of(i + 1, j)));
                  break;
               }
            }
         }
      }

      if (subTopicList.isEmpty()) {
         return new SheetContext(topic, topic, propertyContextList);
      } else {
         return new SheetContext(topic, topic, propertyContextList, subTopicList);
      }
   }

   private HorizontalAlignment getAlignment(String alignment) {
      byte var3 = -1;
      switch(alignment.hashCode()) {
      case -1364013995:
         if (alignment.equals("center")) {
            var3 = 2;
         }
         break;
      case 3317767:
         if (alignment.equals("left")) {
            var3 = 0;
         }
         break;
      case 108511772:
         if (alignment.equals("right")) {
            var3 = 1;
         }
      }

      switch(var3) {
      case 0:
         return HorizontalAlignment.LEFT;
      case 1:
         return HorizontalAlignment.RIGHT;
      case 2:
         return HorizontalAlignment.CENTER;
      default:
         return HorizontalAlignment.GENERAL;
      }
   }

   private IndexedColors getIndexedColorsIfPresent(String color) {
      IndexedColors[] var2 = IndexedColors.values();
      int var3 = var2.length;

      for(int var4 = 0; var4 < var3; ++var4) {
         IndexedColors indexedColors = var2[var4];
         if (indexedColors.name().equalsIgnoreCase(color)) {
            return indexedColors;
         }
      }

      return null;
   }

   private String getOptionValue(TitleDef t, Object value) {
      String optionKey = (String)ConvertUtils.convert(value, String.class);
      if (StringUtils.isBlank(optionKey)) {
         return "";
      } else {
         String optionValue = (String)t.getOptions().get(optionKey);
         return optionValue == null ? optionKey : optionValue;
      }
   }

   public static class TitleDef {
      private String id;
      private String name;
      private String alignment;
      private String type;
      private String parentTitle;
      private String styleFormatter;
      private Map<String, String> options;

      public String getId() {
         return this.id;
      }

      public void setId(String id) {
         this.id = id;
      }

      public String getName() {
         return this.name;
      }

      public void setName(String name) {
         this.name = name;
      }

      public String getAlignment() {
         return this.alignment;
      }

      public void setAlignment(String alignment) {
         this.alignment = alignment;
      }

      public String getParentTitle() {
         return this.parentTitle;
      }

      public void setParentTitle(String parentTitle) {
         this.parentTitle = parentTitle;
      }

      public Map<String, String> getOptions() {
         return this.options;
      }

      public void setOptions(Map<String, String> options) {
         this.options = options;
      }

      public String getStyleFormatter() {
         return this.styleFormatter;
      }

      public void setStyleFormatter(String styleFormatter) {
         this.styleFormatter = styleFormatter;
      }

      public String getType() {
         return this.type;
      }

      public void setType(String type) {
         this.type = type;
      }
   }
}
