/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.restful.data;


import com.alibaba.fastjson.JSON;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.alibaba.fastjson.annotation.JSONField;
import com.alibaba.fastjson.parser.Feature;
import com.alibaba.fastjson.parser.ParserConfig;
import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.data.page.PageRequest;
import net.sinodawn.framework.database.sql.SqlHelper;
import net.sinodawn.framework.exception.HackingDataException;
import net.sinodawn.framework.exception.InvalidDataException;
import net.sinodawn.framework.exception.OutOfRangeException;
import net.sinodawn.framework.json.deserializer.DefaultJsonDeserializer;
import net.sinodawn.framework.mybatis.mapper.FilterParamPattern;
import net.sinodawn.framework.mybatis.mapper.MapperParameter;
import net.sinodawn.framework.mybatis.mapper.MatchPattern;
import net.sinodawn.framework.mybatis.page.PageRowBounds;
import net.sinodawn.framework.support.base.service.GenericService;
import net.sinodawn.framework.utils.*;

import java.sql.Time;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

/**
 * Rest请求体包装类
 */
public class RestJsonWrapperBean {
    private static final String VOID_VALUE = "VOID";
    public static final RestJsonWrapperBean EMPTY = new RestJsonWrapperBean(JSON.toJSONString(new HashMap<>()));
    private final Map<String, String> paramMap;
    private final List<List<String>> bodyList;
    @JSONField(
            serialize = false
    )
    private final List<GenericService<?, ?>> bodyServiceList;
    private final List<String> authorityList = new ArrayList<>();
    private Class<? extends MapperParameter> mapperParameterClass;
    private final Map<String, Object> extFilter = new HashMap<>();
    private final List<FilterParamPattern> extStdFilter = new ArrayList<>();
    private final String requestJson;
    private static ParserConfig parserConfig = new ParserConfig();

    public RestJsonWrapperBean() {
        this.paramMap = new HashMap<>();
        this.bodyList = CollectionUtils.emptyList();
        this.bodyServiceList = CollectionUtils.emptyList();
        this.requestJson = "";
    }

    public RestJsonWrapperBean(String filterJson) {
        this.paramMap = new HashMap<>();
        this.paramMap.put("f", filterJson);
        this.bodyList = CollectionUtils.emptyList();
        this.bodyServiceList = CollectionUtils.emptyList();
        this.requestJson = "";
    }

    public RestJsonWrapperBean(Map<String, String> paramMap, List<List<String>> bodyList, List<GenericService<?, ?>> bodyServiceList) {
        this.paramMap = (Map) Optional.ofNullable(paramMap).orElse(new HashMap<>());
        this.bodyList = (List)Optional.ofNullable(bodyList).orElse(CollectionUtils.emptyList());
        this.bodyServiceList = (List)Optional.ofNullable(bodyServiceList).orElse(CollectionUtils.emptyList());
        this.requestJson = "";
    }

    public RestJsonWrapperBean(String requestjson, Map<String, String> paramMap, List<List<String>> bodyList, List<GenericService<?, ?>> bodyServiceList) {
        this.paramMap = (Map)Optional.ofNullable(paramMap).orElse(new HashMap<>());
        this.bodyList = (List)Optional.ofNullable(bodyList).orElse(CollectionUtils.emptyList());
        this.bodyServiceList = (List)Optional.ofNullable(bodyServiceList).orElse(CollectionUtils.emptyList());
        this.requestJson = requestjson;
    }

    public Map<String, String> getParamMap() {
        return Collections.unmodifiableMap(this.paramMap);
    }

    public String getParamValue(String key) {
        return (String)this.paramMap.get(key);
    }

    public void removeParam(String key) {
        this.paramMap.remove(key);
    }

    public void setParamValue(String key, String value) {
        this.paramMap.put(key, value);
    }

    public List<List<String>> getBodyList() {
        return this.bodyList;
    }

    public List<GenericService<?, ?>> getBodyServiceList() {
        return this.bodyServiceList;
    }

    public String getRequestJson() {
        return this.requestJson;
    }

    public String getFilterValue(String key) {
        String filterJson = this.getParamValue("f");
        if (!StringUtils.isEmpty(filterJson)) {
            Map<String, String> optionsMap = (Map) JSONObject.parseObject(filterJson, new TypeReference<Map<String, String>>() {
            }, new Feature[0]);
            return (String)optionsMap.get(key);
        } else {
            return null;
        }
    }

    public <V> void setFilterValue(String key, V value) {
        String filterJson = this.getParamValue("f");
        if (StringUtils.isEmpty(filterJson)) {
            Map<String, V> map = new HashMap<>();
            map.put(key, value);
            this.paramMap.put("f", JSONObject.toJSONString(map));
        } else {
            Map<String, V> map = (Map)JSONObject.parseObject(filterJson, new TypeReference<Map<String, V>>() {
            }, new Feature[0]);
            map.put(key, value);
            this.paramMap.put("f", JSONObject.toJSONString(map));
        }

    }

    public void setAuthority(String authority) {
        this.setFilterValue("authority_" + authority, "1");
    }

    public void setQueries(String category) {
        this.setFilterValue("queries_" + category, "1");
    }

    public MapperParameter extractMapFilter() {
        return this.mapperParameterClass == null ? this.extractMapFilter(MapperParameter.class) : this.extractMapFilter(this.mapperParameterClass);
    }

    public <M extends MapperParameter> M extractMapFilter(Class<M> mapperParameterClass) {
        M parameter = (M) ClassUtils.newInstance(mapperParameterClass);
        this.extractFilter(parameter, this.getParamValue("f"));
        this.extractFilter(parameter, this.getParamValue("qf"));
        if (!this.extStdFilter.isEmpty()) {
            FilterParamPattern[] paramPatterns = ArrayUtils.concat((FilterParamPattern[]) parameter.get("filterParamPatterns"), this.extStdFilter.toArray(new FilterParamPattern[0]));
            parameter.put("filterParamPatterns", paramPatterns);
        }

        String orderJson = this.getParamValue("o");
        if (!StringUtils.isEmpty(orderJson)) {
            JSONArray array = JSONObject.parseArray(orderJson);
            array.forEach((a) -> ((JSONObject) a).forEach((key, value) -> {
                if (SqlHelper.isRiskySqlSegment((String) key)) {
                    throw new InvalidDataException("Order param contains risky words, the value is [" + (String) key + "]");
                } else {
                    if ("asc".equals(value) || "desc".equals(value)) {
                        parameter.setOrderParam(StringUtils.removeStartIgnoreCase((String) key, "ext$."), (String) value);
                    }

                }
            }));
        }

        String optionsJson = this.getParamValue("r");
        if (!StringUtils.isEmpty(optionsJson)) {
            Map<String, String> optionsMap = JSONObject.parseObject(optionsJson, new TypeReference<>() {
            });
            Map<String, Map<String, String>> resolvedOptionsMap = new HashMap<>();
            optionsMap.forEach((k, v) -> {
                Map<String, String> options = JSONObject.parseObject(v, new TypeReference<>() {
                });
                resolvedOptionsMap.put(k, options);
            });
            parameter.put("optionsMap", resolvedOptionsMap);
        }

        parameter.putAll(this.getExtFilter());
        return parameter;
    }

    public void setTotalFieldNames(String... totalFieldNames) {
        this.paramMap.put("tf", String.join(",", Arrays.asList(totalFieldNames)));
    }

    @JSONField(
            serialize = false
    )
    public String[] getSumProperties() {
        String tf = this.paramMap.get("tf");
        return StringUtils.isBlank(tf) ? new String[0] : StringUtils.split(tf, ",");
    }

    @JSONField(
            serialize = false
    )
    public String[] getCountProperties() {
        String cp = (String)this.paramMap.get("cp");
        return StringUtils.isBlank(cp) ? new String[0] : StringUtils.split(cp, ",");
    }

    public void removePageParams() {
        this.paramMap.put("n", "-1");
        this.paramMap.put("s", "-1");
    }

    public PageRequest extractPageRequest() {
        PageRequest pageRequest = new PageRequest();
        pageRequest.setPageNumber(NumberUtils.parseInt(this.getParamValue("n")));
        pageRequest.setPageSize(NumberUtils.parseInt(this.getParamValue("s")));
        pageRequest.setFilterJson(this.getParamValue("f"));
        pageRequest.setSumProperties(this.getSumProperties());
        pageRequest.setCountProperties(this.getCountProperties());
        return pageRequest;
    }

    public PageRowBounds extractPageRowBounds() {
        return new PageRowBounds(this.extractPageRequest());
    }

    public void setMapperParameterClass(Class<? extends MapperParameter> mapperParameterClass) {
        this.mapperParameterClass = mapperParameterClass;
    }

    public <T> T parseUnique(Class<T> beanClass) {
        int size = this.getBodyList().size();
        List<T> itemList = new ArrayList<>();

        int i;
        for(i = 0; i < size; ++i) {
            GenericService<?, ?> service = (GenericService)this.getBodyServiceList().get(i);
            if (service != null && service.getDao().getType().equals(beanClass)) {
                List<String> contentList = (List)this.getBodyList().get(i);
                if (contentList.size() > 1) {
                    throw new OutOfRangeException("core.rest.json.unable-parse-unique-bean");
                }

                itemList.add(parseObject((String)contentList.get(0), beanClass));
            }
        }

        if (itemList.isEmpty()) {
            for(i = 0; i < size; ++i) {
                if (this.getBodyServiceList().get(i) == null) {
                    List<String> contentList = this.getBodyList().get(i);
                    if (contentList.size() > 1) {
                        throw new OutOfRangeException("core.rest.json.unable-parse-unique-bean");
                    }

                    itemList.add(parseObject((String)contentList.get(0), beanClass));
                }
            }
        }

        if (itemList.size() != 1) {
            throw new OutOfRangeException("core.rest.json.unable-parse-unique-bean");
        } else {
            return itemList.get(0);
        }
    }

    public <T> List<T> parse(Class<T> beanClass) {
        int size = this.getBodyServiceList().size();
        List<T> itemList = new ArrayList<>();

        int i;
        for(i = 0; i < size; ++i) {
            GenericService<?, ?> service = this.getBodyServiceList().get(i);
            if (service != null && service.getDao().getType().equals(beanClass)) {
                this.getBodyList().get(i).forEach((b) -> {
                    itemList.add(parseObject(b, beanClass));
                });
            }
        }

        if (itemList.isEmpty()) {
            for(i = 0; i < size; ++i) {
                if (this.getBodyServiceList().get(i) == null) {
                    this.getBodyList().get(i).forEach((b) -> {
                        itemList.add(parseObject(b, beanClass));
                    });
                }
            }
        }

        return itemList;
    }

    public <T> List<T> parse(Class<T> beanClass, int index) {
        List<String> contentList = this.getBodyList().get(index);
        return contentList.stream().map((c) -> parseObject(c, beanClass)).collect(Collectors.toList());
    }

    public <ID> List<ID> parseId(Class<ID> idType) {
        if (this.bodyList.size() > 1) {
            throw new OutOfRangeException("core.rest.json.unable-parse-id");
        } else {
            return this.bodyList.isEmpty() ? CollectionUtils.emptyList() : (List)((List)this.bodyList.get(0)).stream().map((b) -> {
                Map<String, Object> original = (Map)JSONObject.parseObject(String.valueOf(b), Map.class);
                return ConvertUtils.convert(original.get("id"), idType);
            }).distinct().collect(Collectors.toList());
        }
    }

    public List<String> getAuthorityList() {
        return this.authorityList;
    }

    public void addAuthority(String authority) {
        this.authorityList.add("authority_" + authority);
    }

    public <V> void addExtFilter(String key, V value) {
        this.extFilter.put(key, value);
    }

    public Map<String, Object> getExtFilter() {
        return Collections.unmodifiableMap(this.extFilter);
    }

    public void addExtStdFilter(String key, Object value, MatchPattern matchPattern) {
        if (value == null) {
            this.extStdFilter.add(new FilterParamPattern(matchPattern.name(), key, value));
        } else {
            IdentityHashMap paramMap;
            if (List.class.isAssignableFrom(value.getClass())) {
                paramMap = new IdentityHashMap();
                Iterator var5 = ((List)value).iterator();

                while(var5.hasNext()) {
                    Object v = var5.next();
                    paramMap.put(ObjectUtils.clone(key), v);
                }

                this.extStdFilter.add(new FilterParamPattern(matchPattern.name(), paramMap));
            } else if (value.getClass().isArray()) {
                paramMap = new IdentityHashMap();
                Object[] var9 = (Object[])((Object[])value);
                int var10 = var9.length;

                for(int var7 = 0; var7 < var10; ++var7) {
                    Object v = var9[var7];
                    paramMap.put(ObjectUtils.clone(key), v);
                }

                this.extStdFilter.add(new FilterParamPattern(matchPattern.name(), paramMap));
            } else {
                this.extStdFilter.add(new FilterParamPattern(matchPattern.name(), key, value));
            }
        }

    }

    private static <T> T parseObject(String json, Class<T> clazz) {
        T proxy = BeanUtils.getPropertyListenerProxy(clazz);
        JsonUtils.parse(json, proxy);
        return proxy;
    }

    private <M extends MapperParameter> void extractFilter(M parameter, String filterJson) {
        if (!StringUtils.isEmpty(filterJson)) {
            Map<String, Object> map = JSONObject.parseObject(filterJson, new TypeReference<Map<String, Object>>() {
            });
            map.putAll(this.getExtFilter());
            List<RestJsonWrapperBean.FilterData> filterDataList = new ArrayList<>();
            map.entrySet().forEach((e) -> {
                String key = StringUtils.replace(e.getKey(), "ext$.", "");
                if (SqlHelper.isRiskySqlSegment(key)) {
                    throw new HackingDataException(key);
                } else {
                    if ("target_filter".equals(key)) {
                        Map<String, String> taMap = JSONObject.parseObject((String)e.getValue(), new TypeReference<Map<String, String>>() {
                        });
                        RestJsonWrapperBean.TargetTableFilterData filterData = new RestJsonWrapperBean.TargetTableFilterData();
                        filterData.setType(taMap.get("type"));
                        filterData.setTargetTable(taMap.get("targetTable"));
                        filterData.setTargetMatchColumn(Objects.toString(taMap.get("targetMatchColumn"), "ID"));
                        filterData.setThisMatchColumn(Objects.toString(taMap.get("thisMatchColumn"), "ID"));
                        String filter = taMap.get("filter");
                        if (!StringUtils.isEmpty(filter)) {
                            JSONArray array = JSONObject.parseArray(filter);
                            int i = 0;

                            for(int j = array.size(); i < j; ++i) {
                                JSONObject jsonObject = (JSONObject)array.get(i);
                                filterData.addTargetFilter(jsonObject.getString("targetFilterColumn"), jsonObject.getString("targetFilterValue"));
                            }
                        }

                        if (SqlHelper.isRiskySqlSegment(filterData.getTargetTable())) {
                            throw new HackingDataException("targetTable");
                        }

                        if (SqlHelper.isRiskySqlSegment(filterData.getTargetMatchColumn())) {
                            throw new HackingDataException("targetMatchColumn");
                        }

                        if (SqlHelper.isRiskySqlSegment(filterData.getThisMatchColumn())) {
                            throw new HackingDataException("thisMatchColumn");
                        }

                        filterData.getTargetFilter().keySet().forEach((k) -> {
                            if (SqlHelper.isRiskySqlSegment(k)) {
                                throw new HackingDataException("targetFilterColumn");
                            }
                        });
                        parameter.put("targetFilter", filterData);
                    } else {
                        MatchPattern matchPattern = Arrays.stream(MatchPattern.values()).filter((p) -> StringUtils.endsWithIgnoreCase(key, "_" + p.name())).findFirst().orElse(null);
                        if (matchPattern == null) {
                            parameter.put(key, e.getValue());
                        } else {
                            filterDataList.addAll(parseFilterData(key, e.getValue(), matchPattern));
                        }
                    }

                }
            });
            Map<String, List<RestJsonWrapperBean.FilterData>> filterMap = new HashMap<>();

            RestJsonWrapperBean.FilterData filterData;
            Object list;
            for(Iterator var6 = filterDataList.iterator(); var6.hasNext(); ((List)list).add(filterData)) {
                filterData = (RestJsonWrapperBean.FilterData)var6.next();
                String key = filterData.getKey() + "_" + filterData.getMatchPattern() + (StringUtils.isEmpty(filterData.getGroup()) ? "" : ":" + filterData.getGroup());
                list = filterMap.get(key);
                if (list == null) {
                    list = new ArrayList<>();
                    filterMap.put(key, (List<FilterData>) list);
                }
            }

            if (!filterMap.isEmpty()) {
                List<FilterParamPattern> filterParamList = new ArrayList<>();
                filterMap.forEach((e, k) -> {
                    Map<String, Object> paramMap = new IdentityHashMap();
                    k.forEach((v) -> {
                        paramMap.put(ObjectUtils.clone(v.getKey()), v.getValue());
                    });
                    filterParamList.add(new FilterParamPattern(k.get(0).getMatchPattern(), paramMap));
                });
                FilterParamPattern[] paramPatterns = ArrayUtils.concat((FilterParamPattern[])parameter.get("filterParamPatterns"), filterParamList.toArray(new FilterParamPattern[0]));
                parameter.put("filterParamPatterns", paramPatterns);
            }
        }

    }

    /**
     * 解析过滤器
     * @param originalName 过滤条件key
     * @param originalValue 过滤条件value
     * @param matchPattern 匹配规则
     * @return FilterData List
     */
    private static List<RestJsonWrapperBean.FilterData> parseFilterData(String originalName, Object originalValue, MatchPattern matchPattern) {
        if ("VOID".equals(originalValue)) {
            originalValue = null;
        }

        List<RestJsonWrapperBean.FilterData> filterDataList = new ArrayList<>();
        String name = StringUtils.removeEndIgnoreCase(originalName.toUpperCase(), "_" + matchPattern.name());
        String filterSeparator = ApplicationContextHelper.getEnvironment().getProperty("sino.search.multi-value-matching-seperator");
        filterSeparator = StringUtils.isBlank(filterSeparator) ? "," : filterSeparator;
        if (MatchPattern.DEQ.equals(matchPattern)) {
            LocalDate date = matchPattern.parse(originalValue);
            if (date == null) {
                filterDataList.add(new RestJsonWrapperBean.FilterData(name, null, matchPattern.name()));
            } else {
                filterDataList.add(new RestJsonWrapperBean.FilterData(name, DateTimeUtils.formatLocalDate(date), matchPattern.name()));
            }
        } else if (MatchPattern.TEQ.equals(matchPattern)) {
            LocalDateTime date = matchPattern.parse(originalValue);
            if (date == null) {
                filterDataList.add(new RestJsonWrapperBean.FilterData(name, null, matchPattern.name()));
            } else {
                filterDataList.add(new RestJsonWrapperBean.FilterData(name, DateTimeUtils.formatLocalDateTime(date, "yyyy-MM-dd HH:mm"), matchPattern.name()));
            }
        } else {
            String[] strArray;
            int i;
            String value;
            LocalDate parsedValue;
            String group = null;
            if (MatchPattern.DR.equals(matchPattern)) {
                if (originalValue != null) {
                    group = ((String)originalValue).substring(1, ((String)originalValue).length() - 1);
                }
                strArray = StringUtils.split(group, ",");
                if (strArray.length != 2) {
                    throw new OutOfRangeException("Range filter pattern must contains two values（can be empty）.");
                }

                for(i = 0; i < 2; ++i) {
                    value = strArray[i];
                    if (!StringUtils.isEmpty(value)) {
                        parsedValue = matchPattern.parse(value);
                        if (i == 0) {
                            if (StringUtils.startsWith((String)originalValue, "(")) {
                                filterDataList.add(new RestJsonWrapperBean.FilterData(name, parsedValue, MatchPattern.DG.name()));
                            } else {
                                filterDataList.add(new RestJsonWrapperBean.FilterData(name, parsedValue, MatchPattern.DGOE.name()));
                            }
                        } else if (StringUtils.endsWith((String)originalValue, ")")) {
                            filterDataList.add(new RestJsonWrapperBean.FilterData(name, parsedValue, MatchPattern.DL.name()));
                        } else {
                            filterDataList.add(new RestJsonWrapperBean.FilterData(name, parsedValue.plusDays(1L), MatchPattern.DLOE.name()));
                        }
                    }
                }
            } else if (MatchPattern.TR.equals(matchPattern)) {
                group = ((String)originalValue).substring(1, ((String)originalValue).length() - 1);
                strArray = StringUtils.split(group, ",");
                if (strArray.length != 2) {
                    throw new OutOfRangeException("Range filter pattern must contains two values（can be empty）.");
                }

                for(i = 0; i < 2; ++i) {
                    value = strArray[i];
                    if (!StringUtils.isEmpty(value)) {
                        parsedValue = matchPattern.parse(value);
                        if (i == 0) {
                            if (StringUtils.startsWith((String)originalValue, "(")) {
                                filterDataList.add(new RestJsonWrapperBean.FilterData(name, parsedValue, MatchPattern.TG.name()));
                            } else {
                                filterDataList.add(new RestJsonWrapperBean.FilterData(name, parsedValue, MatchPattern.TGOE.name()));
                            }
                        } else if (StringUtils.endsWith((String)originalValue, ")")) {
                            filterDataList.add(new RestJsonWrapperBean.FilterData(name, parsedValue, MatchPattern.TL.name()));
                        } else {
                            filterDataList.add(new RestJsonWrapperBean.FilterData(name, parsedValue.plusDays(1L), MatchPattern.TLOE.name()));
                        }
                    }
                }
            } else if (MatchPattern.NR.equals(matchPattern)) {
                group = ((String)originalValue).substring(1, ((String)originalValue).length() - 1);
                strArray = StringUtils.split(group, ",");
                if (strArray.length != 2) {
                    throw new OutOfRangeException("Range filter pattern must contains two values（can be empty）.");
                }

                for(i = 0; i < 2; ++i) {
                    value = strArray[i];
                    if (!StringUtils.isEmpty(value)) {
                        parsedValue = matchPattern.parse(value);
                        if (i == 0) {
                            if (StringUtils.startsWith((String)originalValue, "(")) {
                                filterDataList.add(new RestJsonWrapperBean.FilterData(name, parsedValue, MatchPattern.NG.name()));
                            } else {
                                filterDataList.add(new RestJsonWrapperBean.FilterData(name, parsedValue, MatchPattern.NGOE.name()));
                            }
                        } else if (StringUtils.endsWith((String)originalValue, ")")) {
                            filterDataList.add(new RestJsonWrapperBean.FilterData(name, parsedValue, MatchPattern.NL.name()));
                        } else {
                            filterDataList.add(new RestJsonWrapperBean.FilterData(name, parsedValue, MatchPattern.NLOE.name()));
                        }
                    }
                }
            } else {
                String str;
                String[] values;
                int var18;
                int var21;
                if (MatchPattern.MULTIPLE.equals(matchPattern)) {
                    group = System.nanoTime() + "";
                    strArray = StringUtils.split(name, ",");
                    values = strArray;
                    var18 = strArray.length;

                    for(var21 = 0; var21 < var18; ++var21) {
                        str = values[var21];
                        filterDataList.add(new RestJsonWrapperBean.FilterData(str, originalValue, MatchPattern.OR.name(), group));
                    }
                } else if (!MatchPattern.NOR.equals(matchPattern) && !MatchPattern.DOR.equals(matchPattern) && !MatchPattern.TOR.equals(matchPattern) && !MatchPattern.OR.equals(matchPattern)) {
                    String[] var19;
                    int var22;
                    if (!MatchPattern.SB.equals(matchPattern) && !MatchPattern.SC.equals(matchPattern) && !MatchPattern.SE.equals(matchPattern) && !MatchPattern.SEQ.equals(matchPattern) && !MatchPattern.CISB.equals(matchPattern) && !MatchPattern.CISC.equals(matchPattern) && !MatchPattern.CISE.equals(matchPattern) && !MatchPattern.CISEQ.equals(matchPattern)) {
                        if (MatchPattern.CBM.equals(matchPattern)) {
                            group = matchPattern.parse(originalValue);
                            if (!StringUtils.isBlank(group)) {
                                group = System.nanoTime() + "";
                                values = StringUtils.split(group, ",");
                                var19 = values;
                                var21 = values.length;

                                for(var22 = 0; var22 < var21; ++var22) {
                                    value = var19[var22];
                                    filterDataList.add(new RestJsonWrapperBean.FilterData(name, value, MatchPattern.CBM.name(), group));
                                }
                            }
                        } else if (MatchPattern.IN.equals(matchPattern)) {
                            group = StringUtils.randomUUID(16);
                            strArray = StringUtils.split(ConvertUtils.convert(originalValue, String.class), ",");
                            values = strArray;
                            var18 = strArray.length;

                            for(var21 = 0; var21 < var18; ++var21) {
                                str = values[var21];
                                filterDataList.add(new RestJsonWrapperBean.FilterData(name, matchPattern.parse(str), MatchPattern.OR.name(), group));
                            }
                        } else if (originalValue != null && String.class.equals(originalValue.getClass()) && StringUtils.contains((String)originalValue, filterSeparator)) {
                            group = StringUtils.randomUUID(16);
                            strArray = StringUtils.split((String)originalValue, filterSeparator);
                            values = strArray;
                            var18 = strArray.length;

                            for(var21 = 0; var21 < var18; ++var21) {
                                str = values[var21];
                                filterDataList.add(new RestJsonWrapperBean.FilterData(name, matchPattern.parse(str), matchPattern.name(), group));
                            }
                        } else {
                            filterDataList.add(new RestJsonWrapperBean.FilterData(name, matchPattern.parse(originalValue), matchPattern.name()));
                        }
                    } else {
                        group = matchPattern.parse(originalValue);
                        if (group != null && StringUtils.startsWith(matchPattern.name(), "CI")) {
                            group = group.toUpperCase();
                        }

                        if (StringUtils.contains(group, filterSeparator)) {
                            group = StringUtils.randomUUID(16);
                            values = StringUtils.split(group, filterSeparator);
                            var19 = values;
                            var21 = values.length;

                            for(var22 = 0; var22 < var21; ++var22) {
                                value = var19[var22];
                                filterDataList.add(new RestJsonWrapperBean.FilterData(name, value, "O" + matchPattern.name(), group));
                            }
                        } else {
                            filterDataList.add(new RestJsonWrapperBean.FilterData(name, group, matchPattern.name()));
                        }
                    }
                } else if (StringUtils.contains(name, "_")) {
                    int index = name.lastIndexOf("_");
                    filterDataList.add(new RestJsonWrapperBean.FilterData(name.substring(0, index), matchPattern.parse(originalValue), MatchPattern.OR.name(), name.substring(index + 1)));
                } else {
                    filterDataList.add(new RestJsonWrapperBean.FilterData(name, matchPattern.parse(originalValue), MatchPattern.EQ.name()));
                }
            }
        }

        return filterDataList;
    }

    static {
        parserConfig.putDeserializer(LocalDateTime.class, DefaultJsonDeserializer.INSTANCE);
        parserConfig.putDeserializer(LocalDate.class, DefaultJsonDeserializer.INSTANCE);
        parserConfig.putDeserializer(Time.class, DefaultJsonDeserializer.INSTANCE);
        parserConfig.putDeserializer(Date.class, DefaultJsonDeserializer.INSTANCE);
    }

    public static class TargetTableFilterData {
        private String type;
        private String targetTable;
        private String targetMatchColumn;
        private String thisMatchColumn;
        private Map<String, Object> targetFilter = new HashMap<>();

        public TargetTableFilterData() {
        }

        public String getType() {
            return this.type;
        }

        public void setType(String type) {
            this.type = type;
        }

        public String getTargetTable() {
            return this.targetTable;
        }

        public void setTargetTable(String targetTable) {
            this.targetTable = targetTable;
        }

        public String getTargetMatchColumn() {
            return this.targetMatchColumn;
        }

        public void setTargetMatchColumn(String targetMatchColumn) {
            this.targetMatchColumn = targetMatchColumn;
        }

        public String getThisMatchColumn() {
            return this.thisMatchColumn;
        }

        public void setThisMatchColumn(String thisMatchColumn) {
            this.thisMatchColumn = thisMatchColumn;
        }

        public Map<String, Object> getTargetFilter() {
            return this.targetFilter;
        }

        public void addTargetFilter(String targetFilterColumn, String targetFilterValue) {
            this.targetFilter.put(targetFilterColumn, targetFilterValue);
        }
    }

    private static class FilterData {
        private final String key;
        private final Object value;
        private final String matchPattern;
        private String group;

        public FilterData(String key, Object value, String matchPattern, String group) {
            this.key = key;
            this.value = value;
            this.matchPattern = matchPattern;
            this.group = group;
        }

        public FilterData(String key, Object value, String matchPattern) {
            this.key = key;
            this.value = value;
            this.matchPattern = matchPattern;
        }

        public String getKey() {
            return this.key;
        }

        public Object getValue() {
            return this.value;
        }

        public String getMatchPattern() {
            return this.matchPattern;
        }

        public String getGroup() {
            return this.group;
        }
    }
}
