/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.mybatis.dao;

import com.alibaba.fastjson.JSONObject;
import com.alibaba.fastjson.TypeReference;
import com.alibaba.fastjson.parser.Feature;
import net.sinodawn.framework.at.AuditTrailHelper;
import net.sinodawn.framework.at.annotation.AuditTrailType;
import net.sinodawn.framework.at.aspect.AuditTrailAspect;
import net.sinodawn.framework.at.aspect.AuditTrailAspect.AuditTrailEntryData;
import net.sinodawn.framework.audit.AuditDatabaseOperation;
import net.sinodawn.framework.audit.AuditLogHelper;
import net.sinodawn.framework.beans.BeanPropertyDescriptor;
import net.sinodawn.framework.beans.BeanPropertyEvent;
import net.sinodawn.framework.beans.BeanPropertyHelper;
import net.sinodawn.framework.beans.BeanPropertyListener;
import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.context.SinoAopContext;
import net.sinodawn.framework.context.SinoBeanContext;
import net.sinodawn.framework.data.ListChunkIterator;
import net.sinodawn.framework.data.annotation.NotNull;
import net.sinodawn.framework.data.page.Page;
import net.sinodawn.framework.data.page.Pageable;
import net.sinodawn.framework.database.context.EntityColumnContext;
import net.sinodawn.framework.database.context.EntityContext;
import net.sinodawn.framework.context.LocalContextHelper;
import net.sinodawn.framework.database.context.instance.EntityColumnContextInstance;
import net.sinodawn.framework.database.context.instance.EntityContextInstance;
import net.sinodawn.framework.database.context.instance.EntityHelper;
import net.sinodawn.framework.database.core.DatabaseManager;
import net.sinodawn.framework.database.dialect.Dialect;
import net.sinodawn.framework.database.sql.Order;
import net.sinodawn.framework.database.sql.SqlType;
import net.sinodawn.framework.exception.checked.CheckedException;
import net.sinodawn.framework.exception.database.JdbcException;
import net.sinodawn.framework.exception.database.NotUniqueResultJdbcException;
import net.sinodawn.framework.mybatis.mapper.DaoMapper;
import net.sinodawn.framework.mybatis.page.MybatisPageHelper;
import net.sinodawn.framework.mybatis.page.PageRowBounds;
import net.sinodawn.framework.restful.data.RestFieldValueTextContainer;
import net.sinodawn.framework.support.PersistableHelper;
import net.sinodawn.framework.support.PersistableMetadataHelper;
import net.sinodawn.framework.support.base.dao.GenericDao;
import net.sinodawn.framework.support.base.service.GenericService;
import net.sinodawn.framework.support.domain.Orderable;
import net.sinodawn.framework.support.domain.Persistable;
import net.sinodawn.framework.utils.*;
import net.sinodawn.module.item.file.bean.CoreFileBean;
import net.sinodawn.module.sys.at.bean.CoreAuditTrailRecordLineBean;
import org.mybatis.spring.SqlSessionTemplate;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.ResolvableType;
import org.springframework.stereotype.Repository;

import java.io.Serializable;
import java.lang.reflect.Method;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.util.*;
import java.util.stream.Collectors;

@SuppressWarnings({"unused", "unchecked", "rawtypes"})
@Repository
public abstract class MybatisDaoSupport<T extends Persistable<ID>, ID extends Serializable> implements GenericDao<T, ID> {
    @Autowired
    private Dialect dialect;
    @Autowired
    private DaoMapper daoMapper;
    @Autowired
    private SqlSessionTemplate mybatisTemplate;
    /**
     * 泛型T.class
     */
    private final Class<T> type = (Class<T>) ResolvableType.forClass(this.getClass()).as(GenericDao.class).getGeneric(0).resolve();

    public MybatisDaoSupport() {
    }

    public Class<T> getType() {
        return this.type;
    }

    public EntityContext getEntityContext() {
        return DatabaseManager.getEntityContext(this.type);
    }

    public void insert(T item) {
        if (null != item) {
            this.preInsert(Collections.singletonList(item));
            EntityContextInstance contextInstance = EntityContextInstance.insertable(item);
            this.daoMapper.insert(contextInstance);
            if (AuditTrailHelper.auditTrail() && AuditTrailHelper.auditTrailable(this.getTable(), AuditDatabaseOperation.INSERT)) {
                AuditTrailAspect.AuditTrailEntryData auditTrailEntryData = AuditTrailAspect.getAuditTrailEntryData();
                if (auditTrailEntryData != null) {
                    String table = this.getTable();
                    List<EntityColumnContextInstance> columnContextInstanceList = contextInstance.getColumnContextInstanceList();
                    ID id = item.getId();
                    List<CoreAuditTrailRecordLineBean> recordLineList = new ArrayList();

                    for (EntityColumnContextInstance columnContextInstance : columnContextInstanceList) {
                        if (AuditTrailHelper.auditTrailableColumn(table, columnContextInstance.getColumnContext().getColumnName())) {
                            CoreAuditTrailRecordLineBean line = new CoreAuditTrailRecordLineBean();
                            line.setRecordType(SqlType.INSERT.name());
                            line.setTargetId(table + "$" + id);
                            line.setTableName(table);
                            line.setColumn(columnContextInstance.getColumnContext().getColumnName());
                            line.setNewValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(columnContextInstance.getValue())));
                            recordLineList.add(line);
                        }
                    }

                    if (!recordLineList.isEmpty()) {
                        String targetId = AuditTrailAspect.getAncientTargetId(table, id);
                        AuditTrailAspect.addRecordLine(targetId, recordLineList);
                    }
                }
            }

            boolean cacheEvictOverrided = this.isCacheEvictOverrided();
            boolean auditable = AuditLogHelper.auditable(AuditDatabaseOperation.INSERT);
            if (cacheEvictOverrided || auditable) {
                T selectedItem = this.selectById(item.getId());
                if (auditable) {
                    AuditLogHelper.insertAuditLog(this.getTable(), null, selectedItem);
                }

                ((GenericDao) SinoAopContext.currentProxy()).cacheEvict(null, selectedItem);
            }

        }
    }

    /**
     * 批量插入Bean
     */
    public void insert(List<T> itemList) {
        if (null != itemList && !itemList.isEmpty()) {
            this.preInsert(itemList);
            EntityContext context = this.getEntityContext();
            int columnQty = context.getColumnContextList().size();
            int loopItemQty = Math.min(this.dialect.getMaxBatchInsertQty(), this.dialect.getMaxParamQty() / columnQty);
            ListChunkIterator chunkIterator = ListChunkIterator.of(itemList, loopItemQty);

            while(chunkIterator.hasNext()) {
                this.daoMapper.batchInsert(context,
                    (List)chunkIterator.nextChunk().stream().map(
                        (t) -> EntityContextInstance.insertable((T)t)
                    ).collect(Collectors.toList()));
            }

            if (AuditTrailHelper.auditTrail() && AuditTrailHelper.auditTrailable(this.getTable(), AuditDatabaseOperation.INSERT)) {
                AuditTrailAspect.AuditTrailEntryData auditTrailEntryData = AuditTrailAspect.getAuditTrailEntryData();
                if (auditTrailEntryData != null) {
                    String table = this.getTable();
                    Map<ID, List<CoreAuditTrailRecordLineBean>> map = new HashMap();

                    for (T item : itemList) {
                        ID lineId = item.getId();
                        List<CoreAuditTrailRecordLineBean> recordLineList = new ArrayList();

                        for (EntityColumnContext columnContext : context.getColumnContextList()) {
                            if (AuditTrailHelper.auditTrailableColumn(table, columnContext.getColumnName())) {
                                CoreAuditTrailRecordLineBean line = new CoreAuditTrailRecordLineBean();
                                line.setRecordType(SqlType.INSERT.name());
                                line.setTargetId(table + "$" + lineId);
                                line.setTableName(table);
                                line.setColumn(columnContext.getColumnName());
                                line.setNewValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(columnContext.getValue(item))));
                                recordLineList.add(line);
                            }
                        }

                        if (!recordLineList.isEmpty()) {
                            map.put(lineId, recordLineList);
                        }
                    }

                    if (!map.isEmpty()) {
                        Map<ID, String> targetMap = AuditTrailAspect.getAncientTargetMap(table, new ArrayList(map.keySet()));
                        targetMap.forEach((k, v) -> {
                            AuditTrailAspect.addRecordLine(v, map.get(k));
                        });
                    }
                }
            }

            boolean cacheEvictOverrided = this.isCacheEvictOverrided();
            boolean auditable = AuditLogHelper.auditable(AuditDatabaseOperation.INSERT);
            if (cacheEvictOverrided || auditable) {
                List<ID> idList = (List)itemList.stream().map((i) -> (Serializable)i.getId()).collect(Collectors.toList());
                List<T> selectedItemList = this.selectListByIds(idList, new Order[0]);
                Iterator var21;
                Persistable selectedItem;
                if (AuditLogHelper.auditable(AuditDatabaseOperation.INSERT)) {
                    var21 = selectedItemList.iterator();

                    while(var21.hasNext()) {
                        selectedItem = (Persistable)var21.next();
                        AuditLogHelper.insertAuditLog(this.getTable(), null, selectedItem);
                    }
                }

                var21 = selectedItemList.iterator();

                while(var21.hasNext()) {
                    selectedItem = (Persistable)var21.next();
                    ((GenericDao)SinoAopContext.currentProxy()).cacheEvict(null, selectedItem);
                }
            }

        }
    }

    public void delete(ID id) {
        if (id == null) {
            throw new JdbcException("SINO.EXCEPTION.DELETE_REQUIRE_ID");
        } else {
            T selectedItem = this.selectByIdIfPresent(id);
            if (selectedItem != null) {
                EntityContextInstance contextInstance = EntityContextInstance.instance(selectedItem, new String[0]);
                if (AuditTrailHelper.auditTrail() && AuditTrailHelper.auditTrailable(this.getTable(), AuditDatabaseOperation.DELETE)) {
                    AuditTrailAspect.AuditTrailEntryData auditTrailEntryData = AuditTrailAspect.getAuditTrailEntryData();
                    if (auditTrailEntryData != null) {
                        String table = this.getTable();
                        CoreAuditTrailRecordLineBean line = new CoreAuditTrailRecordLineBean();
                        line.setRecordType(SqlType.DELETE.name());
                        line.setTargetId(table + "$" + id);
                        line.setTableName(table);
                        line.setColumn("ID");
                        line.setOldValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(id)));
                        String targetId = AuditTrailAspect.getAncientTargetId(table, id);
                        AuditTrailAspect.addRecordLine(targetId, line);
                    }
                }

                this.daoMapper.deleteById(contextInstance);
                if (AuditLogHelper.auditable(AuditDatabaseOperation.DELETE)) {
                    AuditLogHelper.insertAuditLog(this.getTable(), selectedItem, null);
                }

                ((GenericDao)SinoAopContext.currentProxy()).cacheEvict(selectedItem, null);
            }
        }
    }

    public void deleteByIdList(List<ID> idList) {
        if (idList != null && !idList.isEmpty()) {
            if (idList.stream().anyMatch((i) -> {
                return i == null;
            })) {
                throw new JdbcException("SINO.EXCEPTION.DELETE_REQUIRE_ID");
            } else {
                if (AuditTrailHelper.auditTrail() && AuditTrailHelper.auditTrailable(this.getTable(), AuditDatabaseOperation.DELETE)) {
                    AuditTrailAspect.AuditTrailEntryData auditTrailEntryData = AuditTrailAspect.getAuditTrailEntryData();
                    if (auditTrailEntryData != null) {
                        String table = this.getTable();
                        Map<ID, CoreAuditTrailRecordLineBean> map = new HashMap();

                        for (ID value : idList) {
                            ID id = (ID) value;
                            CoreAuditTrailRecordLineBean line = new CoreAuditTrailRecordLineBean();
                            line.setRecordType(SqlType.DELETE.name());
                            line.setTargetId(table + "$" + id);
                            line.setTableName(table);
                            line.setColumn("ID");
                            line.setOldValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(id)));
                            map.put(id, line);
                        }

                        if (!map.isEmpty()) {
                            Map<ID, String> targetMap = AuditTrailAspect.getAncientTargetMap(table, new ArrayList(map.keySet()));
                            targetMap.forEach((k, v) -> {
                                AuditTrailAspect.addRecordLine(v, (CoreAuditTrailRecordLineBean)map.get(k));
                            });
                        }
                    }
                }

                List<T> selectedItemList = this.selectListByIds(idList, new Order[0]);
                ListChunkIterator chunkIterator = ListChunkIterator.of(selectedItemList, this.dialect.getInClauseIdMaxQty());

                while(chunkIterator.hasNext()) {
                    this.daoMapper.batchDeleteById(this.getEntityContext(), (List)chunkIterator.nextChunk().stream().map((t) -> {
                        return EntityContextInstance.instance((T)t);
                    }).collect(Collectors.toList()));
                }

                if (AuditLogHelper.auditable(AuditDatabaseOperation.DELETE)) {

                    for (T t : selectedItemList) {
                        T selectedItem = (T) t;
                        AuditLogHelper.insertAuditLog(this.getTable(), selectedItem, (Persistable) null);
                    }
                }

                selectedItemList.forEach((i) -> {
                    ((GenericDao)SinoAopContext.currentProxy()).cacheEvict(i, null);
                });
            }
        }
    }

    public void deleteBy(List<T> itemList, String... searchColNames) {
        if (itemList != null && !itemList.isEmpty()) {
            if (ArrayUtils.isEmpty(searchColNames) && itemList.stream().anyMatch((i) -> {
                return BeanUtils.isEmpty(i);
            })) {
                throw new JdbcException("SINO.EXCEPTION.DELETE_REQUIRE_NONE_EMPTY_BEAN");
            } else {
                List<ID> idList = ArrayUtils.isEmpty(searchColNames) ? (List)itemList.stream().filter((i) -> {
                    return i.getId() != null;
                }).map((i) -> {
                    return (Serializable)i.getId();
                }).collect(Collectors.toList()) : this.selectIdList(itemList, ArrayUtils.asList(searchColNames), new Order[0]);
                if (!idList.isEmpty()) {
                    this.deleteByIdList(idList);
                }

            }
        }
    }

    public void update(T item, List<String> updateColNameList, String... searchColNames) {
        List<String> clone = updateColNameList == null ? new ArrayList() : new ArrayList(updateColNameList);
        EntityContextInstance contextInstance = EntityContextInstance.updatable(item);
        if (!clone.isEmpty()) {
            contextInstance.getColumnContextInstanceList().forEach((c) -> {
                c.setActive(clone.stream().anyMatch((n) -> {
                    return n.equalsIgnoreCase(c.getColumnContext().getColumnName());
                }));
            });
        }

        if (EntityHelper.isColumnUpdated(contextInstance)) {
            List<T> oldItemList = new ArrayList();
            List<String> updatedColumnNameList = new ArrayList();
            String table;
            if (ArrayUtils.isEmpty(searchColNames)) {
                T oldItem = this.selectById((ID) item.getId());
                if (AuditTrailHelper.auditTrail() && AuditTrailHelper.auditTrailable(this.getTable(), AuditDatabaseOperation.UPDATE)) {
                    AuditTrailAspect.AuditTrailEntryData auditTrailEntryData = AuditTrailAspect.getAuditTrailEntryData();
                    if (auditTrailEntryData != null) {
                        table = this.getTable();
                        List<CoreAuditTrailRecordLineBean> recordLineList = new ArrayList();
                        String finalTable1 = table;
                        String finalTable2 = table;
                        contextInstance.getColumnContextInstanceList().stream().filter((c) -> {
                            return c.isActive() && AuditTrailHelper.auditTrailableColumn(finalTable1, c.getColumnContext().getColumnName());
                        }).forEach((c) -> {
                            CoreAuditTrailRecordLineBean line = new CoreAuditTrailRecordLineBean();
                            line.setRecordType(SqlType.UPDATE.name());
                            line.setTargetId(finalTable2 + "$" + item.getId());
                            line.setTableName(finalTable2);
                            line.setColumn(c.getColumnContext().getColumnName().toUpperCase());
                            line.setOldValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(c.getColumnContext().getValue(oldItem))));
                            line.setNewValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(c.getValue())));
                            recordLineList.add(line);
                        });
                        if (!recordLineList.isEmpty()) {
                            table = AuditTrailAspect.getAncientTargetId(table, (Serializable)item.getId());
                            AuditTrailAspect.addRecordLine(table, recordLineList);
                        }
                    }
                }

                oldItemList.add(oldItem);
                this.daoMapper.updateById(contextInstance);
                contextInstance.getColumnContextInstanceList().stream().filter(EntityColumnContextInstance::isActive).forEach((c) -> {
                    updatedColumnNameList.add(c.getColumnContext().getColumnName());
                });
            } else {
                oldItemList.addAll(this.selectList(item, ArrayUtils.asList(searchColNames), CollectionUtils.emptyList()));
                if (!oldItemList.isEmpty()) {
                    int chunkSize = this.dialect.getInClauseIdMaxQty();

                    List<Persistable> chunkList;
                    for(ListChunkIterator chunkIterator = ListChunkIterator.of(oldItemList, chunkSize);
                        chunkIterator.hasNext();
                        this.daoMapper.updateByIds(contextInstance,
                                chunkList.stream().map(Persistable::getId)
                                        .collect(Collectors.toList())
                        )) {
                        chunkList = chunkIterator.nextChunk();
                        if (AuditTrailHelper.auditTrail() && AuditTrailHelper.auditTrailable(this.getTable(), AuditDatabaseOperation.UPDATE)) {
                            AuditTrailAspect.AuditTrailEntryData auditTrailEntryData = AuditTrailAspect.getAuditTrailEntryData();
                            if (auditTrailEntryData != null) {
                                table = this.getTable();
                                Map<ID, List<CoreAuditTrailRecordLineBean>> map = new HashMap();

                                for (Object o : chunkList) {
                                    T oldItem = (T) o;
                                    List<CoreAuditTrailRecordLineBean> recordLineList = new ArrayList();
                                    String finalTable = table;
                                    String finalTable3 = table;
                                    contextInstance.getColumnContextInstanceList().stream().filter((c) -> {
                                        return c.isActive() && AuditTrailHelper.auditTrailableColumn(finalTable3, c.getColumnContext().getColumnName());
                                    }).forEach((c) -> {
                                        Object oldValue = c.getColumnContext().getValue(oldItem);
                                        if (!ObjectUtils.equals(oldValue, c.getValue())) {
                                            CoreAuditTrailRecordLineBean line = new CoreAuditTrailRecordLineBean();
                                            line.setRecordType(SqlType.UPDATE.name());
                                            line.setTargetId(finalTable + "$" + oldItem.getId());
                                            line.setTableName(finalTable);
                                            line.setColumn(c.getColumnContext().getColumnName().toUpperCase());
                                            line.setOldValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(oldValue)));
                                            line.setNewValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(c.getValue())));
                                            recordLineList.add(line);
                                        }

                                    });
                                    if (!recordLineList.isEmpty()) {
                                        map.put(oldItem.getId(), recordLineList);
                                    }
                                }

                                if (!map.isEmpty()) {
                                    Map<ID, String> targetMap = AuditTrailAspect.getAncientTargetMap(table, new ArrayList(map.keySet()));
                                    targetMap.forEach((k, v) -> {
                                        AuditTrailAspect.addRecordLine(v, (List)map.get(k));
                                    });
                                }
                            }
                        }
                    }

                    contextInstance.getColumnContextInstanceList().stream().filter((c) -> {
                        return c.isActive();
                    }).forEach((c) -> {
                        updatedColumnNameList.add(c.getColumnContext().getColumnName());
                    });
                }
            }

            List<ID> idList = (List)oldItemList.stream().map((i) -> {
                return (Serializable)i.getId();
            }).collect(Collectors.toList());
            boolean cacheEvictOverrided = this.isCacheEvictOverrided();
            boolean auditable = AuditLogHelper.auditable(AuditDatabaseOperation.UPDATE);
            if (cacheEvictOverrided || auditable) {
                List<T> newItemList = this.selectListByIds(idList);
                this.auditUpdate(oldItemList, newItemList);
                Iterator var26 = newItemList.iterator();

                while(var26.hasNext()) {
                    T newItem = (T) var26.next();
                    T oldItem = oldItemList.stream().filter((i) -> i.getId().equals(newItem.getId())).findAny().orElse(null);
                    ((GenericDao)SinoAopContext.currentProxy()).cacheEvict(oldItem, newItem);
                }
            }

            GenericService<T, ID> service = SinoBeanContext.getServiceByTable(this.getTable());
            if (service != null) {
                service.postUpdate(updatedColumnNameList, idList);
            }

        }
    }

    public void updateByIds(T item, List<ID> idList, String... updateColNames) {
        EntityContextInstance contextInstance = EntityContextInstance.updatable(item);
        List<T> oldItemList;
        if (updateColNames != null && updateColNames.length > 0) {
            List finalOldItemList =  ArrayUtils.asList(updateColNames);
            contextInstance.getColumnContextInstanceList().forEach((p) -> {
                p.setActive(CollectionUtils.containsIgnoreCase(finalOldItemList, p.getColumnContext().getColumnName()));
            });
        }

        if (EntityHelper.isColumnUpdated(contextInstance)) {
            oldItemList = this.selectListByIds(idList);
            int chunkSize = this.dialect.getInClauseIdMaxQty();

            List<Persistable> chunkList;
            Persistable oldItem;
            for(ListChunkIterator chunkIterator = ListChunkIterator.of(oldItemList, chunkSize);
                chunkIterator.hasNext();
                this.daoMapper.updateByIds(contextInstance, chunkList.stream().map(Persistable::getId).collect(Collectors.toList()))
            ) {
                chunkList = chunkIterator.nextChunk();
                if (AuditTrailHelper.auditTrail() && AuditTrailHelper.auditTrailable(this.getTable(), AuditDatabaseOperation.UPDATE)) {
                    AuditTrailAspect.AuditTrailEntryData auditTrailEntryData = AuditTrailAspect.getAuditTrailEntryData();
                    if (auditTrailEntryData != null) {
                        String table = this.getTable();
                        Map<ID, List<CoreAuditTrailRecordLineBean>> map = new HashMap();

                        for (Persistable persistable : chunkList) {
                            oldItem = persistable;
                            List<CoreAuditTrailRecordLineBean> recordLineList = new ArrayList();
                            Persistable finalOldItem = oldItem;
                            contextInstance.getColumnContextInstanceList().stream().filter((c) -> {
                                return c.isActive() && AuditTrailHelper.auditTrailableColumn(table, c.getColumnContext().getColumnName());
                            }).forEach((c) -> {
                                Object oldValue = c.getColumnContext().getValue(finalOldItem);
                                if (!ObjectUtils.equals(oldValue, c.getValue())) {
                                    CoreAuditTrailRecordLineBean line = new CoreAuditTrailRecordLineBean();
                                    line.setRecordType(SqlType.UPDATE.name());
                                    line.setTargetId(table + "$" + finalOldItem.getId());
                                    line.setTableName(table);
                                    line.setColumn(c.getColumnContext().getColumnName().toUpperCase());
                                    line.setOldValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(oldValue)));
                                    line.setNewValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(c.getValue())));
                                    recordLineList.add(line);
                                }

                            });
                            if (!recordLineList.isEmpty()) {
                                map.put((ID) oldItem.getId(), recordLineList);
                            }
                        }

                        if (!map.isEmpty()) {
                            Map<ID, String> targetMap = AuditTrailAspect.getAncientTargetMap(table, new ArrayList(map.keySet()));
                            targetMap.forEach((k, v) -> {
                                AuditTrailAspect.addRecordLine(v, (List)map.get(k));
                            });
                        }
                    }
                }
            }

            boolean cacheEvictOverrided = this.isCacheEvictOverrided();
            boolean auditable = AuditLogHelper.auditable(AuditDatabaseOperation.UPDATE);
            List updatedColumnNameList;
            if (cacheEvictOverrided || auditable) {
                updatedColumnNameList = this.selectListByIds(idList, new Order[0]);
                this.auditUpdate(oldItemList, updatedColumnNameList);

                for (Object o : updatedColumnNameList) {
                    T newItem = (T) o;
                    oldItem = (Persistable) oldItemList.stream().filter((i) -> {
                        return ((Serializable) i.getId()).equals(newItem.getId());
                    }).findAny().orElse(null);
                    ((GenericDao) SinoAopContext.currentProxy()).cacheEvict(oldItem, newItem);
                }
            }

            updatedColumnNameList = (List)contextInstance.getColumnContextInstanceList().stream().filter((c) -> {
                return c.isActive();
            }).map((c) -> {
                return c.getColumnContext().getColumnName();
            }).collect(Collectors.toList());
            GenericService<T, ID> service = SinoBeanContext.getServiceByTable(this.getTable());
            if (service != null) {
                service.postUpdate(updatedColumnNameList, idList);
            }

        }
    }

    public void update(List<T> itemList, String... updateColNames) {
        if (itemList != null && !itemList.isEmpty()) {
            List<String> updateColNameList = new ArrayList();
            if (ArrayUtils.isEmpty(updateColNames)) {
                T item = itemList.get(0);
                EntityContextInstance contextInstance = EntityContextInstance.updatable(item, new String[0]);
                contextInstance.getColumnContextInstanceList().forEach((i) -> {
                    if (i.isActive() && !EntityHelper.isIdInstance(i)) {
                        updateColNameList.add(i.getColumnContext().getColumnName());
                    }

                });
            } else {
                updateColNameList.addAll(Arrays.asList(updateColNames));
            }

            if (!updateColNameList.isEmpty()) {
                List<ID> idList = (List)itemList.stream().map((i) -> {
                    return (Serializable)i.getId();
                }).collect(Collectors.toList());
                List<T> oldItemList = this.selectListByIds(idList, new Order[0]);
                int updateColQty = updateColNameList.size();
                int chunkSize = this.dialect.getInClauseIdMaxQty();
                int maxParamQty = this.dialect.getMaxParamQty();
                if (maxParamQty < chunkSize * (updateColQty * 2 + 1)) {
                    chunkSize = maxParamQty / (updateColQty * 2 + 1);
                }

                List chunkItemList;
                Persistable newItem;
                label120:
                for(ListChunkIterator chunkIterator = ListChunkIterator.of(itemList, chunkSize); chunkIterator.hasNext(); this.daoMapper.batchUpdateByIds(this.getEntityContext(), (List)chunkItemList.stream().map((x$0) -> EntityContextInstance.updatable((T)x$0)).collect(Collectors.toList()), updateColNameList)) {
                    chunkItemList = chunkIterator.nextChunk();
                    if (AuditTrailHelper.auditTrail() && AuditTrailHelper.auditTrailable(this.getTable(), AuditDatabaseOperation.UPDATE)) {
                        AuditTrailAspect.AuditTrailEntryData auditTrailEntryData = AuditTrailAspect.getAuditTrailEntryData();
                        if (auditTrailEntryData != null) {
                            String table = this.getTable();
                            Map<ID, List<CoreAuditTrailRecordLineBean>> map = new HashMap();
                            Iterator var14 = chunkItemList.iterator();

                            while(true) {
                                while(true) {
                                    boolean deleteFile;
                                    Persistable oldItem;
                                    ArrayList recordLineList;
                                    Iterator targetId;
                                    label105:
                                    do {
                                        if (!var14.hasNext()) {
                                            if (!map.isEmpty()) {
                                                Map<ID, String> targetMap = AuditTrailAspect.getAncientTargetMap(table, new ArrayList(map.keySet()));
                                                targetMap.forEach((k, v) -> {
                                                    AuditTrailAspect.addRecordLine(v, (List)map.get(k));
                                                });
                                            }
                                            continue label120;
                                        }

                                        newItem = (Persistable)var14.next();
                                        deleteFile = false;
                                        Persistable finalNewItem = newItem;
                                        oldItem = oldItemList.stream().filter((i) -> i.getId().equals(finalNewItem.getId())).findFirst().get();
                                        recordLineList = new ArrayList();
                                        targetId = updateColNameList.iterator();

                                        while(true) {
                                            String updateColName;
                                            Object oldValue;
                                            Object newValue;
                                            do {
                                                do {
                                                    if (!targetId.hasNext()) {
                                                        continue label105;
                                                    }

                                                    updateColName = (String)targetId.next();
                                                } while(!AuditTrailHelper.auditTrailableColumn(table, updateColName));

                                                oldValue = null;
                                                newValue = null;
                                                if (ReflectionUtils.findField(this.getType(), updateColName) == null) {
                                                    oldValue = oldItem.getExt$().get(updateColName);
                                                    newValue = newItem.getExt$().get(updateColName);
                                                } else {
                                                    oldValue = ReflectionUtils.getFieldValue(oldItem, updateColName);
                                                    newValue = ReflectionUtils.getFieldValue(newItem, updateColName);
                                                }
                                            } while(ObjectUtils.equals(oldValue, newValue));

                                            CoreAuditTrailRecordLineBean line = new CoreAuditTrailRecordLineBean();
                                            line.setTargetId(table + "$" + oldItem.getId());
                                            line.setTableName(table);
                                            if (AuditTrailType.DELETE.equals(auditTrailEntryData.getEntry().value()) && "TARGETID".equalsIgnoreCase(updateColName) && "T_CORE_FILE".equalsIgnoreCase(table)) {
                                                deleteFile = true;
                                                line.setRecordType(SqlType.DELETE.name());
                                                CoreFileBean file = (CoreFileBean)oldItem;
                                                line.setColumn("NAME");
                                                line.setOldValue(file.getName());
                                            } else {
                                                line.setRecordType(SqlType.UPDATE.name());
                                                line.setColumn(updateColName.toUpperCase());
                                                line.setOldValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(oldValue)));
                                                line.setNewValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(newValue)));
                                            }

                                            recordLineList.add(line);
                                        }
                                    } while(recordLineList.isEmpty());

                                    if (deleteFile) {
                                        String strTargetId;
                                        CoreFileBean file = (CoreFileBean)oldItem;
                                        int index = file.getTargetId().indexOf("$");
                                        if (index > 0 && index < file.getTargetId().length()) {
                                            strTargetId = AuditTrailAspect.getAncientTargetId(file.getTargetId().substring(0, index), file.getTargetId().substring(index + 1));
                                        } else {
                                            strTargetId = file.getTargetId();
                                        }

                                        AuditTrailAspect.addRecordLine(strTargetId, recordLineList);
                                    } else {
                                        map.put((ID) newItem.getId(), recordLineList);
                                    }
                                }
                            }
                        }
                    }
                }

                boolean cacheEvictOverrided = this.isCacheEvictOverrided();
                boolean auditable = AuditLogHelper.auditable(AuditDatabaseOperation.UPDATE);
                if (cacheEvictOverrided || auditable) {
                    List<T> newItemList = this.selectListByIds(idList);
                    this.auditUpdate(oldItemList, newItemList);

                    for (T t : newItemList) {
                        newItem =  t;
                        Persistable finalNewItem1 = newItem;
                        newItem = oldItemList.stream().filter((i) -> {
                            return ((Serializable) i.getId()).equals(finalNewItem1.getId());
                        }).findAny().orElse(null);
                        ((GenericDao) SinoAopContext.currentProxy()).cacheEvict(newItem, newItem);
                    }
                }

                GenericService<T, ID> service = SinoBeanContext.getServiceByTable(this.getTable());
                if (service != null) {
                    service.postUpdate(updateColNameList, idList);
                }
            }

        }
    }

    public List<T> updateIfChanged(List<T> rawOrProxyItemList) {
        if (rawOrProxyItemList != null && !rawOrProxyItemList.isEmpty()) {
            List<T> selectedItemList = this.selectListByIds((List)rawOrProxyItemList.stream().map((i) -> {
                return (Serializable)i.getId();
            }).collect(Collectors.toList()), new Order[0]);
            List<T> resultList = new ArrayList();
            Map<ID, List<CoreAuditTrailRecordLineBean>> map = new HashMap();
            boolean proxy = BeanPropertyListener.class.isAssignableFrom(rawOrProxyItemList.get(0).getClass());
            Iterator var6 = rawOrProxyItemList.iterator();

            while(true) {
                T oldItem;
                T proxyItem;
                while(true) {
                    if (!var6.hasNext()) {
                        if (!map.isEmpty()) {
                            Map<ID, String> targetMap = AuditTrailAspect.getAncientTargetMap(this.getTable(), new ArrayList(map.keySet()));
                            targetMap.forEach((k, v) -> {
                                AuditTrailAspect.addRecordLine(v, (List)map.get(k));
                            });
                        }

                        return resultList;
                    }

                    T rawOrProxyItem = (T) var6.next();
                    oldItem = selectedItemList.stream().filter((i) -> i.getId().equals(rawOrProxyItem.getId())).findFirst().orElseThrow(() -> new CheckedException("SINO.EXCEPTION.NO_VALUE_EXISTS"));
                    proxyItem = BeanUtils.getPropertyListenerProxy(ObjectUtils.clone(oldItem));
                    if (proxy) {
                        List<BeanPropertyEvent> eventList = ((BeanPropertyListener)rawOrProxyItem).getChangedPropertyEventList();
                        if (eventList.isEmpty()) {
                            continue;
                        }

                        Persistable finalProxyItem = proxyItem;
                        eventList.forEach((e) -> {
                            e.getPropertyDescriptor().passValue(rawOrProxyItem, finalProxyItem);
                        });
                        ((BeanPropertyListener)proxyItem).getChangedPropertyEventList().removeIf((e) -> !StringUtils.isEmpty(e.getName()) && eventList.stream().noneMatch((re) -> e.getName().equalsIgnoreCase(re.getName())));
                        break;
                    }

                    Persistable finalProxyItem1 = proxyItem;
                    this.getEntityContext().getColumnContextList().forEach((c) -> {
                        if (!c.getPropertyDescriptor().isExt$BeanProperty()) {
                            Object value = c.getPropertyDescriptor().getPropertyValue(rawOrProxyItem);
                            if (value != null) {
                                c.getPropertyDescriptor().passValue(rawOrProxyItem, finalProxyItem1);
                            }
                        }

                    });
                    break;
                }

                if (EntityHelper.isItemUpdated(proxyItem)) {
                    EntityContextInstance proxyContextInstance = EntityContextInstance.updatable(proxyItem, new String[0]);
                    if (AuditTrailHelper.auditTrail() && AuditTrailHelper.auditTrailable(this.getTable(), AuditDatabaseOperation.UPDATE)) {
                        AuditTrailEntryData auditTrailEntryData = AuditTrailAspect.getAuditTrailEntryData();
                        if (auditTrailEntryData != null) {
                            String table = this.getTable();
                            List<CoreAuditTrailRecordLineBean> recordLineList = new ArrayList();
                            Persistable finalOldItem = oldItem;
                            proxyContextInstance.getColumnContextInstanceList().stream().filter((c) -> {
                                return c.isActive() && AuditTrailHelper.auditTrailableColumn(table, c.getColumnContext().getColumnName());
                            }).forEach((c) -> {
                                CoreAuditTrailRecordLineBean line = new CoreAuditTrailRecordLineBean();
                                line.setRecordType(SqlType.UPDATE.name());
                                line.setTargetId(table + "$" + finalOldItem.getId());
                                line.setTableName(table);
                                line.setColumn(c.getColumnContext().getColumnName().toUpperCase());
                                line.setOldValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(c.getColumnContext().getValue(finalOldItem))));
                                line.setNewValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(c.getValue())));
                                recordLineList.add(line);
                            });
                            if (!recordLineList.isEmpty()) {
                                map.put((ID) oldItem.getId(), recordLineList);
                            }
                        }
                    }

                    this.daoMapper.updateById(proxyContextInstance);
                    T newItem = this.selectById((ID) oldItem.getId());
                    this.auditUpdate((List<T>)Arrays.asList(oldItem), Arrays.asList(newItem));
                    ((GenericDao)SinoAopContext.currentProxy()).cacheEvict(oldItem, newItem);
                    List<String> updatedColumnNameList = (List)proxyContextInstance.getColumnContextInstanceList().stream().filter((c) -> {
                        return c.isActive();
                    }).map((c) -> {
                        return c.getColumnContext().getColumnName();
                    }).collect(Collectors.toList());
                    GenericService<T, ID> service = SinoBeanContext.getServiceByTable(this.getTable());
                    if (service != null) {
                        service.postUpdate(updatedColumnNameList, Arrays.asList((ID)proxyItem.getId()));
                    }
                }
            }
        } else {
            return CollectionUtils.emptyList();
        }
    }

    public void updateCreatedBy(List<T> itemList) {
        if (!itemList.isEmpty()) {
            List<EntityColumnContext> createdByColumnList = this.getEntityContext().getColumnContextList().stream().filter((c) -> {
                return EntityHelper.isCreatedByColumn(c.getColumnName());
            }).collect(Collectors.toList());
            if (!createdByColumnList.isEmpty()) {
                List<T> oldItemList = this.selectListByIds((List)itemList.stream().map((i) -> {
                    return (Serializable)i.getId();
                }).collect(Collectors.toList()));
                List<EntityContextInstance> contextInstanceList = (List)itemList.stream().map((i) -> {
                    return EntityContextInstance.updatable(i);
                }).collect(Collectors.toList());
                this.daoMapper.batchUpdateByIds(this.getEntityContext(), contextInstanceList, (List)createdByColumnList.stream().map(EntityColumnContext::getColumnName).collect(Collectors.toList()));
                if (AuditTrailHelper.auditTrail() && AuditTrailHelper.auditTrailable(this.getTable(), AuditDatabaseOperation.UPDATE)) {
                    Map<ID, List<CoreAuditTrailRecordLineBean>> map = new HashMap();
                    AuditTrailEntryData auditTrailEntryData = AuditTrailAspect.getAuditTrailEntryData();
                    if (auditTrailEntryData != null) {
                        String table = this.getTable();

                        for (T t : itemList) {
                            T item = (T) t;
                            List<CoreAuditTrailRecordLineBean> recordLineList = new ArrayList();
                            T oldItem = (T) oldItemList.stream().filter((s) -> {
                                return s.getId().equals(item.getId());
                            }).findAny().get();

                            for (EntityColumnContext columnContext : createdByColumnList) {
                                CoreAuditTrailRecordLineBean line = new CoreAuditTrailRecordLineBean();
                                line.setRecordType(SqlType.UPDATE.name());
                                line.setTargetId(table + "$" + oldItem.getId());
                                line.setTableName(table);
                                line.setColumn(columnContext.getColumnName().toUpperCase());
                                line.setOldValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(columnContext.getValue(oldItem))));
                                line.setNewValue(RestFieldValueTextContainer.getText(line.getColumn(), ObjectUtils.toString(columnContext.getValue(item))));
                                recordLineList.add(line);
                            }

                            if (!recordLineList.isEmpty()) {
                                map.put(oldItem.getId(), recordLineList);
                            }
                        }
                    }

                    if (!map.isEmpty()) {
                        Map<ID, String> targetMap = AuditTrailAspect.getAncientTargetMap(this.getTable(), new ArrayList(map.keySet()));
                        targetMap.forEach((k, v) -> {
                            AuditTrailAspect.addRecordLine(v, (List)map.get(k));
                        });
                    }
                }

                this.auditUpdate(oldItemList, itemList);
                GenericService<T, ID> service = SinoBeanContext.getServiceByTable(this.getTable());
                if (service != null) {
                    service.postUpdate(createdByColumnList.stream().map(EntityColumnContext::getColumnName).collect(Collectors.toList()), (List)itemList.stream().map((i) -> (Serializable)i.getId()).collect(Collectors.toList()));
                }

            }
        }
    }

    public T selectByIdIfPresent(ID id) {
        List<Map<String, Object>> mapList = this.daoMapper.selectByIds(this.getEntityContext(), Arrays.asList(id), CollectionUtils.emptyList(), CollectionUtils.emptyList());
        return mapList.isEmpty() ? null : (T) PersistableHelper.mapToPersistable(mapList.get(0), this.getType());
    }

    public T selectOneIfPresent(T item, String... selectColNames) {
        List<Map<String, Object>> mapList = this.daoMapper.select(EntityContextInstance.instance(item, new String[0]), ArrayUtils.asList(selectColNames), CollectionUtils.emptyList());
        if (mapList.isEmpty()) {
            return null;
        } else if (mapList.size() > 1) {
            throw new NotUniqueResultJdbcException();
        } else {
            return (T) PersistableHelper.mapToPersistable((Map)mapList.get(0), this.getType());
        }
    }

    public <V> V selectColumnById(ID id, String colName, Class<V> colType) {
        List<Map<String, Object>> mapList = this.daoMapper.selectByIds(this.getEntityContext(), Arrays.asList(id), ArrayUtils.asList(new String[]{colName}), CollectionUtils.emptyList());
        if (mapList.isEmpty()) {
            return null;
        } else if (mapList.size() > 1) {
            throw new NotUniqueResultJdbcException();
        } else {
            return ConvertUtils.convert(CollectionUtils.getFirstValue((Map)mapList.get(0)), colType);
        }
    }

    public <V> List<V> selectColumnList(T item, List<String> searchColNames, String colName, Class<V> colType, Order... orders) {
        EntityContextInstance contextInstance = EntityContextInstance.instance(item, new String[0]);
        if (searchColNames != null && !searchColNames.isEmpty()) {
            contextInstance.getColumnContextInstanceList().forEach((c) -> {
                c.setActive(CollectionUtils.containsIgnoreCase(searchColNames, c.getColumnContext().getColumnName()));
            });
        }

        List<Order> orderList = orders == null ? CollectionUtils.emptyList() : Arrays.asList(orders);
        List<Map<String, Object>> mapList = this.daoMapper.select(contextInstance, ArrayUtils.asList(new String[]{colName}), orderList);
        return (List)mapList.parallelStream().map((m) -> {
            return ConvertUtils.convert(CollectionUtils.getFirstValue(m), colType);
        }).collect(Collectors.toList());
    }

    public List<T> selectAll(List<Order> orderList, String... selectColNames) {
        List<Map<String, Object>> mapList = this.daoMapper.select(EntityContextInstance.instance((Persistable)ClassUtils.newInstance(this.getType()), new String[0]), ArrayUtils.asList(selectColNames), orderList);
        return (List)mapList.parallelStream().map((m) -> {
            return (Persistable)PersistableHelper.mapToPersistable(m, this.getType());
        }).collect(Collectors.toList());
    }

    public List<T> selectListByIds(List<ID> idList, List<String> selectColNameList, Order... orders) {
        List<Map<String, Object>> mapList = this.selectMapListByIdList(idList, selectColNameList, orders);
        return (List)mapList.parallelStream().map((m) -> {
            return (Persistable)PersistableHelper.mapToPersistable(m, this.getType());
        }).collect(Collectors.toList());
    }

    public <V> List<V> selectColumnsByIds(List<ID> idList, String colName, Class<V> colType, Order... orders) {
        List<Map<String, Object>> mapList = this.selectMapListByIdList(idList, Collections.singletonList(colName), orders);
        return (List)mapList.parallelStream().map((m) -> {
            return CollectionUtils.getValueIgnorecase(m, colName);
        }).map((v) -> {
            return ConvertUtils.convert(v, colType);
        }).collect(Collectors.toList());
    }

    public List<T> selectList(T item, List<String> searchColNames, List<String> selectColNameList, Order... orders) {
        EntityContextInstance contextInstance = EntityContextInstance.instance(item);
        if (searchColNames != null && !searchColNames.isEmpty()) {
            contextInstance.getColumnContextInstanceList().forEach((c) -> {
                c.setActive(CollectionUtils.containsIgnoreCase(searchColNames, c.getColumnContext().getColumnName()));
            });
        }

        if (selectColNameList == null) {
            selectColNameList = CollectionUtils.emptyList();
        }

        List<Map<String, Object>> mapList = this.daoMapper.select(contextInstance, selectColNameList, ArrayUtils.asList(orders));
        return (List)mapList.parallelStream().map((m) -> {
            return (Persistable)PersistableHelper.mapToPersistable(m, this.getType());
        }).collect(Collectors.toList());
    }

    public List<T> selectList(List<T> itemList, List<String> searchColNameList, List<String> selectColNameList, Order... orders) {
        List<Map<String, Object>> mapList = this.selectMapList(itemList, searchColNameList, selectColNameList, orders);
        return (List)mapList.parallelStream().map((m) -> {
            return (Persistable)PersistableHelper.mapToPersistable(m, this.getType());
        }).collect(Collectors.toList());
    }

    public <V> List<V> selectList(List<T> itemList, List<String> searchColNameList, String colName, Class<V> colType, Order... orders) {
        List<String> selectColNameList = new ArrayList();
        selectColNameList.add(colName);
        List<Map<String, Object>> mapList = this.selectMapList(itemList, searchColNameList, selectColNameList, orders);
        return (List)mapList.parallelStream().map((m) -> {
            return ConvertUtils.convert(CollectionUtils.getValueIgnorecase(m, colName), colType);
        }).collect(Collectors.toList());
    }

    public T selectFirstIfPresent(T item, List<String> searchColNameList, List<String> selectColNameList, Order... orders) {
        EntityContextInstance contextInstance = EntityContextInstance.instance(item, new String[0]);
        if (searchColNameList != null && !searchColNameList.isEmpty()) {
            contextInstance.getColumnContextInstanceList().forEach((c) -> {
                c.setActive(CollectionUtils.containsIgnoreCase(searchColNameList, c.getColumnContext().getColumnName()));
            });
        }

        Map<String, Object> map = this.daoMapper.selectFirst(contextInstance, selectColNameList, ArrayUtils.asList(orders));
        return map == null ? null : (T) PersistableHelper.mapToPersistable(map, this.getType());
    }

    public List<T> selectUnionList(T item, List<String> searchColNameList, List<String> selectColNameList, Order... orders) {
        EntityContextInstance contextInstance = EntityContextInstance.instance(item);
        if (searchColNameList != null && !searchColNameList.isEmpty()) {
            contextInstance.getColumnContextInstanceList().forEach((c) -> {
                c.setActive(CollectionUtils.containsIgnoreCase(searchColNameList, c.getColumnContext().getColumnName()));
            });
        }

        List<Map<String, Object>> mapList = this.daoMapper.selectUnion(contextInstance, selectColNameList, ArrayUtils.asList(orders));
        return (List)mapList.parallelStream().map((m) -> {
            return (Persistable)PersistableHelper.mapToPersistable(m, this.getType());
        }).collect(Collectors.toList());
    }

    public int countBy(T item, String... searchColNames) {
        EntityContextInstance contextInstance = EntityContextInstance.instance(item, new String[0]);
        if (!ArrayUtils.isEmpty(searchColNames)) {
            contextInstance.getColumnContextInstanceList().forEach((c) -> {
                c.setActive(ArrayUtils.containsIgnoreCase(searchColNames, c.getColumnContext().getColumnName()));
            });
        }

        return this.daoMapper.count(contextInstance);
    }

    public int countBy(List<T> itemList, String... searchColNames) {
        List<EntityContextInstance> contextInstanceList = itemList.stream().map((i) -> {
            return EntityContextInstance.instance(i);
        }).collect(Collectors.toList());
        if (!ArrayUtils.isEmpty(searchColNames)) {
            contextInstanceList.forEach((i) -> {
                i.getColumnContextInstanceList().forEach((c) -> {
                    c.setActive(ArrayUtils.containsIgnoreCase(searchColNames, c.getColumnContext().getColumnName()));
                });
            });
        }

        return this.daoMapper.batchCount(this.getEntityContext(), contextInstanceList);
    }

    public Page<T> selectPaginationByMybatis(String mybatisStatementId, Pageable pageable) {
        Map<String, Object> parameter = (Map)JSONObject.parseObject(pageable.getFilterJson(), new TypeReference<Map<String, Object>>() {
        }, new Feature[0]);
        PageRowBounds rowBounds = new PageRowBounds(pageable);
        return MybatisPageHelper.get(rowBounds, () -> this.mybatisTemplate.selectList(mybatisStatementId, parameter));
    }

    private void preInsert(List<T> itemList) {
        boolean isOrderable = Orderable.class.isAssignableFrom(this.getType());
        List<BeanPropertyDescriptor> propertyDescriptorList = BeanPropertyHelper.getBeanPropertyDescriptorList(this.getType());
        List<BeanPropertyDescriptor> notNullPropertyDescriptorList = propertyDescriptorList.stream().filter((p) -> p.isAnnotationPresent(NotNull.class)).collect(Collectors.toList());
        BeanPropertyDescriptor createdByIdDescriptor = propertyDescriptorList.stream().filter(PersistableMetadataHelper::isCreatedByIdPropertyDescriptor).findAny().orElse(null);
        BeanPropertyDescriptor createdByNameDescriptor = propertyDescriptorList.stream().filter(PersistableMetadataHelper::isCreatedByNamePropertyDescriptor).findAny().orElse(null);
        BeanPropertyDescriptor createdByOrgIdDescriptor = propertyDescriptorList.stream().filter(PersistableMetadataHelper::isCreatedByOrgIdPropertyDescriptor).findAny().orElse(null);
        BeanPropertyDescriptor createdByOrgNameDescriptor = propertyDescriptorList.stream().filter(PersistableMetadataHelper::isCreatedByOrgNamePropertyDescriptor).findAny().orElse(null);
        BeanPropertyDescriptor createdTimeDescriptor = propertyDescriptorList.stream().filter(PersistableMetadataHelper::isCreatedTimePropertyDescriptor).findAny().orElse(null);
        String createdById = LocalContextHelper.getLoginUserId();
        String createdByName = LocalContextHelper.getLoginUserName();
        String createdByOrgId = LocalContextHelper.getLoginOrgId();
        String createdByOrgName = LocalContextHelper.getLoginOrgName();
        LocalDate nowDate = LocalDate.now();
        LocalDateTime nowTime = LocalDateTime.now();
        int i = 0;

        for(int j = itemList.size(); i < j; ++i) {
            T item = (T) itemList.get(i);
            if (createdByIdDescriptor != null && createdByIdDescriptor.getPropertyValue(item) == null) {
                createdByIdDescriptor.setPropertyValue(item, createdById);
            }

            if (createdByNameDescriptor != null && createdByNameDescriptor.getPropertyValue(item) == null) {
                createdByNameDescriptor.setPropertyValue(item, createdByName);
            }

            if (createdByOrgIdDescriptor != null && createdByOrgIdDescriptor.getPropertyValue(item) == null) {
                createdByOrgIdDescriptor.setPropertyValue(item, createdByOrgId);
            }

            if (createdByOrgNameDescriptor != null && createdByOrgNameDescriptor.getPropertyValue(item) == null) {
                createdByOrgNameDescriptor.setPropertyValue(item, createdByOrgName);
            }

            if (createdTimeDescriptor != null && createdTimeDescriptor.getPropertyValue(item) == null) {
                createdTimeDescriptor.setPropertyValue(item, nowTime);
            }

            if (!notNullPropertyDescriptorList.isEmpty()) {

                for (BeanPropertyDescriptor notNullPropertyDescriptor : notNullPropertyDescriptorList) {
                    if (notNullPropertyDescriptor.getPropertyValue(item) == null) {
                        NotNull notNull = notNullPropertyDescriptor.getProperty().getAnnotation(NotNull.class);
                        notNullPropertyDescriptor.setPropertyValue(item, notNull.defaultValue());
                    }
                }
            }

            if (isOrderable) {
                Orderable<T> orderable = (Orderable)item;
                if (orderable.getOrderNo() == null) {
                    orderable.setOrderNo(ApplicationContextHelper.getNextOrderNo());
                }
            }
        }

    }

    private List<Map<String, Object>> selectMapListByIdList(List<ID> idList, List<String> selectColNames, Order... orders) {
        if (idList != null && !idList.isEmpty()) {
            ListChunkIterator<ID> chunkIterator = ListChunkIterator.of(idList, this.dialect.getInClauseIdMaxQty());
            boolean loop = idList.size() > this.dialect.getMaxParamQty();
            List<String> cloneSelectColNames = selectColNames == null ? new ArrayList() : new ArrayList(selectColNames);
            if (loop && ArrayUtils.hasElement(orders)) {
                int var8 = orders.length;

                Arrays.stream(orders).filter(order -> cloneSelectColNames.stream().noneMatch((n) -> {
                    return n.equalsIgnoreCase(order.getColumn());
                })).forEach(order -> cloneSelectColNames.add(order.getColumn().toUpperCase()));
            }

            ArrayList mapList = new ArrayList();

            while(chunkIterator.hasNext()) {
                if (loop) {
                    mapList.addAll(this.daoMapper.selectByIds(this.getEntityContext(), CollectionUtils.convert(chunkIterator.nextChunk(), Object.class), cloneSelectColNames, CollectionUtils.emptyList()));
                } else {
                    mapList.addAll(this.daoMapper.selectByIds(this.getEntityContext(), CollectionUtils.convert(chunkIterator.nextChunk(), Object.class), cloneSelectColNames, ArrayUtils.asList(orders)));
                }
            }

            if (!mapList.isEmpty() && loop && ArrayUtils.hasElement(orders)) {
                this.sort(mapList, orders);
            }

            return mapList;
        } else {
            return CollectionUtils.emptyList();
        }
    }

    private List<Map<String, Object>> selectMapList(List<T> itemList, List<String> searchColNames, List<String> selectColNames, Order... orders) {
        if (itemList != null && !itemList.isEmpty()) {
            List<EntityContextInstance> contextInstanceList = (List)itemList.stream().map((i) -> EntityContextInstance.instance(i)).collect(Collectors.toList());
            if (searchColNames != null && !searchColNames.isEmpty()) {
                contextInstanceList.forEach((i) -> {
                    i.getColumnContextInstanceList().forEach((c) -> {
                        c.setActive(CollectionUtils.containsIgnoreCase(searchColNames, c.getColumnContext().getColumnName()));
                    });
                });
            }

            long activeQty = contextInstanceList.stream().mapToLong((i) -> i.getColumnContextInstanceList().stream().filter((c) -> c.isActive()).count()).max().getAsLong();
            boolean loop = activeQty * (long)contextInstanceList.size() > (long)this.dialect.getMaxParamQty();
            List<String> cloneSelectColNames = selectColNames == null ? new ArrayList() : new ArrayList(selectColNames);
            if (loop && ArrayUtils.hasElement(orders) && !cloneSelectColNames.isEmpty()) {
                int var11 = orders.length;

                Arrays.stream(orders).filter(order -> !CollectionUtils.containsIgnoreCase(cloneSelectColNames, order.getColumn())).forEach(order -> cloneSelectColNames.add(order.getColumn().toUpperCase()));
            }

            ListChunkIterator<EntityContextInstance> chunkIterator = ListChunkIterator.of(contextInstanceList, this.dialect.getMaxParamQty());
            ArrayList mapList = new ArrayList();

            while(chunkIterator.hasNext()) {
                if (loop) {
                    mapList.addAll(this.daoMapper.batchSelect(this.getEntityContext(), chunkIterator.nextChunk(), cloneSelectColNames, CollectionUtils.emptyList()));
                } else {
                    mapList.addAll(this.daoMapper.batchSelect(this.getEntityContext(), chunkIterator.nextChunk(), cloneSelectColNames, ArrayUtils.asList(orders)));
                }
            }

            if (!mapList.isEmpty() && loop && ArrayUtils.hasElement(orders)) {
                this.sort(mapList, orders);
            }

            return mapList;
        } else {
            return CollectionUtils.emptyList();
        }
    }

    private void auditUpdate(List<T> selectedOldItemList, List<T> selectedNewItemList) {
        if (AuditLogHelper.auditable(AuditDatabaseOperation.UPDATE)) {

            for (T t : selectedOldItemList) {
                T selectedNewItem = selectedNewItemList.stream().filter((i) -> t.getId().equals(i.getId())).findAny().orElse(null);
                AuditLogHelper.insertAuditLog(this.getTable(), t, selectedNewItem);
            }
        }

    }

    private void sort(List<Map<String, Object>> mapList, Order... orders) {
        mapList.sort((m1, m2) -> {
            int var4 = orders.length;

            int i = 0, ordersLength = orders.length;
            while (i < ordersLength) {
                Order order = orders[i];
                Comparable v1 = (Comparable) CollectionUtils.getValueIgnorecase(m1, order.getColumn());
                Comparable v2 = (Comparable) CollectionUtils.getValueIgnorecase(m2, order.getColumn());
                if (v1 == null) {
                    if (v2 != null) {
                        if (StringUtils.endsWith(order.getDirection(), "ASC")) {
                            return -1;
                        }

                        return 1;
                    }
                } else {
                    if (v2 == null) {
                        if (StringUtils.endsWith(order.getDirection(), "ASC")) {
                            return 1;
                        }

                        return -1;
                    }

                    int cv = v1.compareTo(v2);
                    if (cv != 0) {
                        if (StringUtils.endsWith(order.getDirection(), "ASC")) {
                            return cv;
                        }

                        return -1 * cv;
                    }
                }
                i++;
            }

            return 0;
        });
    }

    private boolean isCacheEvictOverrided() {
        GenericDao<T, ID> proxy = SinoAopContext.currentProxy();
        Class<?> clazz = ClassUtils.getRawType(proxy.getClass());
        Method method = ReflectionUtils.findMethodByName(clazz, "cacheEvict");
        return !GenericDao.class.equals(method.getDeclaringClass());
    }
}


