/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.mybatis;


import net.sinodawn.framework.exception.UnexpectedException;
import net.sinodawn.framework.exception.database.JdbcException;
import net.sinodawn.framework.support.base.mapper.GenericMapper;
import net.sinodawn.framework.utils.ReflectionUtils;
import net.sinodawn.framework.utils.StringUtils;
import org.apache.ibatis.builder.xml.XMLMapperBuilder;
import org.apache.ibatis.mapping.MappedStatement;
import org.apache.ibatis.session.Configuration;
import org.apache.ibatis.session.SqlSessionFactory;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Component;

import javax.annotation.PostConstruct;
import java.io.*;
import java.sql.Connection;
import java.sql.SQLException;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

@SuppressWarnings({"unused", "unchecked", "rawtypes"})
@Component
public class MybatisHelper {
    private static final Logger logger = LogManager.getLogger(MybatisHelper.class);
    private static final ThreadLocal<String> KEY_PREFIX = new ThreadLocal<>();

    @Autowired
    private SqlSessionFactory sqlSessionFactory;
    private static SqlSessionFactory staticSqlSessionFactory;

    @PostConstruct
    public void init() {
        staticSqlSessionFactory = sqlSessionFactory;
    }

    public MybatisHelper() {
    }

    public static SqlSessionFactory getSqlSessionFactory() {
        return staticSqlSessionFactory;
    };

    public static Connection getConnection() {
        try {
            return getSqlSessionFactory().getConfiguration().getEnvironment().getDataSource().getConnection();
        } catch (SQLException var1) {
            throw new JdbcException(var1);
        }
    }

    public static void updateRuntimeConfiguration() {
        Configuration configuration = getSqlSessionFactory().getConfiguration();
        String[] var1 = new String[]{"mappedStatements", "caches", "resultMaps", "parameterMaps", "keyGenerators", "sqlFragments"};
        int var2 = var1.length;

        for (String field : var1) {
            Map oldMap = (Map) ReflectionUtils.getFieldValue(configuration, field);
            Map newMap = new MybatisStrictMap(field + " UpdatableCollection", oldMap);
            ReflectionUtils.setFieldValue(configuration, field, newMap);
        }

    }

//    public static void updateRuntimeMapper() {
//        CoreAdminMybatisStatementService statementService = (CoreAdminMybatisStatementService) ApplicationContextHelper.getBean(CoreAdminMybatisStatementService.class);
//        List<CoreAdminMybatisStatementBean> statementList = statementService.selectActiveList(null, null);
//        if (!statementList.isEmpty()) {
//            Map<String, List<CoreAdminMybatisStatementBean>> mapperIdMap = (Map)statementList.stream().collect(Collectors.groupingBy(CoreAdminMybatisStatementBean::getMapperId));
//            mapperIdMap.forEach((mapperId, v) -> {
//                String mapperContent = getMapperContent(mapperId);
//                if (!StringUtils.isBlank(mapperContent)) {
//                    Map<String, List<CoreAdminMybatisStatementBean>> menuIdMap = (Map)v.stream().collect(Collectors.groupingBy(CoreAdminMybatisStatementBean::getMenuId));
//                    menuIdMap.forEach((menuId, l) -> {
//                        String copyMapperContent = mapperContent;
//
//                        CoreAdminMybatisStatementBean statement;
//                        for(Iterator var6 = statementList.iterator(); var6.hasNext(); copyMapperContent = copyMapperContent.replaceAll(getMapperSelectStatementRegex(statement.getStatementId()), statement.getStatementXml())) {
//                            statement = (CoreAdminMybatisStatementBean)var6.next();
//                        }
//
//                        updateRuntimeMapper(copyMapperContent, menuId, mapperId);
//                    });
//                } else {
//                    logger.warn("No mapper content for mapperId :{}", mapperId);
//                }
//
//            });
//        }
//
//    }

//    public static void updateRuntimeMapper(String menuId, String mapperId) {
//        String mapperContent = getMapperContent(mapperId);
//        if (StringUtils.isBlank(mapperContent)) {
//            logger.warn("No mapper content for mapperId :{}", mapperId);
//        } else {
//            CoreAdminMybatisStatementService statementService = (CoreAdminMybatisStatementService)ApplicationContextHelper.getBean(CoreAdminMybatisStatementService.class);
//            List<CoreAdminMybatisStatementBean> statementList = statementService.selectActiveList(menuId, mapperId);
//
//            CoreAdminMybatisStatementBean statement;
//            for(Iterator var5 = statementList.iterator(); var5.hasNext(); mapperContent = mapperContent.replaceAll(getMapperSelectStatementRegex(statement.getStatementId()), statement.getStatementXml())) {
//                statement = (CoreAdminMybatisStatementBean)var5.next();
//            }
//
//            updateRuntimeMapper(mapperContent, menuId, mapperId);
//        }
//    }

    public static void setDraftMapper(String menuId, String mapperId) {
//        String mapperContent = getMapperContent(mapperId);
//        if (StringUtils.isBlank(mapperContent)) {
//            logger.warn("No mapper content for mapperId :{}", mapperId);
//        } else {
//            CoreAdminMybatisStatementService statementService = (CoreAdminMybatisStatementService)ApplicationContextHelper.getBean(CoreAdminMybatisStatementService.class);
//            List<CoreAdminMybatisStatementBean> statementList = statementService.selectActiveList(menuId, mapperId);
//            CoreAdminMybatisDraftService draftService = (CoreAdminMybatisDraftService)ApplicationContextHelper.getBean(CoreAdminMybatisDraftService.class);
//            CoreAdminMybatisDraftBean draftFilter = new CoreAdminMybatisDraftBean();
//            draftFilter.setMenuId(menuId);
//            draftFilter.setMapperId(mapperId);
//            List<CoreAdminMybatisDraftBean> draftList = draftService.selectList(draftFilter, new Order[0]);
//
//            Iterator var8;
//            CoreAdminMybatisDraftBean draft;
//            for(var8 = draftList.iterator(); var8.hasNext(); mapperContent = mapperContent.replaceAll(getMapperSelectStatementRegex(draft.getStatementId()), draft.getStatementXml())) {
//                draft = (CoreAdminMybatisDraftBean)var8.next();
//            }
//
//            var8 = statementList.iterator();
//
//            while(var8.hasNext()) {
//                CoreAdminMybatisStatementBean statement = (CoreAdminMybatisStatementBean)var8.next();
//                if (draftList.stream().noneMatch((d) -> {
//                    return d.getMenuId().equals(statement.getMenuId()) && d.getMapperId().equals(statement.getMapperId()) && d.getStatementId().equals(statement.getStatementId());
//                })) {
//                    mapperContent = mapperContent.replaceAll(getMapperSelectStatementRegex(statement.getStatementId()), statement.getStatementXml());
//                }
//            }
//
//            updateRuntimeMapper(mapperContent, "draft$" + menuId, mapperId);
//        }
    }

    public static void removeDraftMapper(String menuId, String mapperId) {
        removeRuntimeMapper("draft$" + menuId + "$" + mapperId);
    }

    public static void removeRuntimeMapper(String resource) {
        Configuration configuration = getSqlSessionFactory().getConfiguration();
        Set<String> loadedResourcesSet = (Set)ReflectionUtils.getFieldValue(configuration, "loadedResources");
        if (loadedResourcesSet != null) {
            loadedResourcesSet.remove(resource);
        }
        Map<String, MappedStatement> mappedStatements = (Map)ReflectionUtils.getFieldValue(configuration, "mappedStatements");
        if (mappedStatements != null) {
            mappedStatements.remove(resource);
        }
    }

    public static void updateRuntimeMapper(String mapperXmlContent, String prefix, String resource) {
        try {
            if (!StringUtils.isBlank(prefix)) {
                resource = prefix + "$" + resource;
                setKeyPrefix(prefix);
            }

            Configuration configuration = getSqlSessionFactory().getConfiguration();
            Set<String> loadedResourcesSet = (Set)ReflectionUtils.getFieldValue(configuration, "loadedResources");
            loadedResourcesSet.remove(resource);
            XMLMapperBuilder builder = new XMLMapperBuilder(new ByteArrayInputStream(mapperXmlContent.getBytes()), configuration, resource, configuration.getSqlFragments());
            builder.parse();
        } finally {
            removeKeyPrefix();
        }

    }

    public static String getStatementContent(String mappedStatementId) {
        Configuration configuration = getSqlSessionFactory().getConfiguration();
        Map<String, MappedStatement> mappedStatements = (Map)ReflectionUtils.getFieldValue(configuration, "mappedStatements");
        MappedStatement ms = (MappedStatement)mappedStatements.get(mappedStatementId);
        String resource = ms.getResource();
        int index = Math.max(resource.lastIndexOf("com/sinoworld"), resource.lastIndexOf("com\\sinoworld"));
        String name = "/" + StringUtils.replace(resource.substring(index), "\\", "/");
        if (!StringUtils.endsWith(name, "xml")) {
            for(int length = name.length(); length > 0 && !StringUtils.endsWith(name, "xml"); length = name.length()) {
                name = name.substring(0, length - 1);
            }
        }

        try {
            InputStream is = MybatisHelper.class.getResourceAsStream(name);
            Throwable var8 = null;

            String var11;
            try {
                String mapperContent = (String)(new BufferedReader(new InputStreamReader(is))).lines().collect(Collectors.joining(System.lineSeparator()));
                String simpleId = mappedStatementId.substring(mappedStatementId.lastIndexOf(".") + 1);
                var11 = mapperContent.replaceAll(getMapperSelectStatementRegex(simpleId), "");
            } catch (Throwable var21) {
                var8 = var21;
                throw var21;
            } finally {
                if (is != null) {
                    if (var8 != null) {
                        try {
                            is.close();
                        } catch (Throwable var20) {
                            var8.addSuppressed(var20);
                        }
                    } else {
                        is.close();
                    }
                }

            }

            return var11;
        } catch (IOException var23) {
            throw new UnexpectedException(var23);
        }
    }

    public static String getMapperContent(String mapperId) {
        String mappedStatementId = mapperId + ".selectByCondition";
        Configuration configuration = getSqlSessionFactory().getConfiguration();
        Map<String, MappedStatement> mappedStatements = (Map)ReflectionUtils.getFieldValue(configuration, "mappedStatements");
        MappedStatement ms = (MappedStatement)mappedStatements.get(mappedStatementId);
        if (ms == null) {
            return null;
        } else {
            String resource = ms.getResource();
            int index = Math.max(resource.lastIndexOf("com/sinoworld"), resource.lastIndexOf("com\\sinoworld"));
            String name = "/" + StringUtils.replace(resource.substring(index), "\\", "/");
            if (!StringUtils.endsWith(name, "xml")) {
                for(int length = name.length(); length > 0 && !StringUtils.endsWith(name, "xml"); length = name.length()) {
                    name = name.substring(0, length - 1);
                }
            }

            try {
                InputStream is = MybatisHelper.class.getResourceAsStream(name);
                Throwable var9 = null;

                String var10;
                try {
                    var10 = (String)(new BufferedReader(new InputStreamReader(is))).lines().collect(Collectors.joining(System.lineSeparator()));
                } catch (Throwable var20) {
                    var9 = var20;
                    throw var20;
                } finally {
                    if (is != null) {
                        if (var9 != null) {
                            try {
                                is.close();
                            } catch (Throwable var19) {
                                var9.addSuppressed(var19);
                            }
                        } else {
                            is.close();
                        }
                    }

                }

                return var10;
            } catch (IOException var22) {
                throw new UnexpectedException(var22);
            }
        }
    }

    public static String getMapperContent(Class<? extends GenericMapper<?>> mapperClazz) {
        return getMapperContent(mapperClazz.getName());
    }

    public static void setKeyPrefix(String prefix) {
        KEY_PREFIX.set(prefix);
    }

    public static void removeKeyPrefix() {
        KEY_PREFIX.remove();
    }

    public static String getKeyPrefix() {
        return (String)KEY_PREFIX.get();
    }

    private static final String getMapperSelectStatementRegex(String statementId) {
        return "<(?i)select\\s*id=\"" + statementId + "[\\S\\s]+?<\\s*/(?i)select\\s*>";
    }
}
