/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.log.aspect;

import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.context.LocalContextHelper;
import net.sinodawn.framework.exception.ConcurrentException;
import net.sinodawn.framework.log.LogQueue;
import net.sinodawn.framework.log.annotation.Log;
import net.sinodawn.framework.log.annotation.LogModule;
import net.sinodawn.framework.log.annotation.LogType;
import net.sinodawn.framework.utils.*;
import net.sinodawn.module.sys.log.bean.CoreLogBean;
import org.apache.logging.log4j.Level;
import org.apache.logging.log4j.LogManager;
import org.apache.logging.log4j.Logger;
import org.aspectj.lang.ProceedingJoinPoint;
import org.aspectj.lang.Signature;
import org.aspectj.lang.annotation.Around;
import org.aspectj.lang.annotation.Aspect;
import org.aspectj.lang.reflect.MethodSignature;
import org.springframework.stereotype.Component;

import javax.servlet.http.HttpServletRequest;
import java.lang.reflect.Modifier;
import java.time.LocalDateTime;

@Aspect
@Component
public class LoggingAspect {
   public static final String NOT_LOGIN_USERID = "NIL";
   public static final String NOT_LOGIN_USERNAME = "NIL";
   public static final String UNKNOWN_MODULE_NAME = "UNKNOWN";
   public static final String UNKNOWN_LOG_NAME = "UNKNOWN";
   private static final Logger logger = LogManager.getLogger(LoggingAspect.class);
   private static boolean logging = true;

   public static void disable() {
      logging = false;
   }

   public static void enable() {
      logging = true;
   }

   @Around("within(net.sinodawn..resource..*)||within(net.sinodawn..controller..*)")
   public Object handleResourceException(ProceedingJoinPoint pjp) throws Throwable {
      if (!logging) {
         return pjp.proceed();
      } else {
         CoreLogBean coreLog = null;
         Class<?> clazz = ClassUtils.getRawType(pjp.getTarget().getClass());
         LogModule logModule = ReflectionUtils.getAnnotation(clazz, LogModule.class);
         MethodSignature methodSignature = (MethodSignature)pjp.getSignature();
         Log log = (Log) ReflectionUtils.getAnnotation(methodSignature.getMethod(), Log.class);
         if (log != null) {
            Level level = Level.getLevel(log.level().name());
            coreLog = this.getCoreLog(pjp, logModule == null ? "UNKNOWN" : logModule.value(), log.value(), log.type().name());
            if (logger.isEnabled(level)) {
               logger.log(level, coreLog.toLogString());
            }
         }

         Object var14;
         try {
            var14 = pjp.proceed();
         } catch (Throwable var12) {
            if (ConcurrentException.class.isAssignableFrom(var12.getClass())) {
               coreLog = null;
               throw var12;
            }

            if (coreLog == null) {
               coreLog = this.getCoreLog(pjp, logModule == null ? "UNKNOWN" : logModule.value(), log == null ? "UNKNOWN" : log.value(), LogType.EXCEPTION.name());
            } else {
               coreLog.setType(LogType.EXCEPTION.name());
            }

            StringBuilder sb = new StringBuilder();
            if (!StringUtils.isEmpty(coreLog.getContent())) {
               sb.append(coreLog.getContent()).append("\n");
            }

            sb.append(this.getThrowableContent(var12));
            coreLog.setContent(sb.toString());
//             TODO 鉴权模块
//            if (!AuthenticationException.class.isAssignableFrom(var12.getClass()) && !CheckedException.class.isAssignableFrom(var12.getClass())) {
//               logger.error(coreLog.toLogString());
//            } else {
               logger.warn(var12.getMessage());
//            }

            throw var12;
         } finally {
            if (coreLog != null && (ApplicationContextHelper.isProfileActivated("logging") || LogType.EXCEPTION.name().equals(coreLog.getType()))) {
               coreLog.setEndTime(LocalDateTime.now());
               LogQueue.INSTANCE.offer(coreLog);
            }

         }

         return var14;
      }
   }

   private CoreLogBean getCoreLog(ProceedingJoinPoint pjp, String module, String name, String type) {
      CoreLogBean coreLog = new CoreLogBean();
      coreLog.setModule(module);
      coreLog.setName(name);
      coreLog.setType(type);
      coreLog.setSessionId(ServletUtils.getCurrentRawSessionId());
      if (LocalContextHelper.isUserLogin()) {
         coreLog.setUserId(LocalContextHelper.getLoginUserId());
         coreLog.setUserName(LocalContextHelper.getLoginUser().getUserName());
         coreLog.setLoginIp(ServletUtils.getRemoteIp(ServletUtils.getCurrentRequest()));
      } else {
         coreLog.setUserId("NIL");
         coreLog.setUserName("NIL");
      }

      coreLog.setStartTime(LocalDateTime.now());
      coreLog.setRequestUri(this.getRequestURI());
      coreLog.setSignature(this.getSimpleSignature(pjp.getTarget(), pjp.getSignature()));
      if (pjp.getArgs().length > 0) {
         coreLog.setContent(this.getContent(pjp.getSignature(), pjp.getArgs()));
      }

      return coreLog;
   }

   private String getContent(Signature signature, Object[] args) {
      MethodSignature methodSignature = (MethodSignature)signature;
      StringBuilder sb = new StringBuilder();
      sb.append("[params]");
      if (methodSignature.getParameterNames() != null) {
         int i = 0;

         for(int j = methodSignature.getParameterNames().length; i < j; ++i) {
            sb.append("\n");
            String parameterName = methodSignature.getParameterNames()[i];
            Object value = args[i];
            sb.append(parameterName).append(" : ").append(ObjectUtils.toShortString(value));
         }
      }

      return sb.toString();
   }

   private String getThrowableContent(Throwable throwable) {
      return "[EXCEPTION]\n" + ObjectUtils.toShortString(throwable);
   }

   private String getRequestURI() {
      HttpServletRequest request = ServletUtils.getCurrentRequest();
      return request != null ? ServletUtils.getRequestUri(request) : null;
   }

   private String getSimpleSignature(Object target, Signature signature) {
      MethodSignature methodSignature = (MethodSignature)signature;
      StringBuilder builder = new StringBuilder();
      builder.append(Modifier.toString(methodSignature.getModifiers())).append(" ").append(methodSignature.getReturnType().getSimpleName()).append(" ").append(target.getClass().getSimpleName()).append(".").append(methodSignature.getName()).append("(");
      if (methodSignature.getParameterNames() != null) {
         int i = 0;

         for(int j = methodSignature.getParameterNames().length; i < j; ++i) {
            if (i > 0) {
               builder.append(", ");
            }

            String parameterName = methodSignature.getParameterNames()[i];
            Class<?> parameterType = methodSignature.getParameterTypes()[i];
            builder.append(parameterType.getSimpleName()).append(" ").append(parameterName);
         }

         builder.append(")");
      }

      return builder.toString();
   }
}
