/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.io.excel.utils;

import net.sinodawn.framework.utils.StringUtils;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;

import java.util.Optional;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public abstract class ExcelUtils {
   public static final String getWellFormedSheetName(String sheetName) {
      if (!StringUtils.isEmpty(sheetName)) {
         if (StringUtils.contains(sheetName, "/")) {
            sheetName = StringUtils.replace(sheetName, "/", "-");
         }

         if (StringUtils.contains(sheetName, "\\")) {
            sheetName = StringUtils.replace(sheetName, "\\", "-");
         }

         if (StringUtils.contains(sheetName, "'")) {
            sheetName = StringUtils.replace(sheetName, "'", "‘");
         }

         if (StringUtils.contains(sheetName, "?")) {
            sheetName = StringUtils.replace(sheetName, "?", "？");
         }

         if (StringUtils.contains(sheetName, "*")) {
            sheetName = StringUtils.replace(sheetName, "*", "×");
         }

         return sheetName;
      } else {
         return "";
      }
   }

   public static final CellStyle getTopicCellStyle(Workbook wb) {
      CellStyle cs = wb.createCellStyle();
      Font font = wb.createFont();
      font.setFontHeightInPoints((short)18);
      font.setBold(true);
      cs.setBorderBottom(BorderStyle.DOUBLE);
      cs.setAlignment(HorizontalAlignment.CENTER);
      cs.setVerticalAlignment(VerticalAlignment.CENTER);
      cs.setFont(font);
      return cs;
   }

   public static final CellStyle getTitleCellStyle(Workbook wb) {
      CellStyle cs = wb.createCellStyle();
      Font font = wb.createFont();
      font.setFontHeightInPoints((short)12);
      font.setBold(true);
      cs.setBorderBottom(BorderStyle.MEDIUM);
      cs.setBorderLeft(BorderStyle.THIN);
      cs.setBorderRight(BorderStyle.THIN);
      cs.setAlignment(HorizontalAlignment.CENTER);
      cs.setVerticalAlignment(VerticalAlignment.CENTER);
      cs.setFont(font);
      return cs;
   }

   public static final CellStyle getFooterCenterCellStyle(Workbook wb) {
      CellStyle cs = wb.createCellStyle();
      Font font = wb.createFont();
      font.setFontHeightInPoints((short)12);
      font.setBold(true);
      cs.setBorderBottom(BorderStyle.MEDIUM);
      cs.setBorderLeft(BorderStyle.THIN);
      cs.setBorderRight(BorderStyle.THIN);
      cs.setBorderTop(BorderStyle.MEDIUM);
      cs.setAlignment(HorizontalAlignment.CENTER);
      cs.setVerticalAlignment(VerticalAlignment.CENTER);
      cs.setFont(font);
      return cs;
   }

   public static final CellStyle getFooterRightCellStyle(Workbook wb) {
      CellStyle cs = wb.createCellStyle();
      Font font = wb.createFont();
      font.setFontHeightInPoints((short)12);
      font.setBold(true);
      cs.setBorderBottom(BorderStyle.MEDIUM);
      cs.setBorderLeft(BorderStyle.THIN);
      cs.setBorderRight(BorderStyle.THIN);
      cs.setBorderTop(BorderStyle.MEDIUM);
      cs.setAlignment(HorizontalAlignment.RIGHT);
      cs.setVerticalAlignment(VerticalAlignment.CENTER);
      cs.setFont(font);
      return cs;
   }

   public static final CellStyle getCenterCellStyle(Workbook wb) {
      CellStyle cs = wb.createCellStyle();
      Font font = wb.createFont();
      font.setFontHeightInPoints((short)12);
      cs.setBorderBottom(BorderStyle.MEDIUM);
      cs.setBorderLeft(BorderStyle.THIN);
      cs.setBorderRight(BorderStyle.THIN);
      cs.setAlignment(HorizontalAlignment.CENTER);
      cs.setVerticalAlignment(VerticalAlignment.CENTER);
      cs.setFont(font);
      return cs;
   }

   public static final CellStyle getLeftCellStyle(Workbook wb) {
      CellStyle cs = wb.createCellStyle();
      Font font = wb.createFont();
      font.setFontHeightInPoints((short)12);
      cs.setBorderBottom(BorderStyle.THIN);
      cs.setBorderLeft(BorderStyle.THIN);
      cs.setBorderRight(BorderStyle.THIN);
      cs.setAlignment(HorizontalAlignment.LEFT);
      cs.setVerticalAlignment(VerticalAlignment.CENTER);
      cs.setFont(font);
      return cs;
   }

   public static final CellStyle getRightCellStyle(Workbook wb) {
      CellStyle cs = wb.createCellStyle();
      Font font = wb.createFont();
      font.setFontHeightInPoints((short)12);
      cs.setBorderBottom(BorderStyle.THIN);
      cs.setBorderLeft(BorderStyle.THIN);
      cs.setBorderRight(BorderStyle.THIN);
      cs.setAlignment(HorizontalAlignment.RIGHT);
      cs.setVerticalAlignment(VerticalAlignment.CENTER);
      cs.setFont(font);
      return cs;
   }

   public static final int getCellWidth(String content) {
      return ((String)Optional.ofNullable(content).orElse("")).getBytes().length * 256;
   }

   public static final void copyRow(final Sheet sheet, final int sourceRowNum, final int destRowNum) {
      copyRow(sheet, sourceRowNum, destRowNum, false);
   }

   public static final void copyRow(final Sheet sheet, final int sourceRowNum, final int destRowNum, boolean copyValue) {
      int lastRowNum = sheet.getLastRowNum();
      if (destRowNum <= lastRowNum) {
         sheet.shiftRows(destRowNum, sheet.getLastRowNum(), 1);
      }

      Row sourceRow = sheet.getRow(sourceRowNum);
      Row destRow = sheet.createRow(destRowNum);
      destRow.setHeight(sourceRow.getHeight());
      int lastCellNum = sourceRow.getLastCellNum();

      for(int i = 0; i < lastCellNum; ++i) {
         Cell sourceCell = sourceRow.getCell(i);
         Cell destCell = destRow.getCell(i);
         if (sourceCell != null) {
            if (destCell == null) {
               destCell = destRow.createCell(i);
            }

            destCell.setCellStyle(sourceCell.getCellStyle());
            if (sourceCell.getCellType() == CellType.FORMULA && !StringUtils.isEmpty(sourceCell.getCellFormula())) {
               destCell.setCellFormula(getResolvedFormula(sourceCell.getCellFormula(), sourceRowNum, destRowNum));
            }

            if (copyValue) {
               copyCellValue(sourceCell, destCell);
            }

            boolean merge = false;
            int j = 0;

            int k;
            for(k = sheet.getNumMergedRegions(); j < k; ++j) {
               if (sheet.getMergedRegion(j).isInRange(destRowNum, i)) {
                  merge = true;
               }
            }

            if (!merge) {
               j = 0;

               for(k = sheet.getNumMergedRegions(); j < k; ++j) {
                  CellRangeAddress rangeAddress = sheet.getMergedRegion(j);
                  if (rangeAddress.isInRange(sourceRowNum, i)) {
                     sheet.addMergedRegion(new CellRangeAddress(destRowNum, destRowNum, rangeAddress.getFirstColumn(), rangeAddress.getLastColumn()));
                     break;
                  }
               }
            }
         }
      }

   }

   private static void copyCellValue(Cell sourceCell, Cell destCell) {
      switch(sourceCell.getCellType()) {
      case STRING:
         destCell.setCellValue(sourceCell.getStringCellValue());
         break;
      case NUMERIC:
         destCell.setCellValue(sourceCell.getNumericCellValue());
         break;
      case BOOLEAN:
         destCell.setCellValue(sourceCell.getBooleanCellValue());
         break;
      case ERROR:
         destCell.setCellValue((double)sourceCell.getErrorCellValue());
         break;
      case BLANK:
         destCell.setBlank();
      }

   }

   private static final String getResolvedFormula(final String formula, final int sourceRowNum, final int destRowNum) {
      Pattern p = Pattern.compile("[a-zA-Z]" + (sourceRowNum + 1) + "(\\D|$)");
      Matcher m = p.matcher(formula);

      String resolvedFormula;
      for(resolvedFormula = formula; m.find(); resolvedFormula = StringUtils.replace(resolvedFormula, "" + (sourceRowNum + 1), "" + (destRowNum + 1))) {
      }

      return resolvedFormula;
   }
}
