/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.io.excel.impl;

import net.sinodawn.framework.beans.BeanPropertyDescriptor;
import net.sinodawn.framework.data.Pair;
import net.sinodawn.framework.io.excel.BaseSheetWriter;
import net.sinodawn.framework.io.excel.ExcelHelper;
import net.sinodawn.framework.io.excel.support.PropertyContext;
import net.sinodawn.framework.io.excel.support.SheetContext;
import net.sinodawn.framework.io.excel.utils.ExcelUtils;
import net.sinodawn.framework.support.domain.BaseData;
import net.sinodawn.framework.utils.ConvertUtils;
import net.sinodawn.framework.utils.ObjectUtils;
import net.sinodawn.framework.utils.ReflectionUtils;
import org.apache.poi.ss.usermodel.*;
import org.apache.poi.ss.util.CellRangeAddress;

import java.util.*;
import java.util.Map.Entry;
import java.util.function.Function;

public class DefaultSheetWriter extends BaseSheetWriter {
   public DefaultSheetWriter(SheetContext sheetContext, List<? extends Object> items) {
      super(sheetContext, items);
   }

   public void write(Workbook wb) {
      SheetContext sheetContext = this.getSheetContext();
      Sheet sheet = wb.createSheet(ExcelUtils.getWellFormedSheetName(sheetContext.getSheetName()));
      Row row = null;
      Cell cell = null;
      CellStyle csForTopic = ExcelUtils.getTopicCellStyle(wb);
      CellStyle csForTitle = ExcelUtils.getTitleCellStyle(wb);
      CellStyle csForCenterContent = ExcelUtils.getCenterCellStyle(wb);
      CellStyle csForLeftContent = ExcelUtils.getLeftCellStyle(wb);
      CellStyle csForRightContent = ExcelUtils.getRightCellStyle(wb);
      CellStyle csForFooterCenter = ExcelUtils.getFooterCenterCellStyle(wb);
      CellStyle csForFooterRight = ExcelUtils.getFooterRightCellStyle(wb);
      int rowNumber = 0;
      List<PropertyContext> propertyContextList = sheetContext.getPropertyList();
      int[] cellMaxWidth = new int[propertyContextList.size() + 1];
      row = sheet.createRow(0);
      cell = row.createCell(0);
      cell.setCellStyle(csForTopic);
      ExcelHelper.setCellStringValue(sheetContext.getTopic(), cell, wb);

      for(int i = 1; i <= propertyContextList.size(); ++i) {
         cell = row.createCell(i);
         cell.setCellValue("");
         cell.setCellStyle(csForTopic);
      }

      sheet.addMergedRegion(new CellRangeAddress(0, 0, 0, propertyContextList.size()));
      rowNumber = rowNumber + 1;
      List<Pair<String, Pair<Integer, Integer>>> subTopicList = sheetContext.getSubTopicList();
      int i;
      boolean belongsToSubTitle;
      Iterator var19;
      Pair subTopic;
      Pair pos;
      if (!subTopicList.isEmpty()) {
         row = sheet.createRow(rowNumber);

         for(i = 0; i <= propertyContextList.size(); ++i) {
            cell = row.createCell(i);
            belongsToSubTitle = false;
            var19 = subTopicList.iterator();

            while(var19.hasNext()) {
               subTopic = (Pair)var19.next();
               pos = (Pair)subTopic.getSecond();
               if ((Integer)pos.getFirst() == i) {
                  ExcelHelper.setCellStringValue((String)subTopic.getFirst(), cell, wb);
                  belongsToSubTitle = true;
                  break;
               }
            }

            if (!belongsToSubTitle) {
               cell.setCellValue("");
            }

            cell.setCellStyle(csForTitle);
         }

         Iterator var38 = subTopicList.iterator();

         while(var38.hasNext()) {
            Pair subTopic1 = (Pair) var38.next();
            Pair<Integer, Integer> pos1 = (Pair)subTopic1.getSecond();
            if (pos1.getSecond() > pos1.getFirst()) {
               sheet.addMergedRegion(new CellRangeAddress(1, 1, pos1.getFirst(), pos1.getSecond()));
            }
         }

         ++rowNumber;
      }

      row = sheet.createRow(rowNumber);
      cell = row.createCell(0);
      cell.setCellValue("序号");
      cell.setCellStyle(csForTitle);
      cellMaxWidth[0] = ExcelUtils.getCellWidth("序号");

      for(i = 0; i < propertyContextList.size(); ++i) {
         PropertyContext propertyContext = (PropertyContext)propertyContextList.get(i);
         cell = row.createCell(i + 1);
         cell.setCellStyle(csForTitle);
         ExcelHelper.setCellStringValue(propertyContext.getTitle(), cell, wb);
         cellMaxWidth[i + 1] = ExcelUtils.getCellWidth(propertyContext.getTitle());
      }

      if (!subTopicList.isEmpty()) {
         for(i = 0; i <= propertyContextList.size(); ++i) {
            belongsToSubTitle = false;
            var19 = subTopicList.iterator();

            while(var19.hasNext()) {
               subTopic = (Pair)var19.next();
               pos = (Pair)subTopic.getSecond();
               if (i >= (Integer)pos.getFirst() && i <= (Integer)pos.getSecond()) {
                  belongsToSubTitle = true;
                  break;
               }
            }

            if (!belongsToSubTitle) {
               sheet.getRow(1).getCell(i).setCellValue(sheet.getRow(2).getCell(i).getStringCellValue());
               sheet.addMergedRegion(new CellRangeAddress(1, 2, i, i));
            }
         }
      }

      ++rowNumber;
      Map<String, Double> sumMap = new LinkedHashMap();
      List<Pair<String, List<String>>> mergePropertyPairList = sheetContext.getMergePropertyPairList();
      Map<String, CellMergeInstance> mergeInstanceMap = new HashMap();
      int mergeCellsForSumTitle = 0;

      for(i = this.getItems().size(); mergeCellsForSumTitle < i; ++mergeCellsForSumTitle) {
         Object item = this.getItems().get(mergeCellsForSumTitle);
         row = sheet.createRow(rowNumber);
         cell = row.createCell(0);
         cell.setCellValue((double)(mergeCellsForSumTitle + 1));
         cell.setCellStyle(csForCenterContent);
         int length = ExcelUtils.getCellWidth("" + mergeCellsForSumTitle);
         if (cellMaxWidth[0] < length) {
            cellMaxWidth[0] = length;
         }

         int j = 0;

         for(int ps = propertyContextList.size(); j < ps; ++j) {
            PropertyContext propertyContext = (PropertyContext)propertyContextList.get(j);
            cell = row.createCell(j + 1);
            Object value = this.getPropertyValue(item, propertyContext.getName());
            boolean isNumeric = false;
            if (!Map.class.isAssignableFrom(this.getItemClass())) {
               BeanPropertyDescriptor propertyDescriptor = BeanPropertyDescriptor.of(item.getClass(), propertyContext.getName());
               if (propertyDescriptor != null && propertyDescriptor.getProperty() != null && Double.class.equals(propertyDescriptor.getPropertyType())) {
                  isNumeric = true;
               }
            } else {
               isNumeric = value != null && Double.class.equals(value.getClass());
            }

            boolean hasStyle = propertyContext.getStyleConsumer() != null;
            if (isNumeric) {
               if (hasStyle) {
                  cell.setCellStyle(ExcelUtils.getRightCellStyle(wb));
               } else {
                  cell.setCellStyle(csForRightContent);
               }
            } else if (hasStyle) {
               if (HorizontalAlignment.LEFT.equals(propertyContext.getAlignment())) {
                  cell.setCellStyle(ExcelUtils.getLeftCellStyle(wb));
               } else if (HorizontalAlignment.RIGHT.equals(propertyContext.getAlignment())) {
                  cell.setCellStyle(ExcelUtils.getRightCellStyle(wb));
               } else {
                  cell.setCellStyle(ExcelUtils.getCenterCellStyle(wb));
               }
            } else if (HorizontalAlignment.LEFT.equals(propertyContext.getAlignment())) {
               cell.setCellStyle(csForLeftContent);
            } else if (HorizontalAlignment.RIGHT.equals(propertyContext.getAlignment())) {
               cell.setCellStyle(csForRightContent);
            } else {
               cell.setCellStyle(csForCenterContent);
            }

            if (propertyContext.getCellRowItemConsumer() != null) {
               propertyContext.getCellRowItemConsumer().accept(cell, item);
            }

            if (hasStyle) {
               propertyContext.getStyleConsumer().accept(cell, item);
            }

            if (value != null) {
               Object convertedValue = propertyContext.getConvertFunction().apply(value);
               if (propertyContext.isNumeric()) {
                  cell.setCellValue((Double)convertedValue);
               } else {
                  ExcelHelper.setCellStringValue((String)convertedValue, cell, wb);
               }

               length = ExcelUtils.getCellWidth((String)ConvertUtils.convert(convertedValue, String.class));
               if (cellMaxWidth[j + 1] < length) {
                  cellMaxWidth[j + 1] = length;
               }

               if (this.getSheetContext().getSumPropertyList().stream().anyMatch((n) -> {
                  return propertyContext.getName().equalsIgnoreCase((String)n.getFirst());
               })) {
                  double sum = (Double)ConvertUtils.convert(sumMap.get(propertyContext.getName()), Double.TYPE);
                  sum += (Double)ConvertUtils.convert(value, Double.TYPE);
                  sumMap.put(propertyContext.getName(), sum);
               }
            } else {
               cell.setBlank();
            }

            if (!mergePropertyPairList.isEmpty()) {
               Pair<String, List<String>> pair = (Pair)mergePropertyPairList.stream().filter((p) -> {
                  return ((String)p.getFirst()).equalsIgnoreCase(propertyContext.getName());
               }).findFirst().orElse(null);
               if (pair != null) {
                  CellMergeInstance mergeInstance = (CellMergeInstance)mergeInstanceMap.get(pair.getFirst());
                  if (mergeInstance == null) {
                     CellMergeInstance instance = new CellMergeInstance(propertyContext.getName(), (List)pair.getSecond(), rowNumber, j + 1);
                     List<Object> valueList = new ArrayList();
                     Iterator var58 = ((List)pair.getSecond()).iterator();

                     while(var58.hasNext()) {
                        String name = (String)var58.next();
                        valueList.add(this.getPropertyValue(item, name));
                     }

                     instance.setComparisionValueList(valueList);
                     mergeInstanceMap.put(pair.getFirst(), instance);
                  } else if (!this.isMergeRow(item, mergeInstance)) {
                     if (mergeInstance.getStartRowIndex() + 1 < rowNumber) {
                        sheet.addMergedRegion(new CellRangeAddress(mergeInstance.getStartRowIndex(), rowNumber - 1, mergeInstance.getColumnIndex(), mergeInstance.getColumnIndex()));
                     }

                     mergeInstance.setStartRowIndex(rowNumber);
                     List<Object> valueList = new ArrayList();
                     Iterator var33 = ((List)pair.getSecond()).iterator();

                     while(var33.hasNext()) {
                        String name = (String)var33.next();
                        valueList.add(this.getPropertyValue(item, name));
                     }

                     mergeInstance.setComparisionValueList(valueList);
                  }
               }
            }
         }

         ++rowNumber;
      }

      if (!sumMap.isEmpty()) {
         row = sheet.createRow(rowNumber);
         cell = row.createCell(0);
         cell.setCellValue("合计：");
         cell.setCellStyle(csForFooterCenter);
         mergeCellsForSumTitle = -1;
         i = 0;

         for(int ps = propertyContextList.size(); i < ps; ++i) {
            PropertyContext propertyContext = (PropertyContext)propertyContextList.get(i);
            cell = row.createCell(i + 1);
            cell.setCellStyle(csForFooterRight);
            Double sum = (Double)sumMap.get(propertyContext.getName());
            if (sum != null) {
               if (mergeCellsForSumTitle == -1) {
                  mergeCellsForSumTitle = i;
               }

               Function<Double, String> function = (Function)((Pair)this.getSheetContext().getSumPropertyList().stream().filter((n) -> {
                  return propertyContext.getName().equalsIgnoreCase((String)n.getFirst());
               }).findFirst().get()).getSecond();
               String stringValue = (String)function.apply(sum);
               cell.setCellValue((Double)ConvertUtils.convert(stringValue, Double.class));
               int length = ExcelUtils.getCellWidth(stringValue);
               if (cellMaxWidth[i + 1] < length) {
                  cellMaxWidth[i + 1] = length;
               }
            } else {
               cell.setBlank();
            }
         }

         sheet.addMergedRegion(new CellRangeAddress(row.getRowNum(), row.getRowNum(), 0, mergeCellsForSumTitle));
      }

      for(mergeCellsForSumTitle = 0; mergeCellsForSumTitle < cellMaxWidth.length; ++mergeCellsForSumTitle) {
         i = cellMaxWidth[mergeCellsForSumTitle] + 512;
         if (i > 15360) {
            i = 15360;
         }

         sheet.setColumnWidth(mergeCellsForSumTitle, i);
      }

      if (subTopicList.isEmpty()) {
         sheet.createFreezePane(1, 2);
      } else {
         sheet.createFreezePane(1, 3);
      }

   }

   private Object getPropertyValue(Object item, String propertyName) {
      if (item != null && propertyName != null) {
         if (Map.class.isAssignableFrom(this.getItemClass())) {
            Map<String, Object> item1 = (Map<String, Object>) item;
            return item1.entrySet().stream().filter(
                    (e) -> propertyName.equalsIgnoreCase(e.getKey())
            ).map(
                    (e) -> ObjectUtils.isEmpty(e.getValue()) ? "" : e.getValue()
            ).findFirst().orElse(null);
         } else {
            BeanPropertyDescriptor propertyDescriptor = BeanPropertyDescriptor.of(item.getClass(), propertyName);
            if (propertyDescriptor != null) {
               return ReflectionUtils.invokeMethod(propertyDescriptor.getReadMethod(), item);
            } else {
               if (BaseData.class.isAssignableFrom(item.getClass())) {
                  Entry<String, String> entry = (Entry)((BaseData)item).getExt$().entrySet().stream().filter((e) -> {
                     return propertyName.equalsIgnoreCase((String)e.getKey());
                  }).findFirst().orElse(null);
                  if (entry != null) {
                     return entry.getValue();
                  }
               }

               return null;
            }
         }
      } else {
         return null;
      }
   }

   private boolean isMergeRow(Object item, CellMergeInstance mergeInstance) {
      List<String> comparisionPropertyList = mergeInstance.getComparisionPropertyList();

      for(int i = 0; i < comparisionPropertyList.size(); ++i) {
         Object value = this.getPropertyValue(item, (String)comparisionPropertyList.get(i));
         Object comparisionValue = mergeInstance.getComparisionValueList().get(i);
         if (!ObjectUtils.equals(value, comparisionValue)) {
            return false;
         }
      }

      return true;
   }

   public static class CellMergeInstance {
      private String property;
      private List<String> comparisionPropertyList;
      private int startRowIndex;
      private int columnIndex;
      private List<Object> comparisionValueList;

      public CellMergeInstance(String property, List<String> comparisionPropertyList, int startRowIndex, int columnIndex) {
         this.property = property;
         this.comparisionPropertyList = comparisionPropertyList;
         this.startRowIndex = startRowIndex;
         this.columnIndex = columnIndex;
      }

      public int getStartRowIndex() {
         return this.startRowIndex;
      }

      public void setStartRowIndex(int startRowIndex) {
         this.startRowIndex = startRowIndex;
      }

      public int getColumnIndex() {
         return this.columnIndex;
      }

      public void setColumnIndex(int columnIndex) {
         this.columnIndex = columnIndex;
      }

      public String getProperty() {
         return this.property;
      }

      public void setProperty(String property) {
         this.property = property;
      }

      public List<String> getComparisionPropertyList() {
         return this.comparisionPropertyList;
      }

      public void setComparisionPropertyList(List<String> comparisionPropertyList) {
         this.comparisionPropertyList = comparisionPropertyList;
      }

      public List<Object> getComparisionValueList() {
         return this.comparisionValueList;
      }

      public void setComparisionValueList(List<Object> comparisionValueList) {
         this.comparisionValueList = comparisionValueList;
      }
   }
}
