/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.executor;

import net.sinodawn.framework.exception.TaskException;
import net.sinodawn.framework.executor.custom.CustomFutureTask;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.core.annotation.Order;
import org.springframework.stereotype.Component;
import org.springframework.transaction.support.TransactionSynchronizationAdapter;
import org.springframework.transaction.support.TransactionSynchronizationManager;

import java.util.Timer;
import java.util.TimerTask;
import java.util.concurrent.Callable;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;

@Order(0)
@Component
public class ExecutorHelper {
   private static ThreadPoolExecutor executor;

   @Autowired
   public void setExecutor(ThreadPoolExecutor executor) {
      ExecutorHelper.executor = executor;
   }

   public static <T> CustomFutureTask<T> submit(Callable<T> task) {
      return (CustomFutureTask<T>) executor.submit(task);
   }

   public static void submit(Runnable task) {
      executor.submit(task);
   }

   public static void submit(TimerTask task, long delayMilliSeconds) {
      Timer timer = new Timer();
      timer.schedule(task, delayMilliSeconds);
   }

   public static <T> T submit(Callable<T> task, long timeoutMilliSeconds) {
      CustomFutureTask<T> futureTask = (CustomFutureTask<T>) executor.submit(task);

      try {
         return (T) futureTask.get(timeoutMilliSeconds, TimeUnit.MILLISECONDS);
      } catch (Exception var7) {
         try {
            futureTask.cancel(true);
         } catch (Exception ignored) {
         }

         throw new TaskException(var7.getMessage());
      }
   }

   public static void submitAfterCommitTransaction(Runnable task) {
      TransactionSynchronizationManager.registerSynchronization(new TransactionSynchronizationAdapter() {
         public void afterCommit() {
            ExecutorHelper.executor.submit(task);
         }
      });
   }
}
