/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.database.sql;


import net.sinodawn.framework.context.ApplicationContextHelper;
import net.sinodawn.framework.database.dialect.Dialect;
import net.sinodawn.framework.database.dialect.DialectRepository;
import net.sinodawn.framework.exception.UnexpectedException;
import net.sinodawn.framework.exception.database.JdbcException;
import net.sinodawn.framework.utils.StringUtils;
import org.apache.ibatis.session.SqlSession;
import org.mybatis.spring.SqlSessionTemplate;
import org.mybatis.spring.SqlSessionUtils;

import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

public abstract class SqlHelper {
    public static final String QTY_COLUMN_NAME = "QTY_";
    private static final Pattern SQL_ENDING_ORDER_BY_PATTERN = Pattern.compile("\\s+(?i)order\\s+(?i)by\\s+([^()]*[(][^()]*[)][^()]*)*[^()]*$");
    private static final Pattern SQL_ORDER_BY_PATTERN = Pattern.compile("(?i)order\\s+by");

    public SqlHelper() {
    }

    public static String removeEndingOrderIfPresent(final String sql) {
        if (StringUtils.isEmpty(sql)) {
            return sql;
        } else {
            Matcher matcher = SQL_ENDING_ORDER_BY_PATTERN.matcher(sql);
            return matcher.find() ? matcher.replaceFirst("") : sql;
        }
    }

    public static boolean hasOrderSegment(final String sql) {
        Objects.requireNonNull(sql);
        Matcher matcher = SQL_ENDING_ORDER_BY_PATTERN.matcher(sql);
        return matcher.find();
    }

    public static String getOrderSegment(final String sql) {
        Objects.requireNonNull(sql);
        Matcher matcher = SQL_ENDING_ORDER_BY_PATTERN.matcher(sql);
        if (matcher.find()) {
            int pos = matcher.start();
            String orderBySql = sql.substring(pos).trim();
            return SQL_ORDER_BY_PATTERN.matcher(orderBySql).replaceFirst("").trim();
        } else {
            throw new UnexpectedException("Not an ordered sql.");
        }
    }

    public static String getSelectTotalSql(String sql, String[] sumProperties, String[] countProperties) {
        Dialect dialect = DialectRepository.getDialect();
        StringBuilder newSql = (new StringBuilder("SELECT COUNT(1) ")).append("QTY_");
        if (sumProperties.length > 0) {
            Arrays.asList(sumProperties).forEach((c) -> {
                newSql.append(", SUM(").append(dialect.getNullReplacementClause(c.toUpperCase(), "0")).append(") ").append(c.toUpperCase());
            });
        }

        if (countProperties.length > 0) {
            Arrays.asList(countProperties).forEach((c) -> {
                newSql.append(", COUNT(DISTINCT ").append(c.toUpperCase()).append(") ").append(c.toUpperCase());
            });
        }

        newSql.append(" FROM (\n").append(removeEndingOrderIfPresent(sql)).append("\n").append(") T");
        return newSql.toString();
    }

    public static boolean isRiskySqlSegment(String sqlSegment) {
        if (StringUtils.isEmpty(sqlSegment)) {
            return false;
        } else {
            return StringUtils.contains(sqlSegment, "'") || StringUtils.contains(sqlSegment, " ");
        }
    }

    public static List<String> getSqlColumnList(String sql) {
        List<String> columnList = new ArrayList<>();
        SqlSessionTemplate template = ApplicationContextHelper.getMybatisSqlSessionTemplate();
        SqlSession sqlSession = null;

        try {
            sqlSession = SqlSessionUtils.getSqlSession(template.getSqlSessionFactory(), template.getExecutorType(), template.getPersistenceExceptionTranslator());

            try {
                Statement stmt = sqlSession.getConnection().createStatement();
                Throwable var5 = null;

                try {
                    ResultSet rs = stmt.executeQuery(DialectRepository.getDialect().getLimitSql(sql, 0, 0));
                    Throwable var7 = null;

                    try {
                        ResultSetMetaData rsmd = rs.getMetaData();
                        int i = 0;

                        for(int j = rsmd.getColumnCount(); i < j; ++i) {
                            columnList.add(rsmd.getColumnName(i + 1).toUpperCase());
                        }
                    } catch (Throwable var47) {
                        var7 = var47;
                        throw var47;
                    } finally {
                        if (rs != null) {
                            if (var7 != null) {
                                try {
                                    rs.close();
                                } catch (Throwable var46) {
                                    var7.addSuppressed(var46);
                                }
                            } else {
                                rs.close();
                            }
                        }

                    }
                } catch (Throwable var49) {
                    var5 = var49;
                    throw var49;
                } finally {
                    if (stmt != null) {
                        if (var5 != null) {
                            try {
                                stmt.close();
                            } catch (Throwable var45) {
                                var5.addSuppressed(var45);
                            }
                        } else {
                            stmt.close();
                        }
                    }

                }
            } catch (SQLException var51) {
                throw new JdbcException(var51);
            }
        } finally {
            if (sqlSession != null) {
                SqlSessionUtils.closeSqlSession(sqlSession, template.getSqlSessionFactory());
            }

        }

        return columnList;
    }
}
