/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.database.core;


import net.sinodawn.framework.beans.BeanPropertyDescriptor;
import net.sinodawn.framework.beans.BeanPropertyHelper;
import net.sinodawn.framework.cache.redis.RedisHelper;
import net.sinodawn.framework.database.context.ColumnContext;
import net.sinodawn.framework.database.context.EntityColumnContext;
import net.sinodawn.framework.database.context.EntityContext;
import net.sinodawn.framework.database.context.TableContext;
import net.sinodawn.framework.database.context.instance.EntityHelper;
import net.sinodawn.framework.database.dialect.DialectRepository;
import net.sinodawn.framework.exception.database.JdbcException;
import net.sinodawn.framework.mybatis.MybatisHelper;
import net.sinodawn.framework.support.PersistableMetadataHelper;
import net.sinodawn.framework.support.domain.Persistable;
import net.sinodawn.framework.utils.ClassUtils;
import net.sinodawn.framework.utils.StringUtils;
import org.springframework.jdbc.support.JdbcUtils;

import java.sql.*;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;

// TODO
@SuppressWarnings({"rawtypes", "unused"})
public abstract class DatabaseManager {
    public static final String TABLE_CONTEXT_CACHE_NAME = "TableContext";
    public static final String ENTITY_CONTEXT_CACHE_NAME = "EntityContext";

    public DatabaseManager() {
    }

    public static TableContext getTableContext(String tableName) {
        tableName = tableName.toUpperCase();
        TableContext tableContext = RedisHelper.get("TableContext", tableName);
        if (tableContext == null) {
            tableContext = new TableContext();
            tableContext.setTableName(tableName);
            tableContext.setColumnContextList(selectColumnContextList(DialectRepository.getDialect().getSelectFromTableSql(tableName)));
            RedisHelper.put("TableContext", tableName, tableContext);
        }

        return tableContext;
    }

    public static List<ColumnContext> getTableColumnContextList(String tableName) {
        tableName = tableName.toUpperCase();
        return getTableContext(tableName).getColumnContextList();
    }

    public static EntityContext getEntityContext(Class<? extends Persistable<?>> entityClass) {
        String cacheKey = ClassUtils.getRawType(entityClass).toString();
        String tableName = PersistableMetadataHelper.getTableName(entityClass);
        EntityContext entityContext = RedisHelper.get("EntityContext", cacheKey);
        if (entityContext == null) {
            entityContext = new EntityContext();
            entityContext.setTableName(tableName);
            TableContext tableContext = getTableContext(tableName);
            List<BeanPropertyDescriptor> propertyDescriptorList = BeanPropertyHelper.getBeanPropertyDescriptorList(entityClass);
            List<EntityColumnContext> columnContextList = new ArrayList<>();

            EntityColumnContext entityColumnContext;
            for(Iterator var7 = tableContext.getColumnContextList().iterator(); var7.hasNext(); columnContextList.add(entityColumnContext)) {
                ColumnContext columnContext = (ColumnContext)var7.next();
                BeanPropertyDescriptor propertyDescriptor = propertyDescriptorList.stream().filter((p) -> columnContext.getColumnName().equalsIgnoreCase(p.getName())).findFirst().orElse(null);
                entityColumnContext = new EntityColumnContext();
                entityColumnContext.setEntityClass(entityClass);
                entityColumnContext.setColumnName(columnContext.getColumnName());
                entityColumnContext.setBlob(DialectRepository.getDialect().isBlob(columnContext));
                entityColumnContext.setClob(DialectRepository.getDialect().isClob(columnContext));
                if (propertyDescriptor == null) {
                    entityColumnContext.setType(columnContext.getType());
                } else {
                    entityColumnContext.setType(propertyDescriptor.getPropertyType());
                }

                if ("id".equalsIgnoreCase(columnContext.getColumnName())) {
                    entityContext.setIdContext(entityColumnContext);
                } else if ("version".equalsIgnoreCase(columnContext.getColumnName())) {
                    entityContext.setVersionContext(entityColumnContext);
                }
            }

            EntityHelper.sortColumnContext(columnContextList);
            entityContext.setColumnContextList(columnContextList);
            RedisHelper.put("EntityContext", cacheKey, entityContext);
        }

        return entityContext;
    }

//    public static void evictCache(Class<? extends Persistable<?>> entityClass) {
//        String cacheKey = ClassUtils.getRawType(entityClass).toString();
//        String tableName = PersistableMetadataHelper.getTableName(entityClass);
//        RedisHelper.evict("TableContext", tableName);
//        RedisHelper.evict("EntityContext", cacheKey);
//    }

//    public static void evictCache(String tableName) {
//        ClassUtils.getPredicatedClasses("net.sinodawn", (c) -> {
//            return c.isAnnotationPresent(Table.class) && Persistable.class.isAssignableFrom(c) && tableName.equalsIgnoreCase(PersistableMetadataHelper.getTableName(c));
//        }).forEach((c) -> {
//            evictCache(c);
//        });
//    }

    public static List<ColumnContext> selectColumnContextList(String sql) {
        String selectEmptySql = DialectRepository.getDialect().getLimitSql(sql, 0, 0);
        Connection conn = null;
        PreparedStatement ps = null;
        ResultSet rs = null;

        try {
            conn = MybatisHelper.getConnection();
            ps = conn.prepareStatement(selectEmptySql);
            rs = ps.executeQuery();
            ResultSetMetaData rsmd = rs.getMetaData();
            int qty = rsmd.getColumnCount();
            List<ColumnContext> columnContextList = new ArrayList<>();

            for(int i = 1; i <= qty; ++i) {
                ColumnContext columnContext = new ColumnContext();
                columnContext.setRawColumnName(rsmd.getColumnName(i));
                columnContext.setColumnName(StringUtils.upperCase(columnContext.getRawColumnName()));
                columnContext.setDataType(rsmd.getColumnTypeName(i));
                int scale = rsmd.getScale(i);
                columnContext.setType(DialectRepository.getDialect().getJavaType(columnContext.getColumnName(), columnContext.getDataType(), scale));
                columnContextList.add(columnContext);
            }

            return columnContextList;
        } catch (SQLException var14) {
            throw new JdbcException(var14);
        } finally {
            JdbcUtils.closeResultSet(rs);
            JdbcUtils.closeStatement(ps);
            JdbcUtils.closeConnection(conn);
        }
    }
}
