/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.data.page;


import net.sinodawn.framework.mybatis.page.PageRowBounds;
import net.sinodawn.framework.utils.CollectionUtils;
import net.sinodawn.framework.utils.NumberUtils;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

@SuppressWarnings("unused")
public class Pagination<T> implements Page<T> {
    private static final long serialVersionUID = 5715827699736745767L;
    private int total;
    private int pageSize;
    private int pageNumber;
    private List<T> rows;
    private Map<String, Number> totalMap;
    private final Map<String, String> columnTypeMap;

    public Pagination() {
        this.columnTypeMap = new HashMap<>();
        this.total = 0;
        this.pageSize = 0;
        this.pageNumber = 0;
        this.rows = CollectionUtils.emptyList();
        this.totalMap = CollectionUtils.emptyMap();
    }

    public Pagination(List<T> rows) {
        this.columnTypeMap = new HashMap<>();
        this.total = rows.size();
        this.pageSize = rows.size();
        this.pageNumber = 1;
        this.rows = Objects.requireNonNull(rows);
        this.totalMap = CollectionUtils.emptyMap();
    }

    public Pagination(int total, int pageSize, int pageNumber, List<T> rows) {
        this.columnTypeMap = new HashMap<>();
        this.total = total;
        this.pageSize = pageSize;
        this.pageNumber = pageNumber;
        this.rows = Objects.requireNonNull(rows);
        this.totalMap = CollectionUtils.emptyMap();
    }

    public Pagination(int total, int pageSize, int pageNumber, List<T> rows, Map<String, Number> totalMap) {
        this.columnTypeMap = new HashMap<>();
        this.total = total;
        this.pageSize = pageSize;
        this.pageNumber = pageNumber;
        this.rows = (List)Objects.requireNonNull(rows);
        this.totalMap = (Map)Objects.requireNonNull(totalMap);
    }

    public Pagination(int pageSize, int pageNumber, org.springframework.data.domain.Page<T> page) {
        this.columnTypeMap = new HashMap<>();
        this.total = Long.valueOf(page.getTotalElements()).intValue();
        this.pageSize = pageSize;
        this.pageNumber = pageNumber;
        this.rows = page.getContent();
        this.totalMap = CollectionUtils.emptyMap();
    }

    public Pagination(Page<?> otherPagination, List<T> rows) {
        this.columnTypeMap = new HashMap<>();
        this.total = NumberUtils.max(otherPagination.getTotal(), rows.size());
        this.pageSize = otherPagination.getPageSize();
        this.pageNumber = otherPagination.getPageNumber();
        this.totalMap = otherPagination.getTotalMap();
        this.rows = Objects.requireNonNull(rows);
        this.columnTypeMap.putAll(otherPagination.getColumnTypeMap());
    }

    public Pagination(PageRowBounds rowBounds, List<T> rows) {
        this(NumberUtils.max(rowBounds.getTotal(), rows.size()), rowBounds.getLimit(), rowBounds.getPageNumber(), rows, rowBounds.getTotalMap());
    }

    public int getTotalPages() {
        return Double.valueOf(Math.ceil((double)this.total / (double)this.getPageSize())).intValue();
    }

    public int getTotal() {
        return this.total;
    }

    public int getNumberOfElements() {
        return this.rows.size();
    }

    public boolean isFirst() {
        return this.pageNumber == 1;
    }

    public boolean isLast() {
        return !this.hasNext();
    }

    public boolean hasNext() {
        return this.pageNumber + 1 < this.getTotalPages();
    }

    public boolean hasPrevious() {
        return this.pageNumber > 1;
    }

    public int getPageNumber() {
        return this.pageNumber;
    }

    public List<T> getRows() {
        return this.rows;
    }

    public Map<String, Number> getTotalMap() {
        return this.totalMap;
    }

    public Map<String, String> getColumnTypeMap() {
        return this.columnTypeMap;
    }

    public void setColumnTypeMap(Map<String, String> columnTypeMap) {
        if (columnTypeMap != null) {
            this.columnTypeMap.putAll(columnTypeMap);
        }

    }

    public void setTotal(int total) {
        this.total = total;
    }

    public void setPageNumber(int pageNumber) {
        this.pageNumber = pageNumber;
    }

    public void setRows(List<T> rows) {
        this.rows = rows;
    }

    public void setTotalMap(Map<String, Number> totalMap) {
        this.totalMap = totalMap;
    }

    public int getPageSize() {
        return this.pageSize;
    }

    public void setPageSize(int pageSize) {
        this.pageSize = pageSize;
    }
}
