/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.converter.string;

import net.sinodawn.framework.converter.Converter;
import net.sinodawn.framework.converter.ConverterFactory;
import net.sinodawn.framework.converter.exception.ConverterNotFoundException;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.CharArrayReader;
import java.io.InputStreamReader;
import java.io.LineNumberReader;
import java.io.Reader;
import java.io.StringReader;
import java.nio.charset.StandardCharsets;

public enum StringToReaderConverterFactory implements ConverterFactory<String, Reader> {
   INSTANCE;

   public <T extends Reader> Converter<String, T> getConverter(Class<T> targetType) {
      return new StringToReaderConverter(targetType);
   }

   private class StringToReaderConverter<T extends Reader> implements Converter<String, T> {
      private final Class<T> readerType;

      public StringToReaderConverter(Class<T> readerType) {
         this.readerType = readerType;
      }

      public Class<String> getSourceType() {
         return String.class;
      }

      public Class<T> getTargetType() {
         return this.readerType;
      }

      public T convert(String source) {
         if (source == null) {
            return null;
         } else {
            Class<T> targetType = this.getTargetType();
            if (BufferedReader.class == targetType) {
               return (T) new BufferedReader(new StringReader(source));
            } else if (CharArrayReader.class == this.getTargetType()) {
               return (T) new CharArrayReader(source.toCharArray());
            } else if (InputStreamReader.class == targetType) {
               return (T) new InputStreamReader(new ByteArrayInputStream(source.getBytes(StandardCharsets.UTF_8)));
            } else if (LineNumberReader.class == targetType) {
               return (T) new LineNumberReader(new StringReader(source));
            } else if (StringReader.class == targetType) {
               return (T) new StringReader(source);
            } else {
               throw new ConverterNotFoundException(this.getSourceType(), targetType);
            }
         }
      }
   }
}
