/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.converter.string;

import net.sinodawn.framework.converter.Converter;
import net.sinodawn.framework.converter.ConverterFactory;
import net.sinodawn.framework.converter.exception.ConverterNotFoundException;
import net.sinodawn.framework.utils.DateTimeUtils;
import java.time.LocalDate;
import java.time.LocalDateTime;
import java.time.LocalTime;
import java.time.temporal.Temporal;

public enum StringToJSR310ConverterFactory implements ConverterFactory<String, Temporal> {
   INSTANCE;

   public <T extends Temporal> Converter<String, T> getConverter(Class<T> targetType) {
      return new StringToJSR310Converter(targetType);
   }

   private class StringToJSR310Converter<T extends Temporal> implements Converter<String, T> {
      private final Class<T> jsr310Type;

      public StringToJSR310Converter(Class<T> jsr310Type) {
         this.jsr310Type = jsr310Type;
      }

      public Class<String> getSourceType() {
         return String.class;
      }

      public Class<T> getTargetType() {
         return this.jsr310Type;
      }

      public T convert(String source) {
         if (source == null) {
            return null;
         } else {
            Class<T> targetType = this.getTargetType();
            if (LocalDateTime.class.equals(targetType)) {
               return (T) DateTimeUtils.parseLocalDateTime(source);
            } else if (LocalDate.class.equals(targetType)) {
               return (T) DateTimeUtils.parseLocalDate(source);
            } else if (LocalTime.class.equals(targetType)) {
               return (T) DateTimeUtils.parseLocalTime(source);
            } else {
               throw new ConverterNotFoundException(this.getSourceType(), targetType);
            }
         }
      }
   }
}
