/*
 * Copyright (c) SinoDawn 2021.
 */

package net.sinodawn.framework.converter.string;

import net.sinodawn.framework.converter.Converter;
import net.sinodawn.framework.converter.ConverterFactory;
import net.sinodawn.framework.converter.exception.ConverterNotFoundException;
import java.io.BufferedInputStream;
import java.io.ByteArrayInputStream;
import java.io.DataInputStream;
import java.io.InputStream;
import java.io.PushbackInputStream;
import java.nio.charset.StandardCharsets;

public enum StringToInputStreamConverterFactory implements ConverterFactory<String, InputStream> {
   INSTANCE;

   public <T extends InputStream> Converter<String, T> getConverter(Class<T> targetType) {
      return new StringToInputStreamConverter(targetType);
   }

   private class StringToInputStreamConverter<T extends InputStream> implements Converter<String, T> {
      private final Class<T> inputStreamType;

      public StringToInputStreamConverter(Class<T> inputStreamType) {
         this.inputStreamType = inputStreamType;
      }

      public Class<String> getSourceType() {
         return String.class;
      }

      public Class<T> getTargetType() {
         return this.inputStreamType;
      }

      public T convert(String source) {
         if (source == null) {
            return null;
         } else {
            Class<T> targetType = this.getTargetType();
            if (ByteArrayInputStream.class == targetType) {
               return (T) new ByteArrayInputStream(source.getBytes(StandardCharsets.UTF_8));
            } else if (BufferedInputStream.class == targetType) {
               return (T) new BufferedInputStream(new ByteArrayInputStream(source.getBytes(StandardCharsets.UTF_8)));
            } else if (DataInputStream.class == targetType) {
               return (T) new DataInputStream(new ByteArrayInputStream(source.getBytes(StandardCharsets.UTF_8)));
            } else if (PushbackInputStream.class == targetType) {
               return (T) new PushbackInputStream(new ByteArrayInputStream(source.getBytes(StandardCharsets.UTF_8)));
            } else {
               throw new ConverterNotFoundException(this.getSourceType(), targetType);
            }
         }
      }
   }
}
